/**
 * @file twi.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 * @brief BlueTank ACB-BF592 Application Sample Codes.
 */

/*
 * ===============================================================
 *  BlueTank
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <cdefBF592-A.h>
#include <builtins.h>
#include <sys/exception.h>
#include <stdint.h>
#include "twi.h"
#include "bfin_util.h"

#define TWI_COUNT(x) (DCNT & ((x) << 6))

typedef struct
{
    int32_t  state;
    int32_t  xfer_dir;  /* 0:READ, 1:WRITE */
    uint8_t* p_data;
    int32_t  count;
} twi_xfer_param;

/* TWI割り込みのプロトタイプ宣言 */
static __attribute__((interrupt_handler)) void twi_isr(void);

static twi_xfer_param sf_twi_xfer_param;

void twi_init(void)
{
    *pTWI_CONTROL = 0;
    ssync();
    *pTWI_CONTROL = 20U;
    *pTWI_CLKDIV  = CLKHI(12) | CLKLOW(13);
    *pTWI_INT_MASK = RCVSERV | XMTSERV | MERR | MCOMP;
    *pTWI_INT_STAT = *pTWI_INT_STAT;
    *pTWI_FIFO_CTL = XMTFLUSH | RCVFLUSH;
    ssync();
    *pTWI_FIFO_CTL = 0;

    /* 割り込みハンドラをIVG12に登録 */
    bfin_util_register_handler(ik_ivg12, twi_isr);
    *pSIC_IMASK |= IRQ_TWI;
}

void twi_master_write(uint8_t slave_addr, int32_t rstart, const uint8_t* p_data, int32_t write_cnt)
{
    if(write_cnt <= 0) return;

    sf_twi_xfer_param.state    = TWI_STATE_XFER;
    sf_twi_xfer_param.xfer_dir = 1;
    sf_twi_xfer_param.p_data   = (uint8_t*)&p_data[1];
    sf_twi_xfer_param.count    = write_cnt - 1;

    *pTWI_CONTROL |= TWI_ENA;
    ssync();
    *pTWI_INT_STAT = MCOMP | MERR | XMTSERV | RCVSERV;
    *pTWI_MASTER_STAT = BUFWRERR | BUFRDERR | LOSTARB | ANAK | DNAK;
    *pTWI_MASTER_ADDR = slave_addr;
    ssync();
    // Send first byte
    *pTWI_XMT_DATA8 = p_data[0];
    *pTWI_MASTER_CTL = TWI_COUNT(write_cnt) | FAST | MEN | (rstart != TWI_STOP ? RSTART : 0);
}

void twi_master_read(uint8_t slave_addr, int32_t rstart, uint8_t* p_data, int32_t read_cnt)
{
    if(read_cnt <= 0) return;

    sf_twi_xfer_param.state    = TWI_STATE_XFER;
    sf_twi_xfer_param.xfer_dir = 0;
    sf_twi_xfer_param.p_data   = p_data;
    sf_twi_xfer_param.count    = read_cnt;

    *pTWI_CONTROL |= TWI_ENA;
    ssync();
    *pTWI_INT_STAT = MCOMP | MERR | XMTSERV | RCVSERV;
    *pTWI_MASTER_STAT = BUFWRERR | BUFRDERR | LOSTARB | ANAK | DNAK;
    *pTWI_MASTER_ADDR = slave_addr;
    ssync();
    *pTWI_MASTER_CTL = TWI_COUNT(read_cnt) | FAST | MDIR | MEN | (rstart != TWI_STOP ? RSTART : 0);
}

int32_t twi_master_xfer_state(void)
{
    return sf_twi_xfer_param.state;
}

__attribute__((interrupt_handler)) void twi_isr(void)
{
    uint16_t l_twi_stat = *pTWI_INT_STAT;
    *pTWI_INT_STAT = l_twi_stat;
    ssync();

    if((l_twi_stat & MERR) != 0)
    {
        sf_twi_xfer_param.state = TWI_STATE_ERROR;
        *pTWI_CONTROL &= ~TWI_ENA;
    }
    else if(sf_twi_xfer_param.state != TWI_STATE_ERROR)
    {
        if((l_twi_stat & MCOMP) != 0)
        {
            sf_twi_xfer_param.state = TWI_STATE_COMP;
            *pTWI_CONTROL &= ~TWI_ENA;
        }
        else
        {
            if(0 == sf_twi_xfer_param.xfer_dir)
            {
                while((sf_twi_xfer_param.count > 0) && ((*pTWI_FIFO_STAT & RCVSTAT) != 0))
                {
                    *sf_twi_xfer_param.p_data = *pTWI_RCV_DATA8;
                    sf_twi_xfer_param.p_data++;
                    sf_twi_xfer_param.count--;
                    ssync();
                }
            }
            else
            {
                while((sf_twi_xfer_param.count > 0) && ((*pTWI_FIFO_STAT & XMTSTAT) != XMTSTAT))
                {
                    *pTWI_XMT_DATA8 = *sf_twi_xfer_param.p_data;
                    sf_twi_xfer_param.p_data++;
                    sf_twi_xfer_param.count--;
                    ssync();
                }
            }
        }
    }
}
