//
//  CMRBrowser.m
//  BathyScaphe
//
//  Updated by Tsutomu Sawada on 07/08/26.
//  Copyright 2005-2009 BathyScaphe Project. All rights reserved.
//  encoding="UTF-8"
//

#import "CMRBrowser_p.h"
#import "BSBoardInfoInspector.h"
#import "BSQuickLookPanelController.h"

NSString *const CMRBrowserDidChangeBoardNotification = @"CMRBrowserDidChangeBoardNotification";
NSString *const CMRBrowserThListUpdateDelegateTaskDidFinishNotification = @"CMRBrThListUpdateDelgTaskDidFinishNotification";

static void *kBrowserContext = @"Konata";
static NSString *const kObservingKey = @"isSplitViewVertical";

/*
 * current main browser instance.
 * @see CMRExports.h 
 */
CMRBrowser *CMRMainBrowser = nil;

@implementation CMRBrowser
- (id)init
{
	if (self = [super init]) {
        if (floor(NSAppKitVersionNumber) > NSAppKitVersionNumber10_4) {
            [self setShouldCascadeWindows:YES];
		} else {
            [self setShouldCascadeWindows:NO];
        }

		if (!CMRMainBrowser) {
			CMRMainBrowser = self;
		}

		[CMRPref addObserver:self forKeyPath:kObservingKey options:NSKeyValueObservingOptionNew context:kBrowserContext];
        [self setKeepPaths:nil];
        [self setKeepCondition:CMRAutoscrollNone];
	}
	return self;
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context
{
	if (context == kBrowserContext && object == CMRPref && [keyPath isEqualToString:kObservingKey]) {
		[self setupSplitView];
		[[self splitView] resizeSubviewsWithOldSize:[[self splitView] frame].size];
    } else if (context == kTLVMContext && object == [self document] && [keyPath isEqualToString:@"threadsListViewMode"]) {
        [self storeKeepPath:CMRAutoscrollWhenTLVMChange];
	} else {
		[super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
	}
}

- (NSString *)windowNibName
{
	return @"CMRBrowser";
}

- (NSString *)windowTitleForDocumentDisplayName:(NSString *)displayName
{
	// スーパークラス（CMRThreadViewer）が行なっている処理は CMRBrowser では不要なので、単に displayName を返す。
	return displayName;
}

#pragma mark -
- (void)exchangeOrDisposeMainBrowser
{
	NSArray *curWindows = [NSApp orderedWindows];
	if (!curWindows || [curWindows count] == 0) {
		// Dispose...
		CMRMainBrowser = nil;
		return;
	}

	NSEnumerator *iter_ = [curWindows objectEnumerator];
	NSWindow *eachItem;
	
	while (eachItem = [iter_ nextObject]) {
		NSWindowController *winController = [eachItem windowController];

		if (winController == self) {
			continue;
		}

		if ([winController isKindOfClass:[self class]]) {
			// exchange...
			CMRMainBrowser = (CMRBrowser *)winController;
			break;
		}
	}

	// Dispose...
	if (CMRMainBrowser == self) {
		CMRMainBrowser = nil;
	}
}

- (void)dealloc
{
	[CMRPref removeObserver:self forKeyPath:kObservingKey];
	if ([[[BSQuickLookPanelController sharedInstance] qlPanelParent] windowController] == self) {
		[[BSQuickLookPanelController sharedInstance] setQlPanelParent:nil];
	}

	// dispose main browser...
	if (CMRMainBrowser == self) {
		[self exchangeOrDisposeMainBrowser];
	}

    [self setKeepPaths:nil];
	[m_addBoardSheetController release];
	[m_editBoardSheetController release];
//	[[[self scrollView] horizontalRulerView] release];
	
	[super dealloc];
}

#pragma mark -
- (void)didChangeThread
{
	[super didChangeThread];
	// 履歴メニューから選択した可能性もあるので、
	// 表示したスレッドを一覧でも選択させる
	[self selectRowWithCurrentThread];
}

- (void)document:(NSDocument *)aDocument willRemoveController:(NSWindowController *)aController
{
	[self setCurrentThreadsList:nil];
	[super document:aDocument willRemoveController:aController];
}

- (BOOL)shouldShowContents
{
	return (NSHeight([[self textView] visibleRect]) > 0);
}

- (BOOL)shouldLoadWindowFrameUsingCache
{
	return NO;
}

- (void)closeWindowOfAlert:(NSAlert *)alert
{
    UTILDebugWrite(@"Override to block!");
    [self cleanUpItemsToBeRemoved:nil willReload:NO];
}

- (IBAction)showBoardInspectorPanel:(id)sender
{
	NSString *board = [[self currentThreadsList] boardName];
	if (board) [[BSBoardInfoInspector sharedInstance] showInspectorForTargetBoard:board];
}

- (NSArray *)keepPaths
{
    return m_keepPaths;
}

- (void)setKeepPaths:(NSArray *)array
{
    [array retain];
    [m_keepPaths release];
    m_keepPaths = array;
}

- (void)storeKeepPath:(CMRAutoscrollCondition)type
{
    NSIndexSet *rows = [[self threadsListTable] selectedRowIndexes];
    NSArray *tmp = nil;
    if (rows && ([rows count] > 0)) {
        tmp = [[self currentThreadsList] tableView:[self threadsListTable] threadFilePathsArrayAtRowIndexes:rows];
    }
    [self setKeepPaths:tmp];
    [self setKeepCondition:type];
}

- (CMRAutoscrollCondition)keepCondition
{
    return m_keepCondition;
}

- (void)setKeepCondition:(CMRAutoscrollCondition)type
{
    m_keepCondition = type;
}
@end


@implementation CMRBrowser(ThreadContents)
- (void)addThreadTitleToHistory
{
	NSString *threadTitleAndBoardName;
	BSTitleRulerView *ruler = (BSTitleRulerView *)[[self scrollView] horizontalRulerView];

	[super addThreadTitleToHistory];

	threadTitleAndBoardName = [self titleForTitleBar];
	[ruler setTitleStr:(threadTitleAndBoardName ? threadTitleAndBoardName : @"")];
	[ruler setPathStr:[self path]];
}
@end


@implementation CMRBrowser(SelectingThreads)
- (unsigned int)numberOfSelectedThreads
{
	unsigned int count = [[self threadsListTable] numberOfSelectedRows];

	// 選択していないが表示している
	if ((count == 0) && [self shouldShowContents]) {
		return [super numberOfSelectedThreads];
	}
	return count;
}

- (NSArray *)selectedThreads
{
    NSTableView *tv = [self threadsListTable];
    NSString *path = [self shouldShowContents] ? [self path] : nil;
    NSArray *array = [[self currentThreadsList] tableView:tv threadAttibutesArrayAtRowIndexes:[tv selectedRowIndexes] exceptingPath:path];
    if (!array) {
        array = [NSArray array];
    }
    if ([self shouldShowContents] && [self path]) {
        id obj = [super selectedThread];
        if (obj) {
            return [array arrayByAddingObject:obj];
        }
    }
    return array;
}

- (NSArray *)selectedThreadsReallySelected
{
	NSTableView *tableView = [self threadsListTable];
	NSIndexSet	*selectedRows = [tableView selectedRowIndexes];
	CMRThreadsList	*threadsList = [self currentThreadsList];
	if (!threadsList || !selectedRows || [selectedRows count] == 0) {
		return [NSArray array];
	}

	return [threadsList tableView:tableView threadAttibutesArrayAtRowIndexes:selectedRows exceptingPath:nil];
}
@end
