/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigDecimal;
import java.math.BigInteger;

import net.morilib.lisp.util.Utils;

public final class LispSmallInt extends LispInteger {
	
	//
	private int value;
	
	
	/*package*/ LispSmallInt(int value) {
		this.value = value;
	}
	
	
	public LispNumber add(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					value + c.getRealDouble(),
					c.getImagDouble());
		} else if(x instanceof LispSmallInt) {
			long vr1 = value;
			long vr2 = x.getExactSmallInt();
			
			return LispInteger.valueOf(vr1 + vr2);
		} else if(x instanceof LispBigInt) {
			BigInteger vb = BigInteger.valueOf(value);
			
			return LispInteger.valueOf(vb.add(x.getBigInteger()));
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger vb = BigInteger.valueOf(value);
			BigInteger nd = r.getDenominator();
			BigInteger nn = vb.multiply(
					r.getDenominator()).add(r.getNumerator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value + ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber div(LispNumber x) {
		if(x instanceof LispComplex) {
			double xr = ((LispComplex)x).getRealDouble();
			double xi = ((LispComplex)x).getImagDouble();
			double n  = value;
			
			if(xr == 0.0) {
				return LispComplex.newComplex(0, -n / xi);
			} else {
				return LispComplex.newComplex(
						(n * xr)    / (xr * xr + xi * xi),
						(-(n * xi)) / (xr * xr + xi * xi));
			}
		} else if(x instanceof LispSmallInt) {
			return LispRational.newRational(
					value, x.getExactSmallInt());
		} else if(x instanceof LispBigInt) {
			BigInteger vb = BigInteger.valueOf(value);
			
			return LispRational.newRational(vb, x.getBigInteger());
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger vb = BigInteger.valueOf(value);
			BigInteger nd = r.getNumerator();
			BigInteger nn = vb.multiply(r.getDenominator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value / ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			if(c.getRealDouble() == 0.0) {
				return LispComplex.newComplex(
						0, value * c.getImagDouble());
			} else {
				return LispComplex.newComplex(
						value * c.getRealDouble(),
						value * c.getImagDouble());
			}
		} else if(x instanceof LispSmallInt) {
			long vr1 = value;
			long vr2 = x.getExactSmallInt();
			
			return LispInteger.valueOf(vr1 * vr2);
		} else if(x instanceof LispBigInt) {
			BigInteger vb = BigInteger.valueOf(value);
			
			return LispInteger.valueOf(vb.multiply(x.getBigInteger()));
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger vb = BigInteger.valueOf(value);
			BigInteger nd = r.getDenominator();
			BigInteger nn = vb.multiply(r.getNumerator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value * ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;
			
			return LispComplex.newComplex(
					value - c.getRealDouble(),
					-c.getImagDouble());
		} else if(x instanceof LispSmallInt) {
			long vr1 = value;
			long vr2 = x.getExactSmallInt();
			
			return LispInteger.valueOf(vr1 - vr2);
		} else if(x instanceof LispBigInt) {
			BigInteger vb = BigInteger.valueOf(value);
			
			return LispInteger.valueOf(vb.subtract(x.getBigInteger()));
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger vb = BigInteger.valueOf(value);
			BigInteger nd = r.getDenominator();
			BigInteger nn = vb.multiply(
					r.getDenominator()).subtract(r.getNumerator());
			
			return LispRational.newRational(nn, nd);
		} else if(x instanceof LispDouble) {
			return new LispDouble(
					value - ((LispDouble)x).doubleValue());
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public LispNumber uminus() {
		return new LispSmallInt(-value);
	}
	
	
	public int signum() {
		return Utils.signum(value);
	}

	
	public boolean isEqualTo(LispNumber x) {
		if(x instanceof LispComplex) {
			return false;
		} else if(x instanceof LispSmallInt) {
			return value == x.getExactSmallInt();
		} else if(x instanceof LispBigInt) {
			return false;
		} else if(x instanceof LispRational) {
			return false;
		} else if(x instanceof LispDouble) {
			return value == ((LispDouble)x).doubleValue();
		}
		throw new IllegalArgumentException(x.toString());
	}
	

	public boolean isLessThan(LispReal x) {
		if(x instanceof LispSmallInt) {
			return value < x.getExactSmallInt();
		} else if(x instanceof LispBigInt) {
			BigInteger vb = BigInteger.valueOf(value);
			
			return vb.compareTo(x.getBigInteger()) < 0;
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger vb = BigInteger.valueOf(value);
			BigInteger n1 = vb.multiply(r.getDenominator());
			BigInteger n2 = r.getNumerator();
			
			return n1.compareTo(n2) < 0;
		} else if(x instanceof LispDouble) {
			return value < x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}

	
	public boolean isMoreThan(LispReal x) {
		if(x instanceof LispSmallInt) {
			return value > x.getExactSmallInt();
		} else if(x instanceof LispBigInt) {
			BigInteger vb = BigInteger.valueOf(value);
			
			return vb.compareTo(x.getBigInteger()) > 0;
		} else if(x instanceof LispRational) {
			LispRational r = (LispRational)x;
			BigInteger vb = BigInteger.valueOf(value);
			BigInteger n1 = vb.multiply(r.getDenominator());
			BigInteger n2 = r.getNumerator();
			
			return n1.compareTo(n2) > 0;
		} else if(x instanceof LispDouble) {
			return value > x.getRealDouble();
		}
		throw new IllegalArgumentException(x.toString());
	}
	
	
	public LispNumber toInexact() {
		return new LispDouble(value);
	}
	
	
	public boolean equals(Object x) {
		if(x instanceof LispSmallInt) {
			return value == ((LispSmallInt)x).value;
		}
		return false;
	}
	
	
	public int hashCode() {
		int l = 17;
		
		l = 37 * l + value;
		return l;
	}
	
	
	public String toString() {
		return Integer.toString(value);
	}


	public String print() {
		return Integer.toString(value);
	}


	public String getResult() {
		return Integer.toString(value);
	}
	
	
	public LispString toLispString(int radix) {
		if(radix < 2 || radix > 36) {
			throw new IndexOutOfBoundsException("radix is out of range");
		}
		
		return new LispString(Integer.toString(value, radix));
	}
	
	
	public boolean isOne() {
		return value == 1;
	}


	@Override
	public BigInteger getBigInteger() {
		return BigInteger.valueOf(value);
	}


	@Override
	public int getInt() {
		return value;
	}
	
	
	public long getLong() {
		return value;
	}
	
	
	public BigDecimal getBigDecimal() {
		return new BigDecimal(value);
	}


	@Override
	public int getExactSmallInt() {
		return value;
	}


	@Override
	public double getRealDouble() {
		return value;
	}

}
