package com.sanpudo.formula;

import java.math.BigDecimal;

/**
 * 式を構成するトークン。
 * 
 * @author Sanpudo.
 */
abstract class Token {

	/** 親のトークン。このトークンが式全体の場合はnull。 */
	Token parent;

	/** 式全体。 */
	String formula;

	/** 式における位置。 */
	int location;

	/** コンストラクタ。 */
	Token(String formula, int location) {
		this.formula = formula;
		this.location = location;
		parent = null;
	}

	/** このトークンが値であるときtrueを返す。 */
	boolean isValue() {
		return this instanceof Value;
	}

	/** このトークンが演算子、右括弧、区切り文字であるときtrueを返す。 */
	boolean isNonValue() {
		return isOperator() || isRPar() || isDelimiter();
	}

	/** このトークンが即値であるときtrueを返す。 */
	boolean isDirectValue() {
		return this instanceof DirectValue;
	}

	/** このトークンが即値であるときtrueを返す。 */
	boolean isConstant() {
		return this instanceof Constant;
	}

	/** このトークンが左括弧であるときtrueを返す。 */
	boolean isLPar() {
		return this instanceof LPar;
	}

	/** このトークンが右括弧であるときtrueを返す。 */
	boolean isRPar() {
		return this instanceof RPar;
	}

	/** このトークンが演算子であるときtrueを返す。 */
	boolean isOperator() {
		return this instanceof Operator;
	}

	/** このトークンが関数であるときtrueを返す。 */
	boolean isFunction() {
		return this instanceof Function;
	}

	/** このトークンがコンテナ(関数または左括弧)であるときtrueを返す。 */
	boolean isContainer() {
		return this instanceof Container;
	}

	/** このトークンが区切り文字であるときtrueを返す。 */
	boolean isDelimiter() {
		return this instanceof Delimiter;
	}

	/** ルートのトークンを返す。 */
	Token root() {
		if (this.parent == null) {
			return this;
		}
		return this.parent.root();
	}

	/** このトークンが閉じていないコンテナの中にある場合そのコンテナを、そうでない場合はnullを返す。 */
	Container recentOpenContainer() {
		if (this.parent == null) {
			return null;
		} else if ((this.parent.isContainer())
				&& ((Container) (this.parent)).isOpen()) {
			return (Container) (this.parent);
		} else {
			return this.parent.recentOpenContainer();
		}
	}

	/** このトークンが閉じていない関数の中にある場合その関数を、そうでない場合はnullを返す。 */
	Function recentOpenFunc() {
		if (this.parent == null) {
			return null;
		} else if ((this.parent.isFunction())
				&& ((Function) (this.parent)).isOpen()) {
			return (Function) (this.parent);
		} else {
			return this.parent.recentOpenFunc();
		}
	}

	/**
	 * このトークンを上にたどって、指定したオペレータの第1パラメータになるトークンを 返す
	 */
	Token getF1(Operator op) {
		// 最上位
		if (parent == null) {
			return this;
		}
		// 閉じていない左括弧の子
		if (parent.isContainer()) {
			if (((Container) parent).isOpen()) {
				return this;
			}
		}
		// opより弱いオペレータ
		if (parent.isOperator()) {
			if (op.stronger((Operator) (this.parent))) {
				return this;
			}
		}
		return parent.getF1(op);
	}

	/**
	 * トークンの構造が完全であるかチェックする。左括弧は右括弧で解消されている事 演算子のパラメータが全て解消していること。
	 */
	abstract boolean isComplete();

	/**
	 * トークンの構造が完全で無いときに。その理由メッセージを返す。 そうでないときはnullを返す。
	 */
	abstract String imcompleteMessage();

	/** token文字列を返す。 */
	abstract String strToken();

	/** 正規化した式を返す。 */
	abstract String normalized();

	/** doubleの評価値を返す。 */
	abstract double value() throws FormulaEvaluatorException;

	/** BigDecimalの評価値を返す。 */
	abstract BigDecimal value(Rounding rounding)
			throws FormulaEvaluatorException;
}
