package com.sanpudo.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * システム定義済み関数。doubleの関数は全て、java.lang.Mathに定義された同名のメソッドと同じ。
 * 
 * @author Sanpudo.
 */
public class SystemFunctions extends FunctionAdaptor {

	/** コンストラクタ。 */
	SystemFunctions() {
		super();
	}

	/**
	 * 絶対値を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの絶対値
	 */
	public double abs(double d) {
		return Math.abs(d);
	}

	/**
	 * 絶対値を返す。
	 * 
	 * @param bd
	 *            BigDecimal値
	 * @return bdの絶対値
	 */
	public BigDecimal abs(BigDecimal bd) {
		return bd.abs();
	}

	/**
	 * 立方根を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの立方根
	 */
	public double cbrt(double d) {
		return Math.cbrt(d);
	}

	/**
	 * 引数の値以上で最小の整数を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの値以上で最小の整数
	 */
	public double ceil(double d) {
		return Math.ceil(d);
	}

	/**
	 * 余弦(cos)を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの余弦(cos)
	 */
	public double cos(double d) {
		return Math.cos(d);
	}

	/**
	 * 自然対数の底eの累乗値を返す。
	 * 
	 * @param d
	 *            double値
	 * @return 自然対数の底eのd乗
	 */
	public double exp(double d) {
		return Math.exp(d);
	}

	/**
	 * 引数の値以下で最大の整数を返す。
	 * 
	 * @param d
	 *            double値
	 * @return d以下で最大の整数
	 */
	public double floor(double d) {
		return Math.floor(d);
	}

	/**
	 * 自然対数値を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの自然対数値
	 */
	public double log(double d) {
		return Math.log(d);
	}

	/**
	 * 10を低にした自然対数値を返す。
	 * 
	 * @param d double値
	 * @return dの10を低にした自然対数値
	 */
	public double log10(double d) {
		return Math.log10(d);
	}

	/**
	 * 2つの引数の大きい方の値を返す。
	 * 
	 * @param bd1
	 *            BigDecimal値
	 * @param bd2
	 *            BigDecimal値
	 * @return bd1, bd2の大きい方の値
	 */
	public BigDecimal max(BigDecimal bd1, BigDecimal bd2) {
		return bd1.max(bd2);
	}

	/**
	 * 2つの引数の大きい方の値を返す。
	 * 
	 * @param d1
	 *            double値
	 * @param d2
	 *            double値
	 * @return d1, d2の大きい方の値
	 */
	public double max(double d1, double d2) {
		return Math.max(d1, d2);
	}

	/**
	 * 2つの引数の小さい方の値を返す。
	 * 
	 * @param bd1
	 *            BigDecimal値
	 * @param bd2
	 *            BigDecimal値
	 * @return bd1, bd2の小さい方の値
	 */
	public BigDecimal min(BigDecimal bd1, BigDecimal bd2) {
		return bd1.min(bd2);
	}

	/**
	 * 2つの引数の小さい方の値を返す。
	 * 
	 * @param d1
	 *            double値
	 * @param d2
	 *            double値
	 * @return d1, d2の小さい方の値
	 */
	public double min(double d1, double d2) {
		return Math.min(d1, d2);
	}

	/**
	 * 1番目の引数を、2番目の引数で累乗した値を返す。第2パラメータは整数で無ければならない。
	 * 
	 * @param bd1
	 *            BigDecimal値
	 * @param bd2
	 *            BigDecimal値(整数である事)
	 * @return db1のdb2乗の値
	 */
	public BigDecimal pow(BigDecimal bd1, BigDecimal bd2)
			throws FunctionEvalException {
		try {
			return bd1.pow(bd2.intValueExact());
		} catch (ArithmeticException e) {
			throw new FunctionEvalException(Messages.INVALID_POW, e);
		}
	}

	/**
	 * 1番目の引数を、2番目の引数で累乗した値を返す。
	 * 
	 * @param d1
	 *            double値
	 * @param d2
	 *            double値
	 * @return d1のd2乗の値
	 */
	public double pow(double d1, double d2) {
		return Math.pow(d1, d2);
	}

	/**
	 * 引数の値にもっとも近い整数を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dに最も近い整数
	 */
	public double rint(double d) {
		return Math.rint(d);
	}

	/**
	 * 正弦(sin)を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの正弦(sin)
	 */
	public double sin(double d) {
		return Math.sin(d);
	}

	/**
	 * 平方根を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの平方根
	 */
	public double sqrt(double d) {
		return Math.sqrt(d);
	}

	/**
	 * 正接(tan)を返す。
	 * 
	 * @param d
	 *            double値
	 * @return dの正接(tan)
	 */
	public double tan(double d) {
		return Math.tan(d);
	}

	/**
	 * 指定した小数点以下桁数で四捨五入する。指定桁数が負の場合、小数点以上の桁数で四捨五入する。 第2パラメータは整数で無ければならない。
	 * 
	 * @param bd
	 *            BigDecimal値
	 * @param scale
	 *            BigDecimal値(整数である事)
	 * @return bdの小数点scale未満を四捨五入した値
	 * 
	 * @exception FunctionEvalException
	 *                scaleが整数でない
	 */
	public BigDecimal round(BigDecimal bd, BigDecimal scale)
			throws FunctionEvalException {
		return this.setScale(bd, scale, RoundingMode.HALF_UP);
	}

	/**
	 * 指定した小数点以下桁数を切り捨てる。指定桁数が負の場合、小数点以上の桁数で切り捨てる。 第2パラメータは整数で無ければならない。
	 * 
	 * @param bd
	 *            BigDecimal値
	 * @param scale
	 *            BigDecimal値(整数である事)
	 * @return bdの小数点scale未満を切捨てた値
	 * 
	 * @exception FunctionEvalException
	 *                scaleが整数でない
	 */
	public BigDecimal down(BigDecimal bd, BigDecimal scale)
			throws FunctionEvalException {
		return this.setScale(bd, scale, RoundingMode.DOWN);
	}

	/**
	 * 指定した小数点以下桁数を切り上げる。指定桁数が負の場合、小数点以上の桁数で切り上げる。 第2パラメータは整数で無ければならない。
	 * 
	 * @param bd
	 *            BigDecimal値
	 * @param scale
	 *            BigDecimal値(整数である事)
	 * @return bdの小数点scale未満を切り上げた値
	 * 
	 * @exception FunctionEvalException
	 *                scaleが整数でない
	 */
	public BigDecimal up(BigDecimal bd, BigDecimal scale)
			throws FunctionEvalException {
		return this.setScale(bd, scale, RoundingMode.UP);
	}

	/** 切り上げ／切捨て／四捨五入の共通関数。 */
	private BigDecimal setScale(BigDecimal bd, BigDecimal scale,
			RoundingMode rMode) throws FunctionEvalException {
		try {
			return bd.setScale(scale.intValueExact(), rMode);
		} catch (ArithmeticException e) {
			throw new FunctionEvalException(Messages.INVALID_ROUND, e);
		}

	}
}
