package com.sanpudo.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * 演算子。
 * 
 * @author Sanpudo.
 */
class Operator extends Token {
	/** 演算子。 */
	char operator;

	/** 左演算対象。 */
	Token f1;
	/** 右演算対象。 */
	Token f2;

	/** オペレータの種類。 */
	static final char[] operators = { '+', '-', '*', '/', '^' };

	/** オペレータの強さ。 */
	static final int[] strength = { 0, 0, 1, 1, 1 };

	/** コンストラクタ。 */
	Operator(String formula, int location, char operator) {
		super(formula, location);
		this.operator = operator;
		f1 = null;
		f2 = null;
	}

	/** ファクターの構造が完全であるかチェックする。 */
	boolean isComplete() {
		if ((this.f1 == null) || (this.f2 == null)) {
			return false;
		} else {
			return f1.isComplete() && f2.isComplete();
		}
	}

	/** ファクターの構造が完全でないとき、その理由メッセージを返す。 */
	String imcompleteMessage() {
		if (this.f1 == null) {
			return Messages.VALUE_EXPECT;
		}
		if (this.f2 == null) {
			return Messages.OPERATOR_INCOMPELETE;
		}
		String wk = f1.imcompleteMessage();
		if (wk != null) {
			return wk;
		}
		return f2.imcompleteMessage();
	}

	/** 演算対象1の登録。 */
	void bindF1(Token token) {
		this.f1 = token;
		token.parent = this;
	}

	/** 演算対象2の登録。 */
	void bindF2(Token token) {
		this.f2 = token;
		token.parent = this;
	}

	/** 対象のオペレータより強いオペレータであるかどうかを返す。 */
	boolean stronger(Operator op) {
		return this.strength() > op.strength();
	}

	/** オペレータの強さを返す。 */
	int strength() {
		for (int i = 0; i < operators.length; i++) {
			if (operator == operators[i]) {
				return strength[i];
			}
		}
		return -1;
	}

	/** token文字列を返す。 */
	String strToken() {
		return String.valueOf(operator);
	}

	/** 正規化した式を返す。 */
	String normalized() {
		return "(" + f1.normalized() + this.strToken() + f2.normalized() + ")";
	}

	/** doubleの評価値を返す。 */
	double value() throws FormulaEvaluatorException {
		switch (operator) {
		case '+':
			return f1.value() + f2.value();
		case '-':
			return f1.value() - f2.value();
		case '*':
			return f1.value() * f2.value();
		case '/':
			return f1.value() / f2.value();
		case '^':
			return Math.pow(f1.value(), f2.value());
		}
		throw new FormulaEvaluatorException(Messages.INTERNAL_ERROR,
				strToken(), location);
	}

	/** BigDecimalの評価値を返す。 */
	BigDecimal value(Rounding rounding) throws FormulaEvaluatorException {
		switch (operator) {
		case '+':
			return f1.value(rounding).add(f2.value(rounding));
		case '-':
			return f1.value(rounding).subtract(f2.value(rounding));
		case '*':
			return rounding.everyRound(f1.value(rounding).multiply(
					f2.value(rounding)));
		case '/':
			return rounding.everyRound(f1.value(rounding).divide(
					f2.value(rounding), 20, RoundingMode.HALF_UP));
		case '^':
			try {
				return rounding.everyRound(f1.value(rounding).pow(
						f2.value(rounding).intValueExact()));
			} catch (ArithmeticException e) {
				throw new FormulaEvaluatorException(Messages.INVALID_POW, e,
						formula, location);
			}
		}
		throw new FormulaEvaluatorException(Messages.INTERNAL_ERROR, formula,
				location);

	}

}
