package com.sanpudo.formula;

import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * レキシカルアナライザ。
 * 
 * @author Sanpudo.
 */
class Lex {
	static final char lPar = '(';
	static final char rPar = ')';
	static final char delim = ',';
	static Pattern figure;
	static final String spaceRegex = "\\s+";
	static Pattern space;
	static Pattern function;
	Environments environments;

	/** コンストラクタ。 */
	Lex() {
		this.environments = new Environments();
	}

	/** コンストラクタ。 */
	Lex(Environments environments) {
		this.environments = environments;
	}

	/** 分析ステータス。 */
	private static final int REQUEST_VALUE = 0;
	private static final int REQUEST_OPERATOR = 1;

	static {
		figure = Pattern.compile(DirectValue.regex);
		space = Pattern.compile(spaceRegex);
		function = Pattern.compile(Function.regex);
	}

	/** 文字列を分析しトークンの切り出しを行う。 */
	Vector<Token> analyze(String formula) throws FormulaEvaluatorException {
		return analyze(formula, false);
	}

	/** 文字列を分析しトークンの切り出しを行う。 allowParamがtrueの場合、引数文字を含むことが可能。 */
	Vector<Token> analyze(String formula, boolean allowParam)
			throws FormulaEvaluatorException {
		Pattern constant;
		if (allowParam) {
			constant = Constant.patternParam;
		} else {
			constant = Constant.pattern;
		}

		int status = REQUEST_VALUE;
		Vector<Token> tokens = new Vector<Token>();
		String wk = new String(formula);
		while (!wk.equals("")) {
			int location = formula.length() - wk.length();
			// ホワイトスペース読み飛ばし
			Matcher spaceMatcher = space.matcher(wk);
			if (spaceMatcher.lookingAt()) {
				wk = wk.substring(spaceMatcher.end());
				continue;
			}
			// 値要求
			if (status == REQUEST_VALUE) {
				// 左括弧
				if (wk.charAt(0) == lPar) {
					tokens.add(new LPar(formula, location));
					wk = wk.substring(1);
					status = REQUEST_VALUE;
					continue;
				}
				// 即値
				Matcher matcher = figure.matcher(wk);
				if (matcher.lookingAt()) {
					tokens.add(new DirectValue(formula, location, wk.substring(
							0, matcher.end())));
					wk = wk.substring(matcher.end());
					status = REQUEST_OPERATOR;
					continue;
				}
				// 関数
				matcher = function.matcher(wk);
				if (matcher.lookingAt()) {
					String name = wk.substring(0, matcher.end() - 1);
					if (environments.isFunction(name)) {
						FunctionImplementation fun = environments
								.functions(name);
						tokens.add(new Function(formula, location, name, fun
								.numberOfArgs(name), environments));
						wk = wk.substring(matcher.end());
						status = REQUEST_VALUE;
						continue;
					} else {
						throw new FormulaEvaluatorException(
								Messages.UNDKNOWN_FUNC, formula, location);
					}
				}
				// 定数
				matcher = constant.matcher(wk);
				if (matcher.lookingAt()) {
					String name = wk.substring(0, matcher.end());
					if (environments.isConstant(name)) {
						tokens.add(new Constant(formula, location, name,
								environments));
						wk = wk.substring(matcher.end());
						status = REQUEST_OPERATOR;
						continue;
					} else {
						throw new FormulaEvaluatorException(
								Messages.UNDKNOWN_CONST, formula, location);
					}
				}
				// 右括弧(引数なし関数の場合ありうる)
				if (wk.charAt(0) == rPar) {
					tokens.add(new RPar(formula, location));
					wk = wk.substring(1);
					status = REQUEST_OPERATOR;
					continue;
				}
				// 該当なし
				throw new FormulaEvaluatorException(Messages.VALUE_EXPECT,
						formula, location);
				// オペレータ要求
			} else {
				// 右括弧
				if (wk.charAt(0) == rPar) {
					tokens.add(new RPar(formula, location));
					wk = wk.substring(1);
					status = REQUEST_OPERATOR;
					continue;
				}
				// 区切り文字
				if (wk.charAt(0) == delim) {
					tokens.add(new Delimiter(formula, location));
					wk = wk.substring(1);
					status = REQUEST_VALUE;
					continue;
				}
				// 演算子
				char o = wk.charAt(0);
				int i;
				for (i = 0; i < Operator.operators.length; i++) {
					if (o == Operator.operators[i]) {
						tokens.add(new Operator(formula, location, o));
						wk = wk.substring(1);
						status = REQUEST_VALUE;
						break;
					}
				}
				// 該当なし
				if (i >= Operator.operators.length) {
					throw new FormulaEvaluatorException(
							Messages.OPERATOR_EXPECT, formula, location);
				} else {
					continue;
				}
			}

		}
		return tokens;
	}

	/** 関数名として正しいときにtrueを返す。 */
	static boolean isValidFuncName(String funcName) {
		return Lex.function.matcher(funcName + "(").matches();
	}
}
