package com.sanpudo.formula;

import java.math.BigDecimal;

/**
 * 関数のテキスト評価によるFunctionImplementationの実装。
 * 
 * @author Sanpudo.
 */
class FunctionInterpreter implements FunctionImplementation {

	/** 関数名。 */
	private String funcName;

	/** 関数の引数の個数。 */
	private int numberOfArgs;

	/** 定義式。 */
	private String formula;

	/** 環境。 */
	private Environments environments;

	/** コンストラクタ。 */
	FunctionInterpreter(String funcName, int numberOfArgs, String formula,
			Environments environments) throws FormulaEvaluatorException {
		this.funcName = funcName;
		this.numberOfArgs = numberOfArgs;
		this.formula = formula;
		this.environments = environments.clone();
		checkFormula(funcName, numberOfArgs, formula, this.environments);
	}

	/**
	 * 関数名、式が正しいかどうかチェックする。 正しくない場合はFormulaEvaluatorExceptionをスローする。
	 */
	static void checkFormula(String funcName, int numberOfArgs, String formula,
			Environments environments) throws FormulaEvaluatorException {
		// 関数名
		if (!Lex.isValidFuncName(funcName)) {
			throw new FormulaEvaluatorException(Messages.INVALID_FUNC_NAME,
					funcName, 0);
		}
		// システム関数との競合
		if (Environments.systemFunctions.supports(funcName)) {
			throw new FormulaEvaluatorException(Messages.FUNC_NAME_COMFLICT,
					funcName, 0);
		}
		// 引数を定数として登録(仮の値:"0")
		for (int i = 0; i < numberOfArgs; i++) {
			environments.defineUserConstant("$" + Integer.toString(i), "0",
					true, false);
		}
		// 式
		new Parser(formula).parse(new Lex(environments).analyze(formula, true));
	}

	/** このクラスで定義されている関数名であるtrueを返す。 */
	public boolean supports(String name) {
		return this.funcName.equals(name);
	}

	/** このクラスで定義されている関数名の配列を返す。 */
	public String[] names() {
		String[] ns = new String[1];
		ns[0] = funcName;
		return ns;
	}

	/** 指定した名前の関数の引数の個数を返す。 */
	public int numberOfArgs(String name) {
		return numberOfArgs;
	}

	/** 指定した名前の関数のdoubleの関数実行結果を返す。 */
	public double value(String name, double[] args)
			throws FunctionEvalException {
		try {
			// 引数を定数として登録
			for (int i = 0; i < numberOfArgs; i++) {
				environments.defineUserConstant("$" + Integer.toString(i),
						Double.toString(args[i]), true, false);
			}
			FormulaEvaluator formulaEvaluator = new FormulaEvaluator(
					environments);
			return formulaEvaluator.dEvaluateParam(formula);
		} catch (FormulaEvaluatorException e) {
			throw new FunctionEvalException(e.getMessage(), e);
		}
	}

	/** 指定した名前の関数のBigDecimalの関数実行結果を返す。 */
	public BigDecimal value(String name, BigDecimal[] args)
			throws FunctionEvalException {
		try {
			// 引数を定数として登録
			for (int i = 0; i < numberOfArgs; i++) {
				environments.defineUserConstant("$" + Integer.toString(i),
						args[i].toString(), true, false);
			}
			FormulaEvaluator formulaEvaluator = new FormulaEvaluator(
					environments);
			return formulaEvaluator.bdEvaluateParam(formula);
		} catch (FormulaEvaluatorException e) {
			throw new FunctionEvalException(e.getMessage(), e);
		}
	}
}
