package com.sanpudo.formula;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Hashtable;
import java.util.Vector;

/**
 * 実行環境。システム定数、ユーザ定義定数、関数などの実装。
 * 
 * @author Sanpudo.
 */
class Environments implements Cloneable {

	/** システム定義定数。 */
	static Hashtable<String, String> predefine;

	/** ユーザ定義定数。 */
	Hashtable<String, String> userDefine;

	/** システム定義関数。 */
	static SystemFunctions systemFunctions;

	/** ユーザ定義関数。 */
	Vector<FunctionImplementation> userFunctions;

	/** BigDecimalのMathContext。 */
	Rounding rounding;

	static {
		predefine = new Hashtable<String, String>();
		// 定義済み定数
		predefine.put("PI", new BigDecimal(Math.PI).toString());
		predefine.put("E", new BigDecimal(Math.E).toString());

		systemFunctions = new SystemFunctions();
	}

	/** コンストラクタ。 */
	protected Environments() {
		userDefine = new Hashtable<String, String>();
		userFunctions = new Vector<FunctionImplementation>();
		rounding = new Rounding(RoundPoint.FINAL, 10, RoundingMode.HALF_UP);
	}

	/** Cloneableの実装。 */
	@SuppressWarnings("unchecked")
	public Environments clone() {
		Environments cloned = new Environments();
		cloned.userDefine = (Hashtable<String, String>) (this.userDefine
				.clone());
		cloned.userFunctions = (Vector<FunctionImplementation>) (this.userFunctions
				.clone());
		rounding = new Rounding(rounding.roundPoint, rounding.scale,
				rounding.rMode);
		return cloned;
	}

	/** BigDecimal計算における小数点以下の桁数を変更する。 */
	void setScale(int scale) {
		rounding = new Rounding(rounding.roundPoint, scale, rounding.rMode);
	}

	/** BigDecimal計算における小数点以下の桁数と端数処理方法を変更する。 */
	void setRounding(int scale, RoundingMode rMode) {
		rounding = new Rounding(rounding.roundPoint, scale, rMode);
	}

	/** BigDecimal計算における端数処理のコンテクスト(端数処理時点、小数点以下の桁数および端数処理方法)を変更する。 */
	void setRounding(RoundPoint roundPoint, int scale, RoundingMode rMode)
			throws FormulaEvaluatorException {
		if (rMode == RoundingMode.UNNECESSARY) {
			throw new FormulaEvaluatorException(Messages.INVALID_ROUNDING,
					"UNNECESSARY", 0);
		}
		this.rounding = new Rounding(roundPoint, scale, rMode);
	}

	/**
	 * ユーザ定義定数を登録する。allowParamがtrueの場合は、%0～%9の引数の登録を許す。
	 * overrideがtrueの場合はシステム定義済み定数と同名の定数を登録を許す。
	 */
	void defineUserConstant(String name, String value, boolean allowParam,
			boolean override) throws FormulaEvaluatorException {

		if (!Constant.isValidName(name, allowParam)) {
			throw new FormulaEvaluatorException(Messages.INVALID_CONST_NAME,
					name, 0);
		}
		if (!override) {
			if (predefine.containsKey(name)) {
				throw new FormulaEvaluatorException(
						Messages.CONST_NAME_COMFLICT, name, 0);
			}
		}
		userDefine.put(name, value);

	}

	/**
	 * ユーザ定義定数を登録する。システム定義済み定数と同名の定数を登録しようとすると 例外をスローする。
	 */
	void defineUserConstant(String name, String value)
			throws FormulaEvaluatorException {
		defineUserConstant(name, value, false, false);
	}

	/** 指定した名前が定数として登録されている時にtrueを返す。 */
	boolean isConstant(String name) {
		return predefine.containsKey(name) || userDefine.containsKey(name);
	}

	/**
	 * ユーザ定義関数を登録する。システム定義済み関数と同名の定数を登録しようとすると 例外をスローする。
	 */
	void defineUserFunction(FunctionImplementation func)
			throws FormulaEvaluatorException {
		String[] names = func.names();
		for (int i = 0; i < names.length; i++) {
			// 正しくない関数名
			if (!Lex.isValidFuncName(names[i])) {
				System.out.println(names[i]);
				throw new FormulaEvaluatorException(Messages.INVALID_FUNC_NAME,
						names[i], 0);
			}
			// システム関数との競合
			if (systemFunctions.supports(names[i])) {
				throw new FormulaEvaluatorException(
						Messages.FUNC_NAME_COMFLICT, names[i], 0);
			}
		}
		userFunctions.add(0, func);
	}

	/** 指定した名前のクラスのインスタンスを生成しユーザ定義関数として登録する。 */
	void loadUserFunction(String className) throws FormulaEvaluatorException {
		FunctionImplementation func = null;
		try {
			Class<?> c = ClassLoader.getSystemClassLoader()
					.loadClass(className);
			Object o = c.newInstance();
			if (o instanceof FunctionImplementation) {
				func = (FunctionImplementation) o;
				this.defineUserFunction(func);
			} else {
				throw new FormulaEvaluatorException(
						Messages.CLASS_INCOMPATIBLE, className, 0);
			}
		} catch (Exception e) {
			throw new FormulaEvaluatorException(Messages.CLASS_LOADING_ERROR,
					e, className, 0);
		}

	}

	/** 指定した名前が関数として登録されている時にtrueを返す。 */
	boolean isFunction(String name) {
		return functions(name) != null;
	}

	/** 指定した名前の関数を実装した関数実装を返す。関数が存在しない場合はnullを返す。 */
	FunctionImplementation functions(String name) {
		if (systemFunctions.supports(name)) {
			return systemFunctions;
		}
		for (FunctionImplementation fun : userFunctions) {
			if (fun.supports(name)) {
				return fun;
			}
		}
		return null;
	}
}
