# coding: UTF-8
require File.expand_path(File.dirname(__FILE__) + '../../../spec_helper')

include RDGC::Map
include RDGC::Article

describe RDGC::Map::Floor, "is related to Article" do

  before(:all) do
    @block1 = Block.create(0, 19, 0, 9)
    @block2 = Block.create(0, 19, 10, 19)

    @room1 = Room.create(2, 7, 2, 7)
    @room2 = Room.create(10, 15, 12, 16)

    @road1 = Road.create(5, 5, 8, 9)
    @road2 = Road.create(5, 13, 9, 9)
    @road3 = Road.create(13, 13, 10, 11)

    @block1.room = @room1
    @block2.room = @room2
    @block1.add_road(@road1)
    @block1.add_road(@road2)
    @block2.add_road(@road3)

    @board = Board.create_from_blocks([@block1, @block2])
  end

  before do
    @floor = Floor.create_from_board(@board)

    @step = Step.create_up_step(5, 6)
    @player = Character::Player.new
    @player.set(12, 13)
  end

  it "#add_article will add article on floor" do
    @floor.articles.should be_empty

    @step.floor.should be_nil
    @floor.add_article(@step)
    @floor.articles.should == [@step]
    @step.floor.should == @floor

    @player.floor.should be_nil
    @floor.add_article(@player)
    @floor.articles.should == [@step, @player]
    @player.floor.should == @floor

    @floor.add_article(@step)
    @floor.articles.should == [@step, @player]
  end

  it "#articles_for provides articles at coordinates" do
    @floor.add_article(@step)
    @floor.add_article(@player)

    @floor.movable?(1, 1).should be_false
    @floor.articles_for(1, 1).should be_empty
    @floor.movable?(4, 4).should be_true
    @floor.articles_for(4, 4).should be_empty

    @floor.articles_for(5, 6).should == [@step]
    @floor.articles_for(12, 13).should == [@player]
  end

  it "#overlap_articles_for provides overlaped articles" do
    @floor.add_article(@step)
    @floor.add_article(@player)

    pl = Character::Player.new

    @floor.overlap_articles_for(nil).should be_empty

    pl.set(1, 1)
    @floor.movable?(1, 1).should be_false
    @floor.overlap_articles_for(pl).should be_empty

    pl.set(4, 4)
    @floor.movable?(4, 4).should be_true
    @floor.overlap_articles_for(pl).should be_empty

    pl.set(12, 13)
    @player.overlapable?.should be_false
    @floor.overlap_articles_for(pl).should be_empty

    pl.set(5, 6)
    @step.overlapable?.should be_true
    @floor.overlap_articles_for(pl).should == [@step]

    @step.disable
    @floor.overlap_articles_for(pl).should be_empty
  end

  it "#collision_articles_for provides not overlapable articles at coordinates" do
    @floor.add_article(@step)
    @floor.add_article(@player)

    @floor.movable?(1, 1).should be_false
    @floor.collision_articles_for(1, 1).should be_empty

    @floor.movable?(4, 4).should be_true
    @floor.collision_articles_for(4, 4).should be_empty

    @step.overlapable?.should be_true
    @floor.collision_articles_for(5, 6).should be_empty

    @player.overlapable?.should be_false
    @floor.collision_articles_for(12, 13).should == [@player]

    @player.disable
    @floor.collision_articles_for(12, 13).should be_empty
  end

  it "#clean_articles will delete disabled articles" do
    @floor.add_article(@step)
    @floor.add_article(@player)
    @floor.articles.should == [@step, @player]

    @step.disable
    @floor.clean_articles
    @floor.articles.should == [@player]
  end

  it "#player provides Player's object" do
    @floor.player.should be_nil
    @floor.add_article(@player)
    @floor.player.should == @player
    @floor.add_article(@player)
    @floor.player.should == @player
  end

  it "#search provides in-range alived articles from coordinates" do
    # roomをキャラで埋め尽くす
    @room1.each do |x, y|
      pl = Character::Player.new
      pl.set(x, y)
      @floor.add_article(pl)
    end

    @floor.search(3, 4, -1).should be_empty

    ret1 = @floor.search(3, 4, 1)
    ret1.size.should == 5
    ret1.all?{|pl| pl.distance(3, 4) <= 1}.should be_true

    ret2 = @floor.search(3, 4, 2)
    ret2.size.should == 12 # 左が欠ける
    ret2.all?{|pl| pl.distance(3, 4) <= 2}.should be_true

    ret3 = @floor.search(3, 4, 3)
    ret3.size.should == 20 # 左上が欠ける
    ret3.all?{|pl| pl.distance(3, 4) <= 3}.should be_true

    ret0 = @floor.search(4, 4, 0)
    ret0.size.should == 1
    ret0.first.exist?(4, 4).should be_true
    ret0.first.disable
    ret0.first.exist?(4, 4).should be_false

    @floor.search(3, 4, 2).size.should == 11 # 中央の右がdisable
  end

end
