<?php
/**
 * Research Artisan Lite: Website Access Analyzer
 * Copyright (C) 2009 Research Artisan Project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * @copyright Copyright (C) 2009 Research Artisan Project
 * @license GNU General Public License (see license.txt)
 * @author ossi
 */
class Track {

  public static function numberReplace($replace, $subject) {
    $nums = array('1','2','3','4','5','6','7','8','9','0');
    return str_ireplace($nums, $replace, $subject);
  }

  public static function checkUrl($url) {
    return preg_match('/^(http|https)(:\/\/)[^\/].*$/', $url) ? true : false;
  }

  public static function checkIpAddressFormat($value) {
    return preg_match('/^([0-9])+\.([0-9])+\.([0-9])+\.([0-9])+$/', $value) || preg_match('/^([0-9])+\.([0-9])+\.([0-9])+\.([0-9])+\:([0-9])+$/', $value) ? true : false;
  }

  public static function checkEmail($email) {
    return preg_match('/^[^<]+@[^>]+$/', $email) ? true : false;
  }

  public static function getRemoteHost($remoteAddr) {
    return gethostbyaddr($remoteAddr);
  }

  public static function getDomain($remoteHost, $remoteAddr=null) {
    $domain = Config::NO_DATA;
    if (!is_null($remoteAddr) && $remoteHost == $remoteAddr) return $domain;
    if (self::checkIpaddressFormat($remoteHost)) return $domain;
    $pos = substr_count($remoteHost, '.');
    switch ($pos) {
      case 0:
        break;
      case 1:
        $domain = $remoteHost;
        break;
      default:
        $remoteHost = strtolower(strrev($remoteHost));
        list($d1,$d2,$d3) = explode('.',$remoteHost);
        $d1 = strrev($d1);  
        $d2 = strrev($d2);  
        $d3 = strrev($d3);
        if ($d1 != 'jp') {
          $domain = $d2. '.'. $d1;
        } else {
          if (in_array($d2, Config::$jpSecondLevelDomains)) {
            $domain = $d3. '.'. $d2. '.'. $d1;
          } else {
            $domain = $d2. '.'. $d1;
          }
        }
        break;
    }
    return $domain;
  }

  public static function getJpDomain($domain) {
    $jpDomain = '';
    $revDomain = strtolower(strrev($domain));
    $pos = substr_count($revDomain, '.');
    switch ($pos) {
      case 0:
        break;
      case 1:
        list($d1, $d2) = explode('.',$revDomain);
        $d1 = strrev($d1);
        $d2 = strrev($d2);
        if ($d1 == 'jp') if (!in_array($d2, Config::$otherJpSecondLevelDomains)) $jpDomain = $domain;
        break;
      default:
        list($d1, $d2, $d3) = explode('.',$revDomain);
        $d1 = strrev($d1);
        $d2 = strrev($d2);
        $d3 = strrev($d3);
        if ($d1 == 'jp' ) {
          if (in_array($d2, Config::$orgJpSecondLevelDomains)) {
            if (!in_array($d3, Config::$otherJpThirdLevelDomains)) $jpDomain = $domain;
          } else {
            if (!in_array($d2, Config::$otherJpSecondLevelDomains)) $jpDomain = $domain;
          }
        }
        break;
    }
    return $jpDomain;
  }

  public static function getCountry($domain) {
    $country = Config::NO_DATA;
    $revDomain = strtolower(strrev($domain));
    $pos = substr_count($revDomain, '.');
    switch ($pos) {
      case 0:
        break;
      default:
        list($d1, $d2) = explode('.',$revDomain);
        $d1 = strrev($d1);
        $d2 = strrev($d2);
        if (isset(Config::$countryTopLevelDomains[$d1])) $country = Config::$countryTopLevelDomains[$d1];
        break;
    }
    return $country;
  }

  public static function getPref($remoteHost, $remoteAddr, $domain) {
    $pref = Config::NO_DATA;
    if ($remoteHost !== $remoteAddr) {
      $prefKeyword = self::getPrefKeywordByHost($remoteHost, $domain);  
      if (!is_null($prefKeyword)) {
        if (isset(Config::$prefKeywords[$prefKeyword])) $pref = Config::$prefs[Config::$prefKeywords[$prefKeyword]];
      }
      if ($pref == Config::NO_DATA) {
        $prefDomain = self::getPrefByDomain($domain);
        if (!is_null($prefDomain)) $pref = Config::$prefs[$prefDomain];
      }
      if ($pref == Config::NO_DATA) {
        if (isset(Config::$prefOrgDomains[$domain])) $pref = Config::$prefs[Config::$prefOrgDomains[$domain]];
      }
    }
    return $pref;
  }

  public static function getPrefKeywordByHost($remoteHost, $domain) {
    $prefKeyword = null;
    $hostArray = explode('.', $remoteHost);
    switch ($domain) {
      case 'odn.ne.jp':
      case 'odn.ad.jp':
        $prefKeyword = substr($hostArray[0], 0, 3);
        break;
      case 'ocn.ne.jp':
        $prefKeyword = $hostArray[1];
        break;
      case 'infoweb.ne.jp':
        switch ($hostArray[1]) {
          case 'adsl':
            $prefKeyword = substr($hostArray[0], 2, 4);
            break;
          case 'catv':
            $prefKeyword = substr($hostArray[0], 0, 4);
            break;
          case 'ppp':
          case 'mobile':
          case 'em':
            break;
          default:
            if (strlen($hostArray[1]) == 4) $prefKeyword = $hostArray[1];
            break;
        }
        break;
      case 'mesh.ad.jp':
        $prefKeyword = $hostArray[1];
        break;
      case 'plala.or.jp':
        if (count($hostArray) == 7) $prefKeyword = $hostArray[2];
        break;
      case 'dion.ne.jp':
        break;
      case 'hi-ho.ne.jp':
        $prefKeyword = substr($hostArray[0], 0, 3);
        break;
      case 'so-net.ne.jp':
        $prefKeyword = substr($hostArray[1], 0, 4);
        break;
      case 'dti.ne.jp':
        $prefKeyword = substr($hostArray[1], 0, 4);
        break;
      case 'alpha-net.ne.jp':
        $prefKeyword = $hostArray[1];
        break;
      case 'vectant.ne.jp':
        $rtn = preg_match('/[^a-z]*([a-z]+)[^a-z]*$/', $hostArray[1], $mathces);
        if (is_array($mathces) && count($mathces) > 1) $prefKeyword = $mathces[1];
        break;
      case 'att.ne.jp':
        $tmp = str_ireplace(array('ipc','dsl','ftth','newfamily'), '', $hostArray[1]);
        $tmp = self::numberReplace('', $tmp);
        $prefKeyword = substr($tmp, 1);
        break;
      case 'bbiq.jp':
        $prefKeyword = self::numberReplace('', $hostArray[1]);
        break;
      case 'coara.or.jp':
        $tmpArray = explode('-', $hostArray[0]);
        $tmpArray[0] = self::numberReplace('', $tmpArray[0]);
        $prefKeyword = str_ireplace('ap', '', $tmpArray[0]);
        break;
      case 'highway.ne.jp':
        $tmp = str_ireplace('ip-', '', $hostArray[1]);
        $tmp = str_ireplace('e-', '', $tmp);
        $prefKeyword = $tmp;
        break;
      case 'interq.or.jp':
        $prefKeyword = substr($hostArray[0], 0, stripos($hostArray[0], '-'));
        break;
      case 'mbn.or.jp':
        $prefKeyword = $hostArray[1];
        break;
      case 'psinet.ne.jp':
        $prefKeyword = str_ireplace('fli-', '', $hostArray[1]);
        break;
      case 'sannet.ne.jp':
        $prefKeyword = $hostArray[1];
        break;
      case 'uu.net':
        $prefKeyword = self::numberReplace('', $hostArray[2]);
        break;
      case 'zero.ad.jp':
        $tmpArray = explode('-', $hostArray[0]);
        if ($tmpArray[0] == '-') {
          $tmp = $tmpArray[1];
        } else {
          $tmp = $tmpArray[0];
        }
        $prefKeyword = self::numberReplace('', $tmp);
        break;
      case 'pias.ne.jp':
        $tmp = $hostArray[0];
        $tmp = self::numberReplace('', $tmp);
        $prefKeyword = str_ireplace('-', '', $tmp);
        break;
      case 'nttpc.ne.jp':
        $prefKeyword = substr($hostArray[2], stripos($hostArray[2], '-'));
        break;
      case 'interlink.or.jp':
        $prefKeyword = self::numberReplace('', $hostArray[1]);
        break;
      case 'kcom.ne.jp':
        $tmp = $hostArray[0];
        $tmp = self::numberReplace('', $tmp);
        $prefKeyword = str_ireplace('-', '', $tmp);
        break;
      case 'isao.net':
        $prefKeyword = self::numberReplace('', $hostArray[1]);
        break;
    }
    return $prefKeyword;
  }

  public static function getPrefByDomain($domain) {
    $pref = null;
    $revDomain = strtolower(strrev($domain));
    $pos = substr_count($revDomain, '.');
    switch ($pos) {
      case 0:
        $d1 = '';  
        $tmp = '';
        break;
      case 1:
        list($d1,$tmp) = explode('.',$revDomain);
        break;
      default:
        list($d1,$d2,$tmp) = explode('.',$revDomain);
        break;
    }
    $d1 = strrev($d1);  
    $tmp = strrev($tmp);
    if ($d1 == 'jp') {
      $tmp = self::numberReplace('', $tmp);
      if (substr_count($tmp, '-') > 0) {
        $words = explode('-', $tmp);
        foreach ($words as $value) {
          if (isset(Config::$prefWords[$value])) {
            $pref = Config::$prefWords[$value];
            break;
          }
        }
      } else {
        if (isset(Config::$prefWords[$tmp])) $pref = Config::$prefWords[$tmp];
      }
    }
    return $pref;
  }

  public static function getRefererInfo($httpReferer, $url) {
  
    $refererInfo = array('keyword' => '', 'engine' => '', 'referer_title' => '', 'referer_host' => '');
  
    $referer = strtolower($httpReferer);
    if (trim($referer) == '') {
      $refererInfo['referer_title'] = Config::FROM_BOOKMARK;
      $refererInfo['referer_host'] = Config::FROM_BOOKMARK;
      return $refererInfo;
    }
    if (!self::checkUrl($referer)) {
      $refererInfo['referer_title'] = $httpReferer;
      return $refererInfo;
    }
    $referers = parse_url($referer);
    if (!$referers) {
      $refererInfo['referer_title'] = Config::FROM_BOOKMARK;
      $refererInfo['referer_host'] = Config::FROM_BOOKMARK;
      return $refererInfo;
    }
    $urls = parse_url($url);
    if (!$urls) {
      $refererInfo['referer_title'] = Config::FROM_BOOKMARK;
      $refererInfo['referer_host'] = Config::FROM_BOOKMARK;
      return $refererInfo;
    }

    $extractReferer = true;

    $refererHost = strtolower($referers['host']);
    $urlHost = strtolower(isset($urls['host']) ? $urls['host'] : null);
    $query = isset($referers['query']) ? $referers['query'] : '';
    $domain = self::getDomain($refererHost);
    if (substr_count($domain, '.') > 0) {
      list($hostName,$trash) = explode('.',$domain);  
    } else {
      $hostName = $refererHost;
    }

    $refererUrl = strtolower($referers['scheme']. '://'. $refererHost. $referers['path']);
    if (!is_null($urlHost) && strpos($refererUrl, constant(Config::DEFINE_SITEURL)) !== false) {
      $extractReferer = false;
    }
    if (self::checkIpaddressFormat($refererHost)) {
      if (strpos($query , 'q=cache:') !== false) {
        $extractReferer = false;
      }
    }
    if ($hostName == 'yahoofs') {
      $extractReferer = false;
    }

    $refererInfo['referer_title'] = $httpReferer;

    if ($extractReferer) {

      $refererInfo['referer_host'] = $refererHost;

      $keyValue = '';
      if (strlen(trim($query)) > 0) {
        if (isset(Config::$searchEngines[$hostName])) {
          $extractStrings = Config::$searchEngines[$hostName];
        } else {
          $extractStrings = Config::$searchEngines['default'];
        }
        $queryPregString = '';
        foreach ($extractStrings as $queryString) {
          $queryPregString .= preg_quote($queryString). '|';
        }
        $queryPregString = substr($queryPregString, 0, strlen($queryPregString) - 1);
        if ((!in_array($hostName, Config::$searchEnginesContainMail)) || 
            (in_array($hostName, Config::$searchEnginesContainMail) && substr_count($refererHost, 'mail') == 0)) {
          preg_match('/('. $queryPregString. ').+?(&|$)/', $referer, $matches);
          $keyValue = isset($matches[0]) ? preg_replace('/('. $queryPregString. '|&)/', '', $matches[0]) : '';
        }

        $keyValue_backup = $keyValue;
        $keyValue = urldecode($keyValue);
  
        switch($hostName) {
          case 'yahoo':
            if (strpos($query , 'ei=utf-8') !== false || strpos($query , 'fr=ieas') !== false) {  
              $keyValue = RaUtil::convertEncoding($keyValue, 'UTF-8, JIS, SJIS, ASCII, EUC-JP');  
            } else {
              $keyValue = RaUtil::convertEncoding($keyValue, 'EUC-JP, UTF-8, JIS, SJIS, ASCII');
            }
            break;
          case 'google':
            if (strpos($query , 'ie=sjis') !== false || strpos($query , 'ie=shift_jis') !== false) {
              if (!RaUtil::checkDoubleStr($keyValue_backup)) {
                $keyValue = RaUtil::convertEncoding($keyValue, 'UTF-8, JIS, SJIS, ASCII, EUC-JP');
              } else {
                $keyValue = RaUtil::convertEncoding($keyValue, 'SJIS, JIS, ASCII, UTF-8, EUC-JP');
              }
            } elseif (strpos($query , 'ie=utf-8') !== false) {  
              $keyValue = RaUtil::convertEncoding($keyValue, 'UTF-8, JIS, SJIS, ASCII, EUC-JP');
            } elseif (strpos($query , 'ie=euc-jp') !== false) {
              $keyValue = RaUtil::convertEncoding($keyValue, 'EUC-JP, UTF-8, JIS, SJIS, ASCII');  
            } else {
              $keyValue = RaUtil::convertEncoding($keyValue, 'UTF-8, JIS, SJIS, ASCII, EUC-JP');  
            }
            break;
          case 'msn':
            if (strpos($query , 'cp=932') !== false) {
              $keyValue = RaUtil::convertEncoding($keyValue, 'SJIS, JIS, ASCII, UTF-8, EUC-JP');
            } else {
              $keyValue = RaUtil::convertEncoding($keyValue, 'auto');
            }
            break;
          case 'goo':
            if (strpos($query , 'ie=sjis') !== false) {
              $keyValue = RaUtil::convertEncoding($keyValue, 'SJIS, JIS, ASCII, EUC-JP, UTF-8');
            } else {
              $keyValue = RaUtil::convertEncoding($keyValue, 'EUC-JP, UTF-8, JIS, SJIS, ASCII');
            }
            break;
          default:
            $keyValue = RaUtil::convertEncoding($keyValue, 'EUC-JP, UTF-8, JIS, SJIS, ASCII');
            break;
        }

        if (strlen(trim($keyValue)) > 0) {
          $refererInfo['keyword'] = $keyValue;
          $refererInfo['referer_title'] = '@' . $hostName . ':[' . $keyValue . ']';
          $refererInfo['engine'] = $hostName;
        }

      }
 
    }

    return $refererInfo;
  }

  public static function getOs($httpUserAgent) {
    $os = Config::NO_DATA;
    $httpUserAgent = strtolower(str_replace(' ', '', $httpUserAgent));
    foreach (Config::$osLists as $k => $v) {
      if (strpos($httpUserAgent, $k) !== false) {
        $os = $v;
        break;
      }
    }
    return $os;
  }

  public static function getBrowser($httpUserAgent) {
    $browser = Config::NO_DATA;
    $httpUserAgent = strtolower(str_replace(' ', '', $httpUserAgent));
    foreach (Config::$browserLists as $k => $v) {
      if (strpos($httpUserAgent, $k) !== false) {
        $browser = $v;
        break;
      }
    }
    return $browser;
  }

  public static function getBrowserVersion($httpUserAgent) {
    $brow_ver = Config::NO_DATA;
    $regExp = '';
    foreach (Config::$browserLists as $k => $v) $regExp .= preg_quote($k). '|';
    $regExp = substr($regExp, 0, strlen($regExp) - 1);
    $rtn = preg_match('/('. $regExp. ')[\/: ]([\d\.]+)/', strtolower($httpUserAgent), $matches);
    if ($rtn !== false && is_array($matches)) {
      foreach ($matches as $k => $v) {
        if ($k == 2) {
          $brow_ver = $v;
          break;
        }
      }
    }
    return $brow_ver;
  }

  public static function getCrawler($httpUserAgent) {
    $crawler = '';
    $httpUserAgent = strtolower(str_replace(' ', '', $httpUserAgent));
    foreach (Config::$crawlerLists as $k => $v) {
      if (strpos($httpUserAgent, $k) !== false) {
        $crawler = $v;
        break;
      }
    }
    return $crawler;
  }

  public static function setCookie($key, $value, $enableDays=null, $path=null) {
    if (is_null($enableDays)) $enableDays = Config::COOKIE_ENABLE_DAYS;
    RaUtil::setCookie($key, $value, $enableDays, $path);
  }

  public static function clearCookie($key, $path=null) {
    RaUtil::setCookie($key, '', 0, $path);
  }

  public static function getTrackInfo() {
    $trackInfo = array();

    $now = $_SERVER['REQUEST_TIME'];
    $trackInfo['yyyy'] = date('Y', $now);
    $trackInfo['mm'] = date('m', $now);
    $trackInfo['dd'] = date('d', $now);
    $trackInfo['hh'] = date('H', $now);
    $trackInfo['mi'] = date('i', $now);
    $trackInfo['ss'] = date('s', $now);
    $trackInfo['weekday'] = date('w', mktime($trackInfo['hh'], $trackInfo['mi'], $trackInfo['ss'], $trackInfo['mm'], $trackInfo['dd'], $trackInfo['yyyy']));
    
    $trackInfo['http_user_agent'] = $_SERVER['HTTP_USER_AGENT'];
    if (strlen($trackInfo['http_user_agent']) == 0) $trackInfo['http_user_agent'] = Config::NO_DATA;
    $trackInfo['os'] = self::getOs($trackInfo['http_user_agent']);
    $trackInfo['browser'] = self::getBrowser($trackInfo['http_user_agent']);
    $trackInfo['brow_ver'] = self::getBrowserVersion($trackInfo['http_user_agent']);
    $trackInfo['crawler'] = self::getCrawler($trackInfo['http_user_agent']);
    
    $trackInfo['remote_addr'] = $_SERVER['REMOTE_ADDR'];
    $trackInfo['remote_host'] =  self::getRemoteHost($trackInfo['remote_addr']);
    $trackInfo['remote_host'] = $trackInfo['remote_addr'] != $trackInfo['remote_host'] ? $trackInfo['remote_host'] : Config::NO_DATA;
    $trackInfo['domain'] = self::getDomain($trackInfo['remote_host'], $trackInfo['remote_addr']);
    $trackInfo['jpdomain'] = self::getJpDomain($trackInfo['domain']);
    $trackInfo['country'] = self::getCountry($trackInfo['domain']);
    $trackInfo['pref'] = self::getPref($trackInfo['remote_host'], $trackInfo['remote_addr'], $trackInfo['domain']);
    
    return $trackInfo;
  }

  public static function writeLog($trackInfo) {

    try {

      if (is_array($trackInfo)) {
        $log = new Log();
        $log->setValue('dd', $trackInfo['dd']);
        $log->setValue('logtype', $trackInfo['logtype']);
        $log->setValue('uid', $trackInfo['uid']);
        $log->setValue('hh', $trackInfo['hh']);
        $log->setValue('mi', $trackInfo['mi']);
        $log->setValue('ss', $trackInfo['ss']);
        $log->setValue('weekday', $trackInfo['weekday']);
        $log->setValue('remote_addr', $trackInfo['remote_addr']);
        $log->setValue('remote_host', $trackInfo['remote_host']);
        $log->setValue('domain', $trackInfo['domain']);
        $log->setValue('jpdomain', $trackInfo['jpdomain']);
        $log->setValue('country', $trackInfo['country']);
        $log->setValue('pref', $trackInfo['pref']);
        $log->setValue('title', $trackInfo['title']);
        $log->setValue('url', $trackInfo['url']);
        $log->setValue('screenwh', $trackInfo['screenwh']);
        $log->setValue('screencol', $trackInfo['screencol']);
        $log->setValue('jsck', $trackInfo['jsck']);
        $log->setValue('os', $trackInfo['os']);
        $log->setValue('browser', $trackInfo['browser']);
        $log->setValue('brow_ver', $trackInfo['brow_ver']);
        $log->setValue('crawler', $trackInfo['crawler']);
        $log->setValue('keyword', $trackInfo['keyword']);
        $log->setValue('engine', $trackInfo['engine']);
        $log->setValue('referer_title', $trackInfo['referer_title']);
        $log->setValue('referer_host', $trackInfo['referer_host']);
        $log->setValue('http_user_agent', $trackInfo['http_user_agent']);
        $log->setValue('http_referer', $trackInfo['http_referer']);
        $log->save();
        if ($trackInfo['url'] != $trackInfo['title'] && trim($trackInfo['title']) != '') {
          $title = new Title();
          $title->setIgnore(true);
          $title->setValue('url', $trackInfo['url']);
          $title->setValue('title', $trackInfo['title']);
          $title->setValue('logtype', $trackInfo['logtype']);
          $title->save();
        }
      }

    } catch (Exception $exception) {
    }

  }

  public static function doScript() {
    header('Content-type: application/x-javascript');

    $host = $_SERVER['SERVER_NAME'];
    $httpUserAgent = $_SERVER['HTTP_USER_AGENT'];
    $remoteAddr = $_SERVER['REMOTE_ADDR'];
    $protocol = isset($_SERVER['HTTPS']) ? 'https://' : 'http://';
    $trackUrl = $protocol. $host. str_replace(basename($_SERVER['SCRIPT_NAME']), 'track.php', $_SERVER['PHP_SELF']);
    $clickRel = Config::CLICK_REL;
    $clickNormal = Config::NORMAL_ACCESS;
    $clickLink = Config::CLICK_LINK;
    $clickBtn = Config::CLICK_BTN;
    $clickAdsense = Config::CLICK_ADSENSE;
    $uid = isset($_COOKIE[Config::COOKIE_UID]) ? trim($_COOKIE[Config::COOKIE_UID]) : null;
    if (is_null($uid)) $uid = md5($remoteAddr. $httpUserAgent. microtime(). rand(0, 30000));
    self::setCookie(Config::COOKIE_UID, $uid);

print <<<END
(function(){
  var referer=(location.href!=parent.location.href)?parent.document.referrer:document.referrer,
  requestUrl=location.protocol+'//'+location.host+location.pathname+location.search,
  userAgent=navigator.userAgent.toLowerCase(),
  browser={
    Version: (userAgent.match(/.+(?:rv|it|ra|ie)[\/: ]([\d.]+)/) || [])[1],
    Safari: /webkit/.test(userAgent),
    Opera: /opera/.test(userAgent),
    IE: /msie/.test(userAgent) && !/opera/.test(userAgent),
    Mozilla: /mozilla/.test(userAgent) && !/(compatible|webkit)/.test(userAgent),
    Chrome: /chrome/.test(userAgent)
  },
  RaTracker = function(){
    var t=(new Date()).getTime();
    this.accessTime=t;
    return this;
  };
  RaTracker.prototype = {
    type: {
      Load: '$clickNormal',
      Click: '$clickLink',
      BtnClick: '$clickBtn',
      Adsense: '$clickAdsense'
    },
    clickRel: '$clickRel',
    trackingUrl: '$trackUrl',
    accessTime: null,
    getTitle: function(){
      if(document.getElementById('title')){
        var elmT=document.getElementsByTagName('title')[0];
        return (browser.IE)?elmT.innerHTML:elmT.firstChild.nodeValue;
      }else{
        return document.title;
      }
    },
    getTrackingUrl: function(url,title,referer,type){
      return this.trackingUrl+'?LT='+type
                             +'&RF='+encodeURIComponent(referer)
                             +'&UR='+encodeURIComponent(url)
                             +'&TI='+encodeURIComponent(title)
                             +'&SW='+screen.width
                             +'&SH='+ screen.height
                             +'&SC='+screen.colorDepth
                             +'&s='+Math.floor(Math.random()*100);
    },
    sendServer: function(url,dom){
      if(dom){
        var itemElm=document.getElementById('item');
        var bodyElm=(itemElm)?itemElm:document.getElementsByTagName('body').item(0);
        var scriptElm=document.createElement('script');
        scriptElm.setAttribute('src',url);
        scriptElm.setAttribute('defer','defer');
        bodyElm.appendChild(scriptElm);
        this.wait(0.4);
      }else{
        document.write('<script type="text/javascript" src="'+url+'" defer="defer"></script>');
      }
      return this;
    },
    clickTrack: function(e){
      var url=title=rel='';
      var targetElm=this.getTargetElm(e);
      var targetName=targetElm.nodeName.toLowerCase();
      var clickCheck=function(url,rel,clickRel){
        return (url&&url.match("^(https?|ftp):\/\/")&&(!url.match(location.host)||rel==clickRel))?true:false;
      }
      switch(targetName){
        case 'a':
          url=targetElm.href;
          title=(browser.IE)?targetElm.innerText:targetElm.text;
          rel=(targetElm.rel!==undefined)?targetElm.rel:'';
          if(clickCheck(url,rel,this.clickRel))
            this.sendServer(this.getTrackingUrl(url,title,requestUrl,this.type.Click),true);
          break;
        case 'input':
          if (targetElm.type.toLowerCase() == 'button' || targetElm.type.toLowerCase() == 'submit') {
            url=requestUrl+'#'+targetElm.value;
            title='['+targetElm.value+'] ('+this.getTitle()+')';
            this.sendServer(this.getTrackingUrl(url,title,requestUrl,this.type.BtnClick),true);
          }
          break;
        default:
          if (targetElm.parentNode.href!==undefined) {
            url=targetElm.parentNode.href;
            title=(targetElm.alt!==undefined)?targetElm.alt:((browser.IE)?targetElm.innerText:targetElm.firstChild.nodeValue);
            rel=(targetElm.parentNode.rel!==undefined)?targetElm.parentNode.rel:'';
            if(clickCheck(url,rel,this.clickRel))
              this.sendServer(this.getTrackingUrl(url,title,requestUrl,this.type.Click),true);
          }
          break;
      }
    },
    adsenseElms: new Array(),
    adsenseOnFocus: false,
    adsenseTargetElm: null,
    adsenseTrack: function(){
      if(this.adsenseOnFocus){
        for(var i=0;i<this.adsenseElms.length;i++){
          if(this.adsenseElms[i]==this.adsenseTargetElm){
            var url=encodeURIComponent('Unit='+(i+1)+',Size='+this.adsenseElms[i].width+'x'+this.adsenseElms[i].height);
            this.sendServer(this.getTrackingUrl(url,url,requestUrl,this.type.Adsense),true);
            this.adsenseOnFocus=false;
            break;
          }
        }
      }
    },
    adsenseSearch: function(e){
      var iframeElms=document.getElementsByTagName('iframe');
      for(var i=0;i<iframeElms.length;i++){
        if(iframeElms[i].src.indexOf('googlesyndication.com')>-1||iframeElms[i].src.indexOf('googleads.g.doubleclick.net')>-1){
          this.adsenseElms[this.adsenseElms.length]=iframeElms[i];
          if(browser.IE){
            this.addEvent('focus',RaTracker.transfer.adsenseFocus,iframeElms[i]);
            this.addEvent('blur',RaTracker.transfer.adsenseBlur,iframeElms[i]);
            this.addEvent('beforeunload',RaTracker.transfer.adsenseTrack,window);
          }else{
            this.addEvent('mouseover',RaTracker.transfer.adsenseFocus	,iframeElms[i]);
            this.addEvent('mouseout',RaTracker.transfer.adsenseBlur,iframeElms[i]);
            if(browser.Opera){
              this.addEvent('unload',RaTracker.transfer.adsenseTrack,window);
            }else{
              this.addEvent('beforeunload',RaTracker.transfer.adsenseTrack,window);
            }
          }
        }
      }
    },
    adsenseFocus: function(e){
      this.adsenseOnFocus=true;
      this.adsenseTargetElm=this.getTargetElm(e);
    },
    adsenseBlur: function(){
      this.adsenseOnFocus=false;
      this.adsenseTargetElm=null;
    },
    documentReady: function(callback) {
      if (browser.IE) {
        (function(){
          try {
            document.documentElement.doScroll('left');
          } catch(error) {
            setTimeout(arguments.callee, 0);
            return;
          }
          callback.apply(document);
        })();
      } else {
        if (document.addEventListener) {
          document.addEventListener('DOMContentLoaded',callback,false);
        } else {
          window.attachEvent ? window.attachEvent('onload',callback) : window.addEventListener('load',callback,false);
        }
      }
      return document;
    },
    addEvent: function(e, callback, obj) {
      if ((obj.nodeType !== undefined && (obj.nodeType === 1 || obj.nodeType === 9)) || obj===window) 
        obj.attachEvent ? obj.attachEvent('on'+e,callback) : obj.addEventListener(e,callback,false);
      return obj;
    },
    getTargetElm: function(e) {
      return window.event?window.event.srcElement:e.target;
    },
    wait: function(second){
      var w=(new Date()).getTime()+(second*1000);
      while(true){
        if((new Date()).getTime()>w){return;}
      }
    },
    doTracking: function(){
      this.sendServer(this.getTrackingUrl(requestUrl,this.getTitle(),referer,this.type.Load),false);
      this.addEvent('click',RaTracker.transfer.clickTrack,document);
      if(browser.IE) this.addEvent('contextmenu',RaTracker.transfer.clickTrack,document);
      this.documentReady(RaTracker.transfer.adsenseSearch);
      return this;
    }
  }
  RaTracker.transfer = {
    clickTrack: function(e){__RaTracker.clickTrack(e);},
    adsenseTrack: function(e){__RaTracker.adsenseTrack(e);},
    adsenseSearch: function(e){__RaTracker.adsenseSearch(e);},
    adsenseFocus: function(e){__RaTracker.adsenseFocus(e);},
    adsenseBlur: function(e){__RaTracker.adsenseBlur(e);}
  }
  window.__RaTracker = new RaTracker();
  __RaTracker.doTracking();
})()
END;

  }

  public static function doTrack() {
    $trackInfo = self::getTrackInfo();

    $nocount = isset($_COOKIE[Config::getCookieKeyAdminNocount()]) ? Config::ON : Config::OFF;    

    if (isset($_REQUEST['LT'])) {

      $trackInfo['uid'] = isset($_COOKIE[Config::COOKIE_UID]) ? trim($_COOKIE[Config::COOKIE_UID]) : null;
      if (is_null($trackInfo['uid'])) {
        $trackInfo['uid'] = md5($trackInfo['remote_addr']. $trackInfo['http_user_agent']);
        $trackInfo['jsck'] = Config::JC_ENABLED_J;
      } else {
        $trackInfo['jsck'] = Config::JC_ENABLED_JC;
      }

      header('Content-type: application/x-javascript');
      if ($nocount == Config::ON) return;
      $trackInfo['logtype'] = $_REQUEST['LT'];
      $trackInfo['http_referer'] = $_REQUEST['RF'];
      $trackInfo['url'] = $_REQUEST['UR'];
      $trackInfo['title'] = trim($_REQUEST['TI']) != '' ? RaUtil::convertEncoding($_REQUEST['TI'], 'auto') : $trackInfo['url'];
      if ($trackInfo['logtype'] == Config::CLICK_ADSENSE) {
        $trackInfo['url'] = urldecode($trackInfo['url']);
        $trackInfo['title'] = urldecode($trackInfo['title']);
      }
      $trackInfo['screenwh'] = $_REQUEST['SW'] . 'x' . $_REQUEST['SH'];
      if (strlen($_REQUEST['SW'] . $_REQUEST['SH']) == 0) $trackInfo['screenwh'] = Config::NO_DATA;
      $trackInfo['screencol'] = $_REQUEST['SC'] . 'bit';
      if (strlen($_REQUEST['SC']) == 0) $trackInfo['screencol'] = Config::NO_DATA;
      $refererInfo = self::getRefererInfo($trackInfo['http_referer'], $trackInfo['url']);
      $trackInfo['keyword'] = $refererInfo['keyword'];
      $trackInfo['engine'] = $refererInfo['engine'];
      $trackInfo['referer_title'] = $refererInfo['referer_title'];
      $trackInfo['referer_host'] = $refererInfo['referer_host'];

    } else {

      $uid = md5($trackInfo['remote_addr']. $trackInfo['http_user_agent']);
      self::setCookie(Config::COOKIE_UID, $uid);
      $trackInfo['uid'] = isset($_COOKIE[Config::COOKIE_UID]) ? trim($_COOKIE[Config::COOKIE_UID]) : null;
      if (is_null($trackInfo['uid'])) {
        $trackInfo['uid'] = $uid;
        $trackInfo['jsck'] = Config::JC_ENABLED_N;
      } else {
        $trackInfo['jsck'] = Config::JC_ENABLED_C;
      }
      if (strpos($trackInfo['http_user_agent'], 'J-PHONE') !== false ||
          strpos($trackInfo['http_user_agent'], 'Vodafone') !== false ||
          strpos($trackInfo['http_user_agent'], 'Softbank') !== false  ) {
        header('Content-type: image/png');
        print base64_decode('iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVQI12NgYAAAAAMAASDVlMcAAAAASUVORK5CYII');
      } else {
        header('Content-type: image/gif');
        print base64_decode('R0lGODlhAQABAGAAACH5BAEKAP8ALAAAAAABAAEAAAgEAP8FBAA7');
      }
      if ($nocount == Config::ON) return;
      $trackInfo['logtype'] = Config::NORMAL_ACCESS;
      $page = isset($_REQUEST['page']) ? $_REQUEST['page'] : null;
      $page = RaUtil::convertEncoding($page, 'SJIS, JIS, ASCII, EUC-JP, UTF-8');
      $trackInfo['http_referer'] = Config::FROM_NO_SCRIPT;
      $trackInfo['url'] = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : null;
      if (strlen(trim($trackInfo['url'])) == 0) {
        $trackInfo['url'] = (strlen(trim($page)) == 0) ? Config::NO_DATA : $page;
      }
      $trackInfo['title'] = $trackInfo['url'];
      if (strlen(trim($page)) != 0) $trackInfo['title'] = $page;
      $trackInfo['screenwh'] = Config::NO_DATA;
      $trackInfo['screencol'] = Config::NO_DATA;
      $trackInfo['keyword'] = '';
      $trackInfo['engine'] = '';
      $trackInfo['referer_title'] = Config::FROM_NO_SCRIPT;
      $trackInfo['referer_host'] = '';

    }

    self::writeLog($trackInfo);
  }

  public static function doPhpTrack($title=null) {
    $trackInfo = self::getTrackInfo();
    $trackInfo['uid'] = md5($trackInfo['remote_addr']. $trackInfo['http_user_agent']);

    $nocount = isset($_COOKIE[Config::getCookieKeyAdminNocount()]) ? Config::ON : Config::OFF;    
    if ($nocount == Config::ON) return;
    $trackInfo['logtype'] = Config::NORMAL_ACCESS;
    $trackInfo['http_referer'] = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER']: '';
    $host = $_SERVER['SERVER_NAME'];
    $protocol = isset($_SERVER['HTTPS']) ? 'https://' : 'http://';
    $trackInfo['url'] = $protocol. $host. $_SERVER['REQUEST_URI'];
    $trackInfo['title'] = !is_null($title) && trim($title) != '' ? RaUtil::convertEncoding($title, 'auto') : $trackInfo['url'];
    $trackInfo['screenwh'] = Config::NO_DATA;
    $trackInfo['screencol'] = Config::NO_DATA;
    $trackInfo['jsck'] = Config::NO_DATA;
    $refererInfo = self::getRefererInfo($trackInfo['http_referer'], $trackInfo['url']);
    $trackInfo['keyword'] = $refererInfo['keyword'];
    $trackInfo['engine'] = $refererInfo['engine'];
    $trackInfo['referer_title'] = $refererInfo['referer_title'];
    $trackInfo['referer_host'] = $refererInfo['referer_host'];
    self::writeLog($trackInfo);
  }

}
?>
