<?php
/**
 * Research Artisan Lite: Website Access Analyzer
 * Copyright (C) 2009 Research Artisan Project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * @copyright Copyright (C) 2009 Research Artisan Project
 * @license GNU General Public License (see license.txt)
 * @author ossi
 */

/**
 * RA Main
 */
class Ra {
/* ------------------------------------------------------------------------ */

/* -- Private Property -- */
  /**
   * request
   * @var RaRequest
   */
  private $_request = null;
  /**
   * session
   * @var RaSession
   */
  private $_session = null;
  /**
   * message
   * @var RaMessage
   */
  private $_message = null;
  /**
   * controller
   * @var RaController
   */
  private $_controller = null;
  /**
   * action
   * @var RaAction
   */
  private $_action = null;
  /**
   * exception
   * @var RaException
   */
  private $_exception = null;
  /**
   * nextControllCount
   * @var int
   */
  private $_nextControllCount = 0;
/* ------------------------------------------------------------------------ */

/* -- Public Method -- */
  /**
   * Constructer
   */
  public function __construct() {
    $this->_request = new RaRequest();
    $this->_session = new RaSession();
    $this->_message = new RaMessage();
    $this->_controller = $this->_request->get('controller');
    $this->_action = $this->_request->get('action');
  }

  /**
   * Execute
   */
  public function execute() {
    try {
      set_error_handler(array('RaException', 'raErrorHandler'), error_reporting());
      $result = new RaResult();
      $this->_checkController($this->_controller);
      $controller = $this->_loadController($this->_request, $this->_session, $this->_message, $result, $this->_controller, $this->_action);
      $result = $this->_doAction($controller, $this->_action);
      if (!is_object($result)) {
        $msg = RaConfig::CTRL_RTN_NONE_ERR_MSG. ' (Controller: '. $this->_controller. ', Action: '. $this->_action. ')';
        $result = new RaResult();
        $this->_controller = RaConfig::ERR_CTRL_NAME;
        $this->_action = RaConfig::ERR_ACTION_NAME;
        throw new RaException($msg, RaConfig::CTRL_RTN_NONE_ERR_CODE);
      }
      if (!is_null($result->getNextController())) {
        $this->_nextControllCount++;
        if ($this->_nextControllCount > RaConfig::NEXT_CONTROLL_LIMIT_TIME) {
          $this->_controller = RaConfig::ERR_CTRL_NAME;
          $this->_action = RaConfig::ERR_ACTION_NAME;
          throw new RaException(RaConfig::LIMIT_NEXT_CONTROLL_ERR_MSG. RaConfig::NEXT_CONTROLL_LIMIT_TIME, RaConfig::LIMIT_NEXT_CONTROLL_ERR_CODE); 
        }
        $this->_controller = $result->getNextController();
        $this->_action = $result->getNextAction();
        $this->execute();
      } else {
        $this->_showView($this->_request, $this->_session, $this->_message, $this->_controller, $this->_action, $result, $this->_exception);
        restore_error_handler();
      }
    } catch (Exception $exception) {
      if (($exception->getNativeError()) || ($this->_controller == RaConfig::ERR_CTRL_NAME && $this->_action == RaConfig::ERR_ACTION_NAME)) {
        $this->_showError($this->_request, $this->_session, $this->_message, $result, $exception);
        restore_error_handler();
      } else {
        $this->_exception = $exception;
        $this->_controller =  RaConfig::ERR_CTRL_NAME;
        $this->_action =  RaConfig::ERR_ACTION_NAME;
        $this->execute();
      }
    }
  }
/* ------------------------------------------------------------------------ */

/* -- Private Method -- */
  /**
   * checkController
   * @param string $controller controllerName
   */
  private function _checkController($controller) {
    $controllerFile = RA_CORE_DIR. RA_CTRL_DIR. ucfirst($controller). 'Controller.php';
    $controllerName = ucfirst($controller). 'Controller';
    if (file_exists($controllerFile)) {
      $decleardClasses = get_declared_classes();
      if (!in_array($controllerName, $decleardClasses)) include $controllerFile;
    } else {
      throw new RaException(RaConfig::URL_NOTFOUND_ERR_MSG, RaConfig::URL_NOTFOUND_ERR_CODE);
    }
  }

  /**
   * loadController
   * @param RaRequest $request Request
   * @param RaSession $session Session
   * @param RaMessage $message Message
   * @param RaResult $result Result
   * @param string $controller controllerName
   * @param string $action actionName
   * @return RaController controller
   */
  private function _loadController(RaRequest $request, RaSession $session, RaMessage $message, RaResult $result, $controller, $action) {
    $controllerName = ucfirst($controller). 'Controller';
    return new $controllerName($request, $session, $message, $result, $controller, $action);
  }

  /**
   * doAction
   * @param RaController $controller controller
   * @param string $action actionName
   * @return function user_func
   */
  private function _doAction(RaController $controller, $action) {
    if (!method_exists($controller, $action)) throw new RaException(RaConfig::URL_NOTFOUND_ERR_MSG, RaConfig::URL_NOTFOUND_ERR_CODE);
    return call_user_func(array($controller, $action));
  }

  /**
   * showView
   * @param RaRequest $request Request
   * @param RaSession $session Session
   * @param RaMessage $message Message
   * @param string $controller controllerName
   * @param string $action actionName
   * @param RaResult $result Result
   * @param Exception $exception Exception
   */
  private function _showView(RaRequest $request, RaSession $session, RaMessage $message, $controller, $action, RaResult $result, $exception=null) {
    if (!is_null($exception)) {
      $message->setCode($exception->getCode());
      $message->setMessage($exception->getMessage());
    }
    $rv = new RaView($request, $session, $message, $result);
    $rv->showView($controller, $action);
  }

  /**
   * showError
   * @param RaRequest $request Request
   * @param RaSession $session Session
   * @param RaMessage $message Message
   * @param RaResult $result Result
   * @param Exception $exception Exception
   */
  private function _showError(RaRequest $request, RaSession $session, RaMessage $message, RaResult $result, Exception $exception) {
    $rv = new RaView($request, $session, $message, $result);
    $rv->showError($exception);
  }
/* ------------------------------------------------------------------------ */

}
?>
