// Main.cs
// 2008/12/16

using System;
using System.IO;

// MainApp
static class MainApp {

	static void Main()
	{
		try {
			QVorbis.Module.Initialize();

			using (var t = new Tool()) {
				t.Run();
			}

			Console.Error.WriteLine("DONE.");

		} catch (Exception e) {
			Console.Error.WriteLine("Error: " + e);
		}
	}

} // MainApp

// Tool
class Tool : IDisposable {

	private String _oggPath;
	private String _wavPath;

	public Tool()
	{
		_oggPath = String.Empty;
		_wavPath = String.Empty;
	}

	public void Dispose()
	{
	}

	public void Run()
	{
		if (!CheckArgs()) {
			return;
		}

		using (var oggs = File.OpenRead(_oggPath)) {
			var ogg = new QOgg.Checker(oggs);

			ogg.ReadSetup();

			using (var setup = new QVorbis.Setup()) {
				setup.Create(ogg.Id, ogg.Setup);

				using (var decoder = new QVorbis.Decoder()) {
					decoder.Create(setup, 0x10000);

					using (var wav = new Audio.WaveWriter(
						setup.GetChannels(), setup.GetSamplingRate())) {
						wav.Create(_wavPath);

						UInt64 pos = 0;

						while (ogg.ReadPage()) {
							Byte[][] packets = ogg.Packets;

							for (Int32 i = 0; i < packets.Length; i++) {
								Single[][] samples = decoder.Decode(packets[i]);
								if (samples == null) {
									continue;
								}

								Int32 length = samples[0].Length;

								if ((ogg.Flags & QOgg.Header.HT_EOS) != 0 &&
									i == packets.Length - 1) {
									// Last Packet
									length = (Int32)(ogg.GranulePosition - pos);
								}

								wav.Write(samples, length);

								pos += (UInt64)length;
							}
						}

						wav.Finish();
					}
				}
			}
		}
	}

	private bool CheckArgs()
	{
		String[] args = Environment.GetCommandLineArgs();
		if (args.Length != 3) {
			Console.WriteLine("QDecoder [ogg File] [wave File]");
			return false;
		}

		_oggPath = args[1];
		_wavPath = args[2];

		return true;
	}

} // Tool

