/* FrameDecoder.c */
/* 2008/11/07     */

#include "StdAfx.h"

#include "FrameDecoder.h"

#include "VorbisDecoder.h"

/* */

BOOL FrameDecoder_Setup(
	DecoderContext_t* ctx,
	DecoderSetup_t*   setup,
	FrameDecoder_t*   frame)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	INT32 i, j;
	INT32 size;

	frame->Setup = setup;

	size = 0;
	for (i = 0; i < setup->Floor1Count; i++) {
		if (size < setup->Floor1Setup[i].Values) {
			size = setup->Floor1Setup[i].Values;
		}
	}

	for (i = 0; i < setup->Channels; i++) {
		frame->Y[i] = (INT16*)MemoryPool_Allocate(pool, sizeof(INT16) * size);
		if (frame->Y[i] == NULL) {
			THROW0(ctx, "FrameDecoder_Setup", "OutOfMemory")
		}

		frame->Sample[i] = (FLOAT*)MemoryPool_Allocate(pool, sizeof(FLOAT) * setup->BlockSize1 / 2);
		if (frame->Sample[i] == NULL) {
			THROW0(ctx, "FrameDecoder_Setup", "OutOfMemory")
		}

		for (j = 0; j < 2; j++) {
			frame->PCM[j][i] = (FLOAT*)MemoryPool_Allocate(pool, sizeof(FLOAT) * setup->BlockSize1);
			if (frame->PCM[j][i] == NULL) {
				THROW0(ctx, "FrameDecoder_Setup", "OutOfMemory")
			}
		}

		frame->Output[i] = (FLOAT*)MemoryPool_Allocate(pool, sizeof(FLOAT) * setup->BlockSize1 / 2);
		if (frame->Output[i] == NULL) {
			THROW0(ctx, "FrameDecoder_Setup", "OutOfMemory")
		}

#ifdef TEST_FRAME_DECODER
		frame->Test[i] = (FLOAT*)MemoryPool_Allocate(pool, sizeof(FLOAT) * setup->BlockSize1);
		if (frame->Test[i] == NULL) {
			THROW0(ctx, "FrameDecoder_Setup", "OutOfMemory")
		}
#endif
	}

#ifdef TEST_FRAME_DECODER
	frame->TestLength = 0;
	frame->TestType   = -1;
#endif

	frame->Transform0 = CreateTransformDecoder(pool, setup->BlockSize0);
	frame->Transform1 = CreateTransformDecoder(pool, setup->BlockSize1);
	if (frame->Transform0 == NULL || frame->Transform1 == NULL) {
		THROW0(ctx, "FrameDecoder_Setup", "OutOfMemory")
	}

	FrameDecoder_Reset(frame);

	result = TRUE;

ErrorEnd:

	return result;
}

void FrameDecoder_Reset(
	FrameDecoder_t* frame)
{
	frame->Index         = 0;
	frame->PrevBlockSize = 0;
	frame->OutputLength  = 0;
}

/* */

static BOOL DecodeMode(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame);

static BOOL ParseFloor(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame);

static BOOL DecodeResidue(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame);

/* */

static void DecodeInverseCoupling(
	FrameDecoder_t* frame);

static void DecodeFloor(
	FrameDecoder_t* frame);

static void DecodeInverseTransform(
	FrameDecoder_t* frame);

static void DecodeFinishFrame(
	FrameDecoder_t* frame);

/* */

static void DecodeTestFrame(
	FrameDecoder_t* frame,
	INT32           type)
{
#ifdef TEST_FRAME_DECODER
	if (type != frame->TestType) {
		return;
	}

	switch (frame->TestType) {
	case QVH_RES:
	case QVH_CC:
	case QVH_SE:
	{
		INT32 i;

		frame->TestLength = frame->BlockSize / 2;
		for (i = 0; i < frame->Setup->Channels; i++) {
			memcpy(
				frame->Test[i],
				frame->Sample[i],
				sizeof(FLOAT) * frame->TestLength);
		}

		break;
	}

	case QVH_IMDCT:
	{
		INT32 i;

		frame->TestLength = frame->BlockSize;
		for (i = 0; i < frame->Setup->Channels; i++) {
			const FLOAT* cur = frame->PCM[frame->Index][i];
			memcpy(
				frame->Test[i],
				cur,
				sizeof(FLOAT) * frame->TestLength);
		}

		break;
	}

	} /* switch */
#endif
}

/* */

BOOL FrameDecoder_Decode(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame)
{
	BOOL result = FALSE;

	UINT32 value;

	value = BitDecoder_GetBits(d, 1);
	if (value != 0) {
		THROW0(ctx, "FrameDecoder_Decode", "InvalidPacketType")
	}

	if (!DecodeMode(ctx, d, frame)) {
		goto ErrorEnd;
	}

	if (!ParseFloor(ctx, d, frame)) {
		goto ErrorEnd;
	}

	if (!DecodeResidue(ctx, d, frame)) {
		goto ErrorEnd;
	}

	/* */

	DecodeTestFrame(frame, QVH_RES);

	DecodeInverseCoupling(frame);

	DecodeTestFrame(frame, QVH_CC);

	DecodeFloor(frame);

	DecodeTestFrame(frame, QVH_SE);

	DecodeInverseTransform(frame);

	DecodeTestFrame(frame, QVH_IMDCT);

	DecodeFinishFrame(frame);

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

BOOL DecodeMode(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame)
{
	BOOL result = FALSE;

	DecoderSetup_t* setup = frame->Setup;

	INT32 mode = BitDecoder_GetBits(d, setup->ModeBits);
	if (mode >= setup->ModeCount) {
		THROW0(ctx, "FrameDecoder_Decode", "InvalidModeIndex")
	}

	frame->ModeSetup = setup->ModeSetup + mode;

	if (frame->ModeSetup->BlockFlag == 0) {
		frame->BlockSize = setup->BlockSize0;

	} else {
		INT32 prev = BitDecoder_GetBits(d, 1);
		INT32 next = BitDecoder_GetBits(d, 1);
		frame->BlockSize = setup->BlockSize1;
	}

	frame->MappingSetup = setup->MappingSetup + frame->ModeSetup->Mapping;

	frame->Floor1Setup  = setup->Floor1Setup  + frame->MappingSetup->Floor;
	frame->ResidueSetup = setup->ResidueSetup + frame->MappingSetup->Residue;

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL ParseFloor(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame)
{
	BOOL result = FALSE;

	INT32 i;

	for (i = 0; i < frame->Setup->Channels; i++) {
		if (!Floor1Decoder_Parse(
			ctx,
			d,
			frame->Floor1Setup,
			frame->Setup->HuffmanDecoder,
			frame->Y[i])) {
			goto ErrorEnd;
		}
	}

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL DecodeResidue(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	FrameDecoder_t*   frame)
{
	BOOL result = FALSE;

	BOOL decoded = FALSE;

	ResidueSetup_t* setup = frame->ResidueSetup;

	INT32 channels = frame->Setup->Channels;

	INT32 i;

	for (i = 0; i < channels; i++) {
		memset(frame->Sample[i], 0, sizeof(FLOAT) * frame->BlockSize / 2);
	}

	switch (channels) {
	case 1:
		if (setup->Type == 1) {
			if (!ResidueDecoder_Decode_1M(
				ctx,
				d,
				setup,
				frame->Setup->HuffmanDecoder,
				frame->Sample)) {
				goto ErrorEnd;
			}
			decoded = TRUE;
		}
		break;

	case 2:
		if (setup->Type == 2) {
			if (!ResidueDecoder_Decode_2S(
				ctx,
				d,
				setup,
				frame->Setup->HuffmanDecoder,
				frame->Sample)) {
				goto ErrorEnd;
			}
			decoded = TRUE;
		}
		break;
	}

	if (!decoded) {
		THROW0(ctx, "FrameDecoder_Decode", "InvalidResidueType")
	}

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

void DecodeInverseCoupling(
	FrameDecoder_t* frame)
{
	MappingSetup_t* mapping = frame->MappingSetup;

	switch (mapping->CouplingSteps) {
	case 0:
		break;

	case 1:
	{
		FLOAT* mag = frame->Sample[mapping->Magnitude];
		FLOAT* ang = frame->Sample[mapping->Angle    ];

		FLOAT* end = mag + frame->BlockSize / 2;
		while (mag < end) {
			if (*mag > 0) {
				if (*ang > 0) {
					*ang = *mag - *ang;
				} else {
					FLOAT t = *ang;
					*ang  = *mag;
					*mag += t;
				}
			} else {
				if (*ang > 0) {
					*ang += *mag;
				} else {
					FLOAT t = *ang;
					*ang  = *mag;
					*mag -= t;
				}
			}

			mag++;
			ang++;
		}

		break;
	}

	default:
		break;
	}
}

void DecodeFloor(
	FrameDecoder_t* frame)
{
	INT32 i;

	for (i = 0; i < frame->Setup->Channels; i++) {
		Floor1Decoder_Decode(
			frame->Floor1Setup,
			frame->Y[i],
			frame->Sample[i],
			frame->BlockSize / 2);
	}
}

void DecodeInverseTransform(
	FrameDecoder_t* frame)
{
	INT32 i;

	for (i = 0; i < frame->Setup->Channels; i++) {
		FLOAT* pcm = frame->PCM[frame->Index][i];

		if (frame->ModeSetup->BlockFlag == 0) {
			frame->Transform0->Transform(
				frame->Transform0,
				frame->Sample[i],
				pcm);
		} else {
			frame->Transform1->Transform(
				frame->Transform1,
				frame->Sample[i],
				pcm);
		}
	}
}

void DecodeFinishFrame(
	FrameDecoder_t* frame)
{
	DecoderSetup_t* setup = frame->Setup;

	if (frame->PrevBlockSize > 0) {
		INT32 i;

		WindowingDecoder_t* w = setup->Windowing0;
		if (frame->PrevBlockSize == setup->BlockSize1 && frame->ModeSetup->BlockFlag == 1) {
			w = setup->Windowing1;
		}

		for (i = 0; i < setup->Channels; i++) {
			const FLOAT* prv = frame->PCM[frame->Index ^ 1][i] + frame->PrevBlockSize / 2;
			const FLOAT* cur = frame->PCM[frame->Index    ][i];

			if (frame->BlockSize == frame->PrevBlockSize) {
				w->Windowing(
					w,
					frame->Output[i],
					prv,
					cur);

			} else {
				INT32 start = (setup->BlockSize1 - setup->BlockSize0) / 4;

				if (frame->BlockSize > frame->PrevBlockSize) { /* S -> L */
					w->Windowing(
						w,
						frame->Output[i],
						prv,
						cur + start);

					start += setup->BlockSize0 / 2;

					memcpy(
						frame->Output[i] + setup->BlockSize0 / 2,
						cur + start,
						(setup->BlockSize1 / 2 - start) * sizeof(FLOAT));

				} else { /* L -> S */
					memcpy(
						frame->Output[i],
						prv,
						start * sizeof(FLOAT));

					w->Windowing(
						w,
						frame->Output[i] + start,
						prv + start,
						cur);
				}
			}
		}

		frame->OutputLength = (frame->BlockSize + frame->PrevBlockSize) / 4;

	} else {
		frame->OutputLength = 0;
	}

	frame->Index         ^= 1;
	frame->PrevBlockSize  = frame->BlockSize;
}

/* */

