// QAXDecoder.h
// 2008/11/27

#pragma once

/* */

#ifndef NO_RETURN
#define NO_RETURN __declspec(noreturn)
#endif

#ifndef STDCALL
#define STDCALL __stdcall
#endif

/* */

#ifdef QAXDECODER_EXPORTS
#define API_EXTERN __declspec(dllexport)
#else
#define API_EXTERN
#endif

/* */

namespace QAX {

/* */

// CompPtr<T>
template<typename T>
class CompPtr {

	T* m_p;

public:

	CompPtr(T* p) : m_p(p)
	{
	}

	~CompPtr()
	{
		if (m_p != 0) {
			m_p->Release();
		}
	}

	operator T*()
	{
		return m_p;
	}

	T* operator->()
	{
		return m_p;
	}

}; // CompPtr<T>

/* */

// Exception
class Exception {

public:

	virtual ~Exception()
	{
	}

	virtual std::string ToString() = 0;

}; // Exception

// RuntimeError
class RuntimeError : public Exception {

}; // RuntimeError

// FormatError
class FormatError : public RuntimeError {

	const char* m_Message;

public:

	FormatError(const char* m) : m_Message(m)
	{
	}

	virtual ~FormatError()
	{
	}

	virtual std::string ToString()
	{
		return std::string(m_Message);
	}

	static NO_RETURN void Throw(const char* m)
	{
		throw FormatError(m);
	}

}; // FormatError

// DecoderError
class DecoderError : public RuntimeError {

	const char* m_Message;

public:

	DecoderError(const char* m) : m_Message(m)
	{
	}

	virtual ~DecoderError()
	{
	}

	virtual std::string ToString()
	{
		return std::string(m_Message);
	}

	static NO_RETURN void Throw(const char* m)
	{
		throw DecoderError(m);
	}

}; // DecoderError

// WinError
class WinError : public RuntimeError {

	const char* m_Message;

	DWORD m_Error;

public:

	WinError(const char* m, DWORD e) : m_Message(m), m_Error(e)
	{
	}

	virtual ~WinError()
	{
	}

	DWORD GetError() const
	{
		return m_Error;
	}

	virtual std::string ToString()
	{
		std::ostringstream os;

		os << m_Message;
		os << " (";
		os << m_Error;
		os << ")";

		return os.str();
	}

	static NO_RETURN void Throw(const char* m)
	{
		DWORD e = GetLastError();
		throw WinError(m, e);
	}

	static NO_RETURN void Throw(const char* m, DWORD e)
	{
		throw WinError(m, e);
	}

}; // WinError

// COMError
class COMError : public RuntimeError {

	const char* m_Message;

	HRESULT m_Error;

public:

	COMError(const char* m, HRESULT e) : m_Message(m), m_Error(e)
	{
	}

	virtual ~COMError()
	{
	}

	HRESULT GetError() const
	{
		return m_Error;
	}

	virtual std::string ToString()
	{
		std::ostringstream os;

		os << m_Message;
		os << " (";
		os << m_Error;
		os << ")";

		return os.str();
	}

	static NO_RETURN void Throw(const char* m, HRESULT e)
	{
		throw COMError(m, e);
	}

}; // COMError

/* */

// QComponent
class QComponent {

public:

	virtual void STDCALL Release() = 0;

}; // QComponent

// QFileMapping
class QFileMapping : public QComponent {

public:

	virtual const VOID* STDCALL MapView(
		UINT64 offset,
		SIZE_T size) = 0;

	virtual void STDCALL UnmapView(
		const VOID* p) = 0;

}; // QFileMapping

// QVorbisDecoder
class QVorbisDecoder;

// QDecoderFactory
class QDecoderFactory : public QComponent {

public:

	virtual QVorbisDecoder* STDCALL CreateVorbisDecoder(
		const BYTE  key[20],
		const VOID* id,
		SIZE_T      idSize,
		const VOID* setup,
		SIZE_T      setupSize) = 0;

}; // QDecoderFactory

// QDecoderOutput
class QDecoderOutput;

// QItem
struct QItem {

	std::string Type;
	std::string Name;

}; // QItem

// QDecoder
class QDecoder : public QComponent {

public:

	virtual QDecoderOutput* STDCALL CreateOutput(
		const char* name) = 0;

	virtual INT32 STDCALL GetItemCount() = 0;

	virtual QItem STDCALL GetItem(
		INT32 index) = 0;

}; // QDecoder

// QDecoderSpec
struct QDecoderSpec {

	INT32 Channels;
	INT32 SamplingRate;

	UINT64 Samples;

}; // QDecoderSpec

// QDecoderOutput
class QDecoderOutput : public QComponent {

public:

	virtual QDecoderSpec STDCALL GetDecoderSpec() = 0;

	virtual UINT32 STDCALL Decode(
		VOID*  buffer,
		UINT32 samples) = 0;

	virtual void STDCALL Seek(
		UINT64 sample) = 0;

	virtual std::string STDCALL QueryProperty(
		const char* name) = 0;

}; // QDecoderOutput

/* */

API_EXTERN bool STDCALL Initialize();

API_EXTERN QFileMapping* STDCALL QFileMappingCreate(
	LPCWSTR path);

API_EXTERN QDecoderFactory* STDCALL QDecoderFactoryCreate();

API_EXTERN QDecoder* STDCALL QDecoderCreate(
	QDecoderFactory* pFactory,
	QFileMapping*    pFileMapping);

/* */

API_EXTERN IStream* STDCALL StreamReaderCreate(
	LPCWSTR path);

API_EXTERN QDecoderOutput* STDCALL QASDecoderOutputCreate(
	QDecoderFactory* pFactory,
	IStream*         pStream);

/* */

} // namespace QAX

