//
// MainFrame.java
// This file is part of PosterDivider.
//
package jp.sourceforge.posterdivider;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileFilter;

public class MainFrame extends JFrame {
	private static final long serialVersionUID = 1L;

    // Components
    private JFileChooser fileChooserSrcFile;
    private JFileChooser fileChooserOutputFile;
	private JLabel labelSrcFile;
	private JTextField textFieldSrcFileName;
	private JLabel labelSrcFileType;
	private JButton buttonSrcFileChange;
	private JLabel labelSrcPageNum;
	private JComboBox comboBoxSrcPageNum;
	private JLabel labelSrcImageSize;
	private NumberField numberFieldSrcImageWidth;
	private NumberField numberFieldSrcImageHeight;
	private JComboBox comboBoxSrcImageSizeUnit;
	private JLabel labelSrcRect;
	private JCheckBox checkBoxSrcRectWholeOfPage;
	private NumberField numberFieldSrcLeft;
	private NumberField numberFieldSrcBottom;
	private NumberField numberFieldSrcRight;
	private NumberField numberFieldSrcTop;
	private JComboBox comboBoxSrcRectUnit;
	private JLabel labelPosterSize;
	private JComboBox comboBoxPosterSize;
	private NumberField numberFieldPosterWidth;
	private NumberField numberFieldPosterHeight;
	private JComboBox comboBoxPosterSizeUnit;
	private JCheckBox checkBoxPosterSizeKeepAspectRatio;
	private JLabel labelPaperSize;
	private JComboBox comboBoxPaperSize;
	private NumberField numberFieldPaperWidth;
	private NumberField numberFieldPaperHeight;
	private JComboBox comboBoxPaperSizeUnit;
	private JLabel labelPaperMargin;
	private NumberField numberFieldPaperMargin;
	private JComboBox comboBoxPaperMarginUnit;
	private JCheckBox checkBoxPaperRotate;
	private JButton buttonPaperOptimize;
	private JLabel labelPreview;
	private JLabel labelOutputPaperCount;
	private NumberField numberFieldOutputPaperCount;
	private CanvasPreview canvasPreview;
	private JLabel labelOutputFileName;
	private JTextField textFieldOutputFileName;
    //private JLabel labelMessage;
	private JButton buttonOutput;

	// Variables
	private File srcFile;
	private FileContainer fc;
	private ImageContainer ic;
	private float srcImageWidth;
	private float srcImageHeight;
	private ArrayList<Double> listSrcImageSizeUnit = new ArrayList<Double>();
	private float srcLeft;
	private float srcBottom;
	private float srcRight;
	private float srcTop;
	private ArrayList<Double> listSrcRectUnit = new ArrayList<Double>();
	private ArrayList<PaperSize> listPosterSize = new ArrayList<PaperSize>();
	// preferredPosterSize と keepAspectRatio から posterSize が決まる
	private float preferredPosterWidth;
	private float preferredPosterHeight;
	private float posterWidth;
	private float posterHeight;
	private ArrayList<Double> listPosterSizeUnit = new ArrayList<Double>();
	private ArrayList<PaperSize> listPaperSize = new ArrayList<PaperSize>();
	private float paperWidth;
	private float paperHeight;
	private boolean paperRotated;
	private ArrayList<Double> listPaperSizeUnit = new ArrayList<Double>();
	private float paperMargin;
	private ArrayList<Double> listPaperMarginUnit = new ArrayList<Double>();

    private String generateFileName(String str) {
	str = str.replace("{pwd}", System.getProperty("user.dir"));
	if (this.srcFile != null) {
	    str = str.replace("{input}", this.srcFile.getName());
	}
	str = str.replace("{date}", new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()));
	return str;
    }
    
    public File getSrcFile() {
	return this.srcFile;
    }
    public boolean checkFileLoaded() {
	return this.ic != null;
    }
    
    private void simulateOutput() {
	// 設定値が変更されたときに canvasPreview 等を更新する
	
	//labelMessage.setText("");
	
	PdfGenerator.CreatePosterArgs args = new PdfGenerator.CreatePosterArgs();
	args.setSrcRect(srcLeft, srcBottom, srcRight, srcTop);
	args.setPosterSize(posterWidth, posterHeight);
	args.setPaperSize(paperWidth, paperHeight);
	args.setMargin(paperMargin, paperMargin, paperMargin, paperMargin);
	
	int outputPaperCount = PdfGenerator.simulateOutputPaperCount(args);
	numberFieldOutputPaperCount.setValue(outputPaperCount);
	
	canvasPreview.setArgs(args);
	canvasPreview.repaint();
    }
    
    private void optimizePaperRotated() {
	// 出力枚数が少なくなるように印刷用紙の向きを更新する
	
	PdfGenerator.CreatePosterArgs args1 = new PdfGenerator.CreatePosterArgs();
	args1.setSrcRect(srcLeft, srcBottom, srcRight, srcTop);
	args1.setPosterSize(posterWidth, posterHeight);
	args1.setPaperSize(paperWidth, paperHeight);
	args1.setMargin(paperMargin, paperMargin, paperMargin, paperMargin);
	int outputPaperCount1 = PdfGenerator.simulateOutputPaperCount(args1);
	if (outputPaperCount1 < 0) {
	    // 不正なとき
	    outputPaperCount1 = Integer.MAX_VALUE;
	}
	
	PdfGenerator.CreatePosterArgs args2 = new PdfGenerator.CreatePosterArgs();
	args2.setSrcRect(srcLeft, srcBottom, srcRight, srcTop);
	args2.setPosterSize(posterWidth, posterHeight);
	args2.setPaperSize(paperHeight, paperWidth); // 印刷用紙を回転する
	args2.setMargin(paperMargin, paperMargin, paperMargin, paperMargin);
	int outputPaperCount2 = PdfGenerator.simulateOutputPaperCount(args2);
	if (outputPaperCount2 < 0) {
	    outputPaperCount2 = Integer.MAX_VALUE;
	}
	
	if ((outputPaperCount1 > outputPaperCount2)
	    || (outputPaperCount1 == outputPaperCount2 && paperRotated)) {
	    // 最適化
	    checkBoxPaperRotate.setSelected(!paperRotated);
	    checkBoxPaperRotate_itemStateChanged();
	}
    }
    
    public void outputPdf(File file) {
	if (!checkFileLoaded()) {
	    return;
	}
	
	PdfGenerator.CreatePosterArgs args = new PdfGenerator.CreatePosterArgs();
	args.setSrcRect(srcLeft, srcBottom, srcRight, srcTop);
	args.setPosterSize(posterWidth, posterHeight);
	args.setPaperSize(paperWidth, paperHeight);
	args.setMargin(paperMargin, paperMargin, paperMargin, paperMargin);
	try {
	    ByteArrayOutputStream baos = new ByteArrayOutputStream();
	    PdfGenerator.createPoster(ic, args, baos);
	    try {
		FileOutputStream fos = new FileOutputStream(file);
		baos.writeTo(fos);
		fos.close();
		
		//labelMessage.setText(String.format(Lib.getMessage("Text.Saved"), outputFileName));
	    } catch (IOException ex) {
		showErrorMessage(String.format(Lib.getMessage("Text.FailedWrite"), Lib.fileGetCanonicalPath(file)));
		//labelMessage.setText(String.format(Lib.getMessage("Text.FailedWrite"), outputFileName));
	    }
	} catch (Exception ex) {
	    showErrorMessage(Lib.getMessage("Text.FailedGenerate"));
	    //labelMessage.setText(Lib.getMessage("Text.FailedGenerate"));
	}
	//this.validateTree();
    }

    private void changePageSuccess(ImageContainer ic) {
	this.ic = ic;
	
	comboBoxSrcPageNum.setSelectedIndex(ic.getPageNum() - 1);
	
	srcImageWidth = ic.getImageWidth();
	srcImageHeight = ic.getImageHeight();
	numberFieldSrcImageSize_setValue();
	
	comboBoxSrcImageSizeUnit.setEnabled(true);
	checkBoxSrcRectWholeOfPage.setEnabled(true);
	comboBoxSrcRectUnit.setEnabled(true);
	buttonPaperOptimize.setEnabled(true);
	buttonOutput.setEnabled(true);
	
	canvasPreview.setImageContainer(ic);
	simulateOutput();
    }
    
    public void changePage(int pageNum) {
	if (fc == null) {
	    return;
	}
	try {
	    ImageContainer ic = new ImageContainer(fc, pageNum);
	    changePageSuccess(ic);
	} catch (Exception ex) {
	    
	}
	//this.validateTree();
    }
    
    private void changeFileSuccess(File file, FileContainer fc) {
	this.srcFile = file;
	this.fc = fc;
	
	textFieldSrcFileName.setText(Lib.fileGetCanonicalPath(file));
	
	labelSrcFileType.setText(Lib.getFileTypeName(fc.getFileType()));
	
	comboBoxSrcPageNum_initialize(fc.getPageCount());
	comboBoxSrcPageNum.setSelectedIndex(0);
	comboBoxSrcPageNum.setEnabled(fc.getFileType() == Lib.FT_PDF);
	
	comboBoxSrcImageSizeUnit_initialize(fc.getFileType());
	comboBoxSrcImageSizeUnit.setSelectedIndex(0);
	
	comboBoxSrcRectUnit_initialize(fc.getFileType());
	comboBoxSrcRectUnit.setSelectedIndex(0);
	
	changePage(comboBoxSrcPageNum.getSelectedIndex() + 1);
	
	checkBoxSrcRectWholeOfPage.setSelected(true);
	checkBoxSrcRectWholeOfPage_itemStateChanged();
    }
    
    public void changeFile(File file) {
	try {
	    FileContainer fc = new FileContainer(file);
	    changeFileSuccess(file, fc);
	} catch (Exception ex) {
	    showErrorMessage(String.format(Lib.getMessage("Text.FailedOpen"), Lib.fileGetCanonicalPath(file)));
	    //labelMessage.setText(String.format(Lib.getMessage("Text.FailedOpen"), Lib.fileGetCanonicalPath(file)));
	}
	//this.validateTree();
    }
    
    private void updatePosterSize() {
	// preferredPosterSize から posterSize を更新する
	// KeepAspectRatio なら srcRect を拡大して preferredPosterSize に入る最大の Size を
	// posterSize とする
	posterWidth = preferredPosterWidth;
	posterHeight = preferredPosterHeight;
	if (checkBoxPosterSizeKeepAspectRatio.isSelected()) {
	    if ((srcRight - srcLeft) * preferredPosterHeight > (srcTop - srcBottom) * preferredPosterWidth) {
		// Src が横長
		if (srcRight != srcLeft) {
		    posterHeight = posterWidth * (srcTop - srcBottom) / (srcRight - srcLeft);
		}
	    } else {
		if (srcTop != srcBottom) {
		    posterWidth = posterHeight * (srcRight - srcLeft) / (srcTop - srcBottom);
		}
	    }
	}
    }

	/*
	 * private void revisePosterSize(int direction) { // direction は固定する方向
	 * 0->Width, 1->Height boolean changed = false; if
	 * (checkboxPosterSizeKeepAspectRatio.isSelected()) { switch (direction) {
	 * case 0: if (srcRight != srcLeft) { posterHeight = posterWidth * (srcTop -
	 * srcBottom) / (srcRight - srcLeft); changed = true; } break; case 1: if
	 * (srcTop != srcBottom) { posterWidth = posterHeight * (srcRight - srcLeft) /
	 * (srcTop - srcBottom); changed = true; } break; default: break; } } if
	 * (changed) { numberFieldPosterSize_setValue(); simulateOutput(); } }
	 */

    private void showErrorMessage(String msg) {
	JOptionPane.showMessageDialog(this, msg, Lib.getMessage("App.LowerTitle"), JOptionPane.ERROR_MESSAGE);
    }

    private File fileChooserSrcFile_show() {
	if (fileChooserSrcFile.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
	    return fileChooserSrcFile.getSelectedFile();
	} else {
	    return null;
	}
    }

    private void menuItemNewWindow_actionPerformed() {
	Program.createFrame();
    }

    private void menuItemOpenFile_actionPerformed() {
	File file = fileChooserSrcFile_show();
	if (file != null) {
	    switch(Lib.getOS()) {
	    case Lib.OS_MAC:
		MainFrame frame = Program.getSameFileFrame(file);
		if(frame != null) {
		    frame.setVisible(true);
		} else if (!checkFileLoaded()) {
		    changeFile(file);
		} else {
		    Program.createFrame(file, false);
		}
		break;
	    default:
		changeFile(file);
		break;
	    }
	}
    }

	private void menuItemCloseWindow_actionPerformed() {
	        this.dispose();
	}

    private void menuItemOutput_actionPerformed() {
	if (!checkFileLoaded()) {
	    //labelMessage.setText(Lib.getMessage("Text.NoInputFiles"));
	    //this.validateTree();
	    return;
	}
	
	String outputFileName = generateFileName(textFieldOutputFileName.getText());
	fileChooserOutputFile.setSelectedFile(new File(outputFileName));
	if (fileChooserOutputFile.showSaveDialog(this) != JFileChooser.APPROVE_OPTION) {
	    return;
	}
	outputPdf(fileChooserOutputFile.getSelectedFile());
    }

	private void menuItemExit_actionPerformed() {
		System.exit(0);
	}

    private void buttonSrcFileChange_actionPerformed() {
	File file = fileChooserSrcFile_show();
	if (file != null) {
	    switch(Lib.getOS()) {
	    case Lib.OS_MAC:
		MainFrame frame = Program.getSameFileFrame(file);
		if(frame != null) {
		    frame.setVisible(true);
		} else {
		    changeFile(file);
		}
		break;
	    default:
		changeFile(file);
		break;
	    }
	}
    }

    private boolean updatingComboBoxSrcPageNum = false;
    private void comboBoxSrcPageNum_initialize(int pageCount) {
	updatingComboBoxSrcPageNum = true;
	comboBoxSrcPageNum.removeAllItems();
	for (int i = 1; i <= pageCount; i++) {
	    comboBoxSrcPageNum.addItem(Integer.toString(i));
	}
	updatingComboBoxSrcPageNum = false;
    }

    private void comboBoxSrcPageNum_itemStateChanged() {
	if(updatingComboBoxSrcPageNum) return;

	int pageNum = comboBoxSrcPageNum.getSelectedIndex() + 1;
	if (this.ic.getPageNum() != pageNum) {
	    changePage(pageNum);
	}
    }

	private void numberFieldSrcImageSize_setValue() {
		double rate = listSrcImageSizeUnit.get(comboBoxSrcImageSizeUnit
				.getSelectedIndex());
		numberFieldSrcImageWidth.setValue(srcImageWidth * rate);
		numberFieldSrcImageHeight.setValue(srcImageHeight * rate);
	}

	private void comboBoxSrcImageSizeUnit_addItem(String key, double value) {
		comboBoxSrcImageSizeUnit.addItem(key);
		listSrcImageSizeUnit.add(value);
	}

    private boolean updatingComboBoxImageSizeUnit = false;
	private void comboBoxSrcImageSizeUnit_initialize(int fileType) {
	    updatingComboBoxImageSizeUnit = true;
		comboBoxSrcImageSizeUnit.removeAllItems();
		listSrcImageSizeUnit.clear();
		switch (fileType) {
		case Lib.FT_PDF:
			comboBoxSrcImageSizeUnit_addItem(Lib.getMessage("Str.MM"), Lib.UNIT_POINT_MM);
			comboBoxSrcImageSizeUnit_addItem(Lib.getMessage("Str.CM"), Lib.UNIT_POINT_CM);
			comboBoxSrcImageSizeUnit_addItem(Lib.getMessage("Str.M"), Lib.UNIT_POINT_M);
			comboBoxSrcImageSizeUnit_addItem(Lib.getMessage("Str.Inch"), Lib.UNIT_POINT_INCH);
			comboBoxSrcImageSizeUnit_addItem(Lib.getMessage("Str.Point"), Lib.UNIT_POINT_POINT);
			break;
		case Lib.FT_BMP:
		case Lib.FT_GIF:
		case Lib.FT_JBIG2:
		case Lib.FT_JPEG:
		case Lib.FT_JPEG2000:
		case Lib.FT_PNG:
		case Lib.FT_TIFF:
		case Lib.FT_WMF:
		case Lib.FT_OTHERIMAGE:
			comboBoxSrcImageSizeUnit_addItem(Lib.getMessage("Str.Pixel"), Lib.UNIT_PIXEL_PIXEL);
			break;
		default:
			break;
		}
		updatingComboBoxImageSizeUnit = false;
	}

	private void comboBoxSrcImageSizeUnit_itemStateChanged() {
	    if(updatingComboBoxImageSizeUnit) return;

		numberFieldSrcImageSize_setValue();
	}

	private void checkBoxSrcRectWholeOfPage_itemStateChanged() {
		boolean b = checkBoxSrcRectWholeOfPage.isSelected();
		numberFieldSrcLeft.setEnabled(!b);
		numberFieldSrcTop.setEnabled(!b);
		numberFieldSrcRight.setEnabled(!b);
		numberFieldSrcBottom.setEnabled(!b);
		if (b) {
			srcLeft = 0;
			srcBottom = 0;
			srcRight = srcImageWidth;
			srcTop = srcImageHeight;
			numberFieldSrcRect_setValue();
			updatePosterSize();
			numberFieldPosterSize_setValue();
			// revisePosterSize(0);
			simulateOutput();
		}
	}

	private void numberFieldSrcRect_actionPerformed(Object obj) {
		double rate = listSrcRectUnit.get(comboBoxSrcRectUnit.getSelectedIndex());
		srcLeft = (float) (numberFieldSrcLeft.getValue() / rate);
		srcTop = (float) (numberFieldSrcTop.getValue() / rate);
		srcRight = (float) (numberFieldSrcRight.getValue() / rate);
		srcBottom = (float) (numberFieldSrcBottom.getValue() / rate);
		numberFieldSrcRect_setValue();
		updatePosterSize();
		numberFieldPosterSize_setValue();
		// revisePosterSize(0);
		simulateOutput();
	}

	private void numberFieldSrcRect_setValue() {
		if (listSrcRectUnit.isEmpty()) {
			return;
		}
		double rate = listSrcRectUnit.get(comboBoxSrcRectUnit.getSelectedIndex());
		numberFieldSrcLeft.setValue(srcLeft * rate);
		numberFieldSrcTop.setValue(srcTop * rate);
		numberFieldSrcRight.setValue(srcRight * rate);
		numberFieldSrcBottom.setValue(srcBottom * rate);
	}

	private void comboBoxSrcRectUnit_addItem(String key, double value) {
		comboBoxSrcRectUnit.addItem(key);
		listSrcRectUnit.add(value);
	}

    private boolean updatingComboBoxSrcRectUnit = false;
	private void comboBoxSrcRectUnit_initialize(int fileType) {
	    updatingComboBoxSrcRectUnit = true;
		comboBoxSrcRectUnit.removeAllItems();
		listSrcRectUnit.clear();
		switch (fileType) {
		case Lib.FT_PDF:
			comboBoxSrcRectUnit_addItem(Lib.getMessage("Str.MM"), Lib.UNIT_POINT_MM);
			comboBoxSrcRectUnit_addItem(Lib.getMessage("Str.CM"), Lib.UNIT_POINT_CM);
			comboBoxSrcRectUnit_addItem(Lib.getMessage("Str.M"), Lib.UNIT_POINT_M);
			comboBoxSrcRectUnit_addItem(Lib.getMessage("Str.Inch"), Lib.UNIT_POINT_INCH);
			comboBoxSrcRectUnit_addItem(Lib.getMessage("Str.Point"), Lib.UNIT_POINT_POINT);
			break;
		case Lib.FT_BMP:
		case Lib.FT_GIF:
		case Lib.FT_JBIG2:
		case Lib.FT_JPEG:
		case Lib.FT_JPEG2000:
		case Lib.FT_PNG:
		case Lib.FT_TIFF:
		case Lib.FT_WMF:
		case Lib.FT_OTHERIMAGE:
			comboBoxSrcRectUnit_addItem(Lib.getMessage("Str.Pixel"), Lib.UNIT_PIXEL_PIXEL);
			break;
		default:
			break;
		}
		updatingComboBoxSrcRectUnit = false;
	}

	private void comboBoxSrcRectUnit_itemStateChanged() {
	    if(updatingComboBoxSrcRectUnit) return;

		numberFieldSrcRect_setValue();
	}

	private void comboBoxPosterSize_addItem(String key, PaperSize value) {
		comboBoxPosterSize.addItem(key);
		listPosterSize.add(value);
	}

	private void comboBoxPosterSize_initialize() {
		comboBoxPosterSize.removeAllItems();
		listPosterSize.clear();
		// TODO:後で減らす
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A0"), PaperSize.A0);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A1"), PaperSize.A1);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A2"), PaperSize.A2);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A3"), PaperSize.A3);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A4"), PaperSize.A4);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A5"), PaperSize.A5);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.A6"), PaperSize.A6);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B0"), PaperSize.B0);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B1"), PaperSize.B1);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B2"), PaperSize.B2);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B3"), PaperSize.B3);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B4"), PaperSize.B4);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B5"), PaperSize.B5);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.B6"), PaperSize.B6);
		comboBoxPosterSize_addItem(Lib.getMessage("Str.Custom"), PaperSize.CUSTOM);
	}

	private void comboBoxPosterSize_itemStateChanged() {
		PaperSize size = listPosterSize.get(comboBoxPosterSize.getSelectedIndex());
		boolean custom = size.isCustom();
		numberFieldPosterWidth.setEnabled(custom);
		numberFieldPosterHeight.setEnabled(custom);
		// checkBoxPosterSizeKeepAspectRatio.setEnabled(custom);
		// if (!custom) {
		// checkBoxPosterSizeKeepAspectRatio.setSelected(false);
		// posterWidth = size.getWidth();
		// posterHeight = size.getHeight();
		// numberFieldPosterSize_setValue();
		// simulateOutput();
		// }
		if (!custom) {
			preferredPosterWidth = size.getWidth();
			preferredPosterHeight = size.getHeight();
			updatePosterSize();
			numberFieldPosterSize_setValue();
			simulateOutput();
		}
	}

	private void numberFieldPosterSize_actionPerformed(Object obj) {
		double rate = listPosterSizeUnit.get(comboBoxPosterSizeUnit
				.getSelectedIndex());
		// posterWidth = (float) (numberFieldPosterWidth.getValue() / rate);
		// posterHeight = (float) (numberFieldPosterHeight.getValue() / rate);
		// if (obj == numberFieldPosterWidth) {
		// revisePosterSize(0);
		// }
		// if (obj == numberFieldPosterHeight) {
		// revisePosterSize(1);
		// }
		preferredPosterWidth = (float) (numberFieldPosterWidth.getValue() / rate);
		preferredPosterHeight = (float) (numberFieldPosterHeight.getValue() / rate);
		if (checkBoxPosterSizeKeepAspectRatio.isSelected()) {
			if (obj == numberFieldPosterWidth) {
				if (srcRight != srcLeft) {
					preferredPosterHeight = preferredPosterWidth
							* (srcTop - srcBottom) / (srcRight - srcLeft);
				}
			}
			if (obj == numberFieldPosterHeight) {
				if (srcTop != srcBottom) {
					preferredPosterWidth = preferredPosterHeight
							* (srcRight - srcLeft) / (srcTop - srcBottom);
				}
			}
		}
		updatePosterSize();
		numberFieldPosterSize_setValue();
		simulateOutput();
	}

	private void numberFieldPosterSize_setValue() {
		double rate = listPosterSizeUnit.get(comboBoxPosterSizeUnit
				.getSelectedIndex());
		numberFieldPosterWidth.setValue(posterWidth * rate);
		numberFieldPosterHeight.setValue(posterHeight * rate);
	}

	private void comboBoxPosterSizeUnit_addItem(String key, double value) {
		comboBoxPosterSizeUnit.addItem(key);
		listPosterSizeUnit.add(value);
	}

	private void comboBoxPosterSizeUnit_initialize() {
		comboBoxPosterSizeUnit.removeAllItems();
		listPosterSizeUnit.clear();
		comboBoxPosterSizeUnit_addItem(Lib.getMessage("Str.MM"), Lib.UNIT_POINT_MM);
		comboBoxPosterSizeUnit_addItem(Lib.getMessage("Str.CM"), Lib.UNIT_POINT_CM);
		comboBoxPosterSizeUnit_addItem(Lib.getMessage("Str.M"), Lib.UNIT_POINT_M);
		comboBoxPosterSizeUnit_addItem(Lib.getMessage("Str.Inch"), Lib.UNIT_POINT_INCH);
		comboBoxPosterSizeUnit_addItem(Lib.getMessage("Str.Point"), Lib.UNIT_POINT_POINT);
	}

	private void comboBoxPosterSizeUnit_itemStateChanged() {
		numberFieldPosterSize_setValue();
	}

	private void checkBoxPosterSizeKeepAspectRatio_itemStateChanged() {
		// revisePosterSize(0);
		updatePosterSize();
		numberFieldPosterSize_setValue();
		simulateOutput();
	}

	private void comboBoxPaperSize_addItem(String key, PaperSize value) {
		comboBoxPaperSize.addItem(key);
		listPaperSize.add(value);
	}

	private void comboBoxPaperSize_initialize() {
		comboBoxPaperSize.removeAllItems();
		listPaperSize.clear();
		comboBoxPaperSize_addItem(Lib.getMessage("Str.A4"), PaperSize.A4);
		comboBoxPaperSize_addItem(Lib.getMessage("Str.A3"), PaperSize.A3);
		comboBoxPaperSize_addItem(Lib.getMessage("Str.B5"), PaperSize.B5);
		comboBoxPaperSize_addItem(Lib.getMessage("Str.B4"), PaperSize.B4);
		comboBoxPaperSize_addItem(Lib.getMessage("Str.Custom"), PaperSize.CUSTOM);
	}

	private void comboBoxPaperSize_itemStateChanged() {
		PaperSize size = listPaperSize.get(comboBoxPaperSize.getSelectedIndex());
		boolean custom = size.isCustom();
		numberFieldPaperWidth.setEnabled(custom);
		numberFieldPaperHeight.setEnabled(custom);
		if (!custom) {
			if (paperRotated) {
				paperWidth = size.getHeight();
				paperHeight = size.getWidth();
			} else {
				paperWidth = size.getWidth();
				paperHeight = size.getHeight();
			}
			numberFieldPaperSize_setValue();
			simulateOutput();
		}
	}

	private void numberFieldPaperSize_actionPerformed(Object obj) {
		double rate = listPaperSizeUnit.get(comboBoxPaperSizeUnit
				.getSelectedIndex());
		paperWidth = (float) (numberFieldPaperWidth.getValue() / rate);
		paperHeight = (float) (numberFieldPaperHeight.getValue() / rate);
		numberFieldPaperSize_setValue();
		simulateOutput();
	}

	private void numberFieldPaperSize_setValue() {
		double rate = listPaperSizeUnit.get(comboBoxPaperSizeUnit
				.getSelectedIndex());
		numberFieldPaperWidth.setValue(paperWidth * rate);
		numberFieldPaperHeight.setValue(paperHeight * rate);
	}

	private void comboBoxPaperSizeUnit_addItem(String key, double value) {
		comboBoxPaperSizeUnit.addItem(key);
		listPaperSizeUnit.add(value);
	}

	private void comboBoxPaperSizeUnit_initialize() {
		comboBoxPaperSizeUnit.removeAllItems();
		listPaperSizeUnit.clear();
		comboBoxPaperSizeUnit_addItem(Lib.getMessage("Str.MM"), Lib.UNIT_POINT_MM);
		comboBoxPaperSizeUnit_addItem(Lib.getMessage("Str.CM"), Lib.UNIT_POINT_CM);
		comboBoxPaperSizeUnit_addItem(Lib.getMessage("Str.M"), Lib.UNIT_POINT_M);
		comboBoxPaperSizeUnit_addItem(Lib.getMessage("Str.Inch"), Lib.UNIT_POINT_INCH);
		comboBoxPaperSizeUnit_addItem(Lib.getMessage("Str.Point"), Lib.UNIT_POINT_POINT);
	}

	private void comboBoxPaperSizeUnit_itemStateChanged() {
		numberFieldPaperSize_setValue();
	}

	private void numberFieldPaperMargin_actionPerformed(Object obj) {
		double rate = listPaperMarginUnit.get(comboBoxPaperMarginUnit
				.getSelectedIndex());
		paperMargin = (float) (numberFieldPaperMargin.getValue() / rate);
		numberFieldPaperMargin_setValue();
		simulateOutput();
	}

	private void numberFieldPaperMargin_setValue() {
		double rate = listPaperMarginUnit.get(comboBoxPaperMarginUnit
				.getSelectedIndex());
		numberFieldPaperMargin.setValue(paperMargin * rate);
	}

	private void comboBoxPaperMarginUnit_addItem(String key, double value) {
		comboBoxPaperMarginUnit.addItem(key);
		listPaperMarginUnit.add(value);
	}

	private void comboBoxPaperMarginUnit_initialize() {
		comboBoxPaperMarginUnit.removeAllItems();
		listPaperMarginUnit.clear();
		comboBoxPaperMarginUnit_addItem(Lib.getMessage("Str.MM"), Lib.UNIT_POINT_MM);
		comboBoxPaperMarginUnit_addItem(Lib.getMessage("Str.CM"), Lib.UNIT_POINT_CM);
		comboBoxPaperMarginUnit_addItem(Lib.getMessage("Str.M"), Lib.UNIT_POINT_M);
		comboBoxPaperMarginUnit_addItem(Lib.getMessage("Str.Inch"), Lib.UNIT_POINT_INCH);
		comboBoxPaperMarginUnit_addItem(Lib.getMessage("Str.Point"), Lib.UNIT_POINT_POINT);
	}

	private void comboBoxPaperMarginUnit_itemStateChanged() {
		numberFieldPaperMargin_setValue();
	}

	private void checkBoxPaperRotate_itemStateChanged() {
		boolean paperRotated = checkBoxPaperRotate.isSelected();
		if (this.paperRotated != paperRotated) {
			float a = this.paperWidth;
			this.paperWidth = this.paperHeight;
			this.paperHeight = a;
			this.paperRotated = paperRotated;
			numberFieldPaperSize_setValue();
			simulateOutput();
		}
	}

	private void buttonPaperOptimize_actionPerformed() {
		optimizePaperRotated();
	}

    private void buttonOutput_actionPerformed() {
	if (!checkFileLoaded()) {
	    //labelMessage.setText(Lib.getMessage("Text.NoInputFiles"));
	    //this.validateTree();
	    return;
	}
	
	String outputFileName = generateFileName(textFieldOutputFileName.getText());
	fileChooserOutputFile.setSelectedFile(new File(outputFileName));
	if (fileChooserOutputFile.showSaveDialog(this) != JFileChooser.APPROVE_OPTION) {
	    return;
	}
	outputPdf(fileChooserOutputFile.getSelectedFile());
    }

	private void numberField_initialize(NumberField numberField) {
		numberField.setColumns(7);
		numberField.setDisplayDigit(2);
		numberField.setUnitIncrement(0.1);
	}

	private void mainFrame_initializeComponents() {
	    this.setTitle(Lib.getMessage("App.LowerTitle"));
		this.setSize(800, 600);
		this.setLocationByPlatform(true);
		//this.setLocationRelativeTo(null);
		this.addWindowListener(new WindowListener() {
			public void windowActivated(WindowEvent e) {
			}

			public void windowClosed(WindowEvent e) {
			}

			public void windowClosing(WindowEvent e) {
				dispose();
			}

			public void windowDeactivated(WindowEvent e) {
			}

			public void windowDeiconified(WindowEvent e) {
			}

			public void windowIconified(WindowEvent e) {
			}

			public void windowOpened(WindowEvent e) {
			}
		});

		FileFilter fileFilterPdfAndImage = new FileFilter() {
			public boolean accept(File f) {
			    if(f.isDirectory()) {
				return true;
			    }
			    if(Lib.judgeFileTypeFromExtention(f) != Lib.FT_NONE) {
				return true;
			    } else {
				return false;
			    }
			}
			public String getDescription() {
			    return Lib.getMessage("fileFilterPdfAndImage.Description");
			}
		    };
		fileChooserSrcFile = new JFileChooser();
		fileChooserSrcFile.addChoosableFileFilter(fileFilterPdfAndImage);
		fileChooserSrcFile.setFileFilter(fileFilterPdfAndImage);

		fileChooserOutputFile = new JFileChooser();

		String menuOSName;
		int menuMask;
		switch(Lib.getOS()) {
		case Lib.OS_WIN:
		    menuOSName = ".Win";
		    menuMask = InputEvent.CTRL_DOWN_MASK;
		    break;
		case Lib.OS_MAC:
		    menuOSName = ".Mac";
		    menuMask = InputEvent.META_DOWN_MASK;
		    break;
		default:
		    menuOSName = "";
		    menuMask = InputEvent.CTRL_DOWN_MASK;
		    break;
		}
		JMenuBar menuBar = new JMenuBar();
		JMenu menuFile = new JMenu();
		menuFile.setText(Lib.getMessage("menuFile.Text" + menuOSName));
		JMenuItem menuItemNewWindow = new JMenuItem();
		menuItemNewWindow.setText(Lib.getMessage("menuItemNewWindow.Text" + menuOSName));
		menuItemNewWindow.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, menuMask));
		menuItemNewWindow.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				menuItemNewWindow_actionPerformed();
			}
		});
		JMenuItem menuItemOpenFile = new JMenuItem();
		menuItemOpenFile.setText(Lib.getMessage("menuItemOpenFile.Text" + menuOSName) + "...");
		menuItemOpenFile.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, menuMask));
		menuItemOpenFile.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				menuItemOpenFile_actionPerformed();
			}
		});
		JMenuItem menuItemCloseWindow = new JMenuItem();
		menuItemCloseWindow.setText(Lib.getMessage("menuItemCloseWindow.Text" + menuOSName));
		menuItemCloseWindow.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W, menuMask));
		menuItemCloseWindow.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				menuItemCloseWindow_actionPerformed();
			}
		});
		JMenuItem menuItemOutput = new JMenuItem();
		menuItemOutput.setText(Lib.getMessage("menuItemOutput.Text" + menuOSName) + "...");
		menuItemOutput.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				menuItemOutput_actionPerformed();
			}
		});
		JMenuItem menuItemExit = new JMenuItem();
		menuItemExit.setText(Lib.getMessage("menuItemExit.Text" + menuOSName));
		menuItemExit.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				menuItemExit_actionPerformed();
			}
		});

		labelSrcFile = new JLabel();
		labelSrcFile.setText(Lib.getMessage("labelSrcFile.Text"));

		textFieldSrcFileName = new JTextField();
		textFieldSrcFileName.setEditable(false);
		textFieldSrcFileName.setColumns(50);

		labelSrcFileType = new JLabel();

		buttonSrcFileChange = new JButton();
		buttonSrcFileChange.setText(Lib.getMessage("buttonSrcFileChange.Text") + "...");
		buttonSrcFileChange.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				buttonSrcFileChange_actionPerformed();
			}
		});

		labelSrcPageNum = new JLabel();
		labelSrcPageNum.setText(Lib.getMessage("labelSrcPageNum.Text"));

		comboBoxSrcPageNum = new JComboBox();
		comboBoxSrcPageNum.setEnabled(false);
		comboBoxSrcPageNum.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
			    if(!updatingComboBoxSrcPageNum) {
				comboBoxSrcPageNum_itemStateChanged();
			    }
			}
		});

		labelSrcImageSize = new JLabel();
		labelSrcImageSize.setText(Lib.getMessage("labelSrcImageSize.Text"));

		numberFieldSrcImageWidth = new NumberField();
		numberField_initialize(numberFieldSrcImageWidth);
		numberFieldSrcImageWidth.setEnabled(false);

		numberFieldSrcImageHeight = new NumberField();
		numberField_initialize(numberFieldSrcImageHeight);
		numberFieldSrcImageHeight.setEnabled(false);

		comboBoxSrcImageSizeUnit = new JComboBox();
		comboBoxSrcImageSizeUnit.setEnabled(false);
		comboBoxSrcImageSizeUnit.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxSrcImageSizeUnit_itemStateChanged();
			}
		});

		labelSrcRect = new JLabel();
		labelSrcRect.setText(Lib.getMessage("labelSrcRect.Text"));

		checkBoxSrcRectWholeOfPage = new JCheckBox();
		checkBoxSrcRectWholeOfPage.setText(Lib.getMessage("checkBoxSrcRectWholeOfPage.Text"));
		checkBoxSrcRectWholeOfPage.setEnabled(false);
		checkBoxSrcRectWholeOfPage.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				checkBoxSrcRectWholeOfPage_itemStateChanged();
			}
		});

		numberFieldSrcLeft = new NumberField();
		numberField_initialize(numberFieldSrcLeft);
		numberFieldSrcLeft.setEnabled(false);
		numberFieldSrcLeft.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldSrcRect_actionPerformed(e.getSource());
			}
		});

		numberFieldSrcBottom = new NumberField();
		numberField_initialize(numberFieldSrcBottom);
		numberFieldSrcBottom.setEnabled(false);
		numberFieldSrcBottom.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldSrcRect_actionPerformed(e.getSource());
			}
		});

		numberFieldSrcRight = new NumberField();
		numberField_initialize(numberFieldSrcRight);
		numberFieldSrcRight.setEnabled(false);
		numberFieldSrcRight.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldSrcRect_actionPerformed(e.getSource());
			}
		});

		numberFieldSrcTop = new NumberField();
		numberField_initialize(numberFieldSrcTop);
		numberFieldSrcTop.setEnabled(false);
		numberFieldSrcTop.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldSrcRect_actionPerformed(e.getSource());
			}
		});

		comboBoxSrcRectUnit = new JComboBox();
		comboBoxSrcRectUnit.setEnabled(false);
		comboBoxSrcRectUnit.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxSrcRectUnit_itemStateChanged();
			}
		});

		labelPosterSize = new JLabel();
		labelPosterSize.setText(Lib.getMessage("labelPosterSize.Text"));

		comboBoxPosterSize = new JComboBox();
		comboBoxPosterSize_initialize();
		comboBoxPosterSize.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxPosterSize_itemStateChanged();
			}
		});

		numberFieldPosterWidth = new NumberField();
		numberField_initialize(numberFieldPosterWidth);
		numberFieldPosterWidth.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldPosterSize_actionPerformed(e.getSource());
			}
		});

		numberFieldPosterHeight = new NumberField();
		numberField_initialize(numberFieldPosterHeight);
		numberFieldPosterHeight.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldPosterSize_actionPerformed(e.getSource());
			}
		});

		comboBoxPosterSizeUnit = new JComboBox();
		comboBoxPosterSizeUnit_initialize();
		comboBoxPosterSizeUnit.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxPosterSizeUnit_itemStateChanged();
			}
		});

		checkBoxPosterSizeKeepAspectRatio = new JCheckBox();
		checkBoxPosterSizeKeepAspectRatio.setText(Lib.getMessage("checkBoxPosterSizeKeepAspectRatio.Text"));
		checkBoxPosterSizeKeepAspectRatio.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				checkBoxPosterSizeKeepAspectRatio_itemStateChanged();
			}
		});

		labelPaperSize = new JLabel();
		labelPaperSize.setText(Lib.getMessage("labelPaperSize.Text"));

		comboBoxPaperSize = new JComboBox();
		comboBoxPaperSize_initialize();
		comboBoxPaperSize.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxPaperSize_itemStateChanged();
			}
		});

		numberFieldPaperWidth = new NumberField();
		numberField_initialize(numberFieldPaperWidth);
		numberFieldPaperWidth.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldPaperSize_actionPerformed(e.getSource());
			}
		});

		numberFieldPaperHeight = new NumberField();
		numberField_initialize(numberFieldPaperHeight);
		numberFieldPaperHeight.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldPaperSize_actionPerformed(e.getSource());
			}
		});

		comboBoxPaperSizeUnit = new JComboBox();
		comboBoxPaperSizeUnit_initialize();
		comboBoxPaperSizeUnit.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxPaperSizeUnit_itemStateChanged();
			}
		});

		labelPaperMargin = new JLabel();
		labelPaperMargin.setText(Lib.getMessage("labelPaperMargin.Text"));

		numberFieldPaperMargin = new NumberField();
		numberField_initialize(numberFieldPaperMargin);
		numberFieldPaperMargin.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				numberFieldPaperMargin_actionPerformed(e.getSource());
			}
		});

		comboBoxPaperMarginUnit = new JComboBox();
		comboBoxPaperMarginUnit_initialize();
		comboBoxPaperMarginUnit.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				comboBoxPaperMarginUnit_itemStateChanged();
			}
		});

		checkBoxPaperRotate = new JCheckBox();
		checkBoxPaperRotate.setText(Lib.getMessage("checkBoxPaperRotate.Text"));
		checkBoxPaperRotate.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				checkBoxPaperRotate_itemStateChanged();
			}
		});

		buttonPaperOptimize = new JButton();
		buttonPaperOptimize.setText(Lib.getMessage("buttonPaperOptimize.Text"));
		buttonPaperOptimize.setEnabled(false);
		buttonPaperOptimize.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				buttonPaperOptimize_actionPerformed();
			}
		});

		labelPreview = new JLabel();
		labelPreview.setText(Lib.getMessage("labelPreview.Text"));

		labelOutputPaperCount = new JLabel();
		labelOutputPaperCount.setText(Lib.getMessage("labelOutputPaperCount.Text"));

		numberFieldOutputPaperCount = new NumberField();
		numberFieldOutputPaperCount.setColumns(7);
		numberFieldOutputPaperCount.setDisplayDigit(0);
		numberFieldOutputPaperCount.setEditable(false);

		canvasPreview = new CanvasPreview();
		//canvasPreview.setMaximumSize(new Dimension(500, 400));
		//canvasPreview.setMinimumSize(new Dimension(0, 0));
		//canvasPreview.setPreferredSize(new Dimension(500, 200));

		labelOutputFileName = new JLabel();
		labelOutputFileName.setText(Lib.getMessage("labelOutputFileName.Text"));

		textFieldOutputFileName = new JTextField();
		textFieldOutputFileName.setColumns(40);

		//labelMessage = new JLabel();

		buttonOutput = new JButton();
		buttonOutput.setText(Lib.getMessage("buttonOutput.Text") + "...");
		buttonOutput.setEnabled(false);
		buttonOutput.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				buttonOutput_actionPerformed();
			}
		});

		if(Lib.getOS() == Lib.OS_MAC) {
			menuFile.add(menuItemNewWindow);
			menuFile.add(menuItemOpenFile);
			menuFile.addSeparator();
			menuFile.add(menuItemCloseWindow);
			menuFile.add(menuItemOutput);
		} else {
			menuFile.add(menuItemNewWindow);
			menuFile.add(menuItemOpenFile);
			menuFile.add(menuItemCloseWindow);
			menuFile.addSeparator();
			menuFile.add(menuItemOutput);
			menuFile.addSeparator();
			menuFile.add(menuItemExit);
		}

		menuBar.add(menuFile);

		JPanel panelSrcFile = new JPanel();
		panelSrcFile.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelSrcFile.add(labelSrcFile);
		panelSrcFile.add(textFieldSrcFileName);
		panelSrcFile.add(labelSrcFileType);
		panelSrcFile.add(buttonSrcFileChange);

		JPanel panelSrc = new JPanel();
		panelSrc.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelSrc.add(labelSrcPageNum);
		panelSrc.add(comboBoxSrcPageNum);
		panelSrc.add(labelSrcImageSize);
		panelSrc.add(numberFieldSrcImageWidth);
		panelSrc.add(numberFieldSrcImageHeight);
		panelSrc.add(comboBoxSrcImageSizeUnit);

		JPanel panelSrcRect = new JPanel();
		panelSrcRect.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelSrcRect.add(labelSrcRect);
		panelSrcRect.add(checkBoxSrcRectWholeOfPage);
		panelSrcRect.add(numberFieldSrcLeft);
		panelSrcRect.add(numberFieldSrcBottom);
		panelSrcRect.add(numberFieldSrcRight);
		panelSrcRect.add(numberFieldSrcTop);
		panelSrcRect.add(comboBoxSrcRectUnit);

		JPanel panelPosterSize = new JPanel();
		panelPosterSize.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelPosterSize.add(labelPosterSize);
		panelPosterSize.add(comboBoxPosterSize);
		panelPosterSize.add(numberFieldPosterWidth);
		panelPosterSize.add(numberFieldPosterHeight);
		panelPosterSize.add(comboBoxPosterSizeUnit);
		panelPosterSize.add(checkBoxPosterSizeKeepAspectRatio);

		JPanel panelPaperSize = new JPanel();
		panelPaperSize.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelPaperSize.add(labelPaperSize);
		panelPaperSize.add(comboBoxPaperSize);
		panelPaperSize.add(numberFieldPaperWidth);
		panelPaperSize.add(numberFieldPaperHeight);
		panelPaperSize.add(comboBoxPaperSizeUnit);
		panelPaperSize.add(labelPaperMargin);
		panelPaperSize.add(numberFieldPaperMargin);
		panelPaperSize.add(comboBoxPaperMarginUnit);
		panelPaperSize.add(checkBoxPaperRotate);
		panelPaperSize.add(buttonPaperOptimize);

		JPanel panelPreviewText = new JPanel();
		panelPreviewText.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelPreviewText.add(labelPreview);
		panelPreviewText.add(labelOutputPaperCount);
		panelPreviewText.add(numberFieldOutputPaperCount);

		JPanel panelOutputFileName = new JPanel();
		panelOutputFileName.setLayout(new FlowLayout(FlowLayout.LEFT));
		panelOutputFileName.add(labelOutputFileName);
		panelOutputFileName.add(textFieldOutputFileName);

		JPanel panelOutput = new JPanel();
		switch(Lib.getOS()) {
		case Lib.OS_MAC:
		    panelOutput.setLayout(new FlowLayout(FlowLayout.RIGHT, 15, 15));
		    //panelOutput.add(labelMessage);
		    panelOutput.add(buttonOutput);
		    break;
		default:
		    panelOutput.setLayout(new FlowLayout(FlowLayout.LEFT, 15, 15));
		    panelOutput.add(buttonOutput);
		    //panelOutput.add(labelMessage);
		    break;
		}

		JPanel panelNorth = new JPanel();
		panelNorth.setLayout(new BoxLayout(panelNorth, BoxLayout.Y_AXIS));
		panelNorth.add(panelSrcFile);
		panelNorth.add(panelSrc);
		panelNorth.add(panelSrcRect);
		panelNorth.add(panelPosterSize);
		panelNorth.add(panelPaperSize);
		panelNorth.add(panelPreviewText);

		JPanel panelSouth = new JPanel();
		panelSouth.setLayout(new BoxLayout(panelSouth, BoxLayout.Y_AXIS));
		//panelSouth.add(panelOutputFileName);
		panelSouth.add(panelOutput);

		this.setJMenuBar(menuBar);
		this.getRootPane().setDefaultButton(buttonOutput);

		this.setLayout(new BorderLayout());
		this.add(panelNorth, BorderLayout.NORTH);
		this.add(panelSouth, BorderLayout.SOUTH);
		this.add(canvasPreview, BorderLayout.CENTER);
		/*
		this.setLayout(new BoxLayout(this.getContentPane(), BoxLayout.Y_AXIS));
		this.add(panelSrcFile);
		this.add(panelSrc);
		this.add(panelSrcRect);
		this.add(panelPosterSize);
		this.add(panelPaperSize);
		this.add(panelPreviewText);
		this.add(canvasPreview);
		this.add(panelOutputFileName);
		this.add(panelOutput);*/
		/*
		GroupLayout layout = new GroupLayout(this.getContentPane());
		this.getContentPane().setLayout(layout);
		layout.setAutoCreateGaps(true);
		layout.setAutoCreateContainerGaps(true);

		GroupLayout.ParallelGroup pGroupSrcFile = layout.createParallelGroup(GroupLayout.Alignment.BASELINE)
		    .addComponent(labelSrcFile)
		    .addComponent(textFieldSrcFileName)
		    .addComponent(labelSrcFileType)
		    .addComponent(buttonSrcFileChange);

		GroupLayout.SequentialGroup vGroup = layout.createSequentialGroup();
		vGroup.addGroup(pGroupSrcFile);
		layout.setVerticalGroup(vGroup);

		GroupLayout.SequentialGroup sGroupSrcFile = layout.createSequentialGroup()
		    .addComponent(labelSrcFile)
		    .addComponent(textFieldSrcFileName)
		    .addComponent(labelSrcFileType)
		    .addComponent(buttonSrcFileChange);
		GroupLayout.SequentialGroup hGroup = layout.createSequentialGroup();
		hGroup.addGroup(layout.createParallelGroup().addGroup(sGroupSrcFile));
		layout.setHorizontalGroup(hGroup);*/
	}

	private void mainFrame_initialize() {

		this.preferredPosterWidth = PaperSize.A1.getWidth();
		this.preferredPosterHeight = PaperSize.A1.getHeight();
		updatePosterSize();
		numberFieldPosterSize_setValue();
		comboBoxPosterSize.setSelectedIndex(comboBoxPosterSize.getItemCount() - 1);
		comboBoxPosterSize_itemStateChanged();
		checkBoxPosterSizeKeepAspectRatio.setSelected(true);
		checkBoxPosterSizeKeepAspectRatio_itemStateChanged();

		comboBoxPaperSize.setSelectedIndex(0);
		comboBoxPaperSize_itemStateChanged();
		comboBoxPaperSizeUnit.setSelectedIndex(0);
		comboBoxPaperSizeUnit_itemStateChanged();
		this.paperMargin = 72;
		numberFieldPaperMargin_setValue();
		comboBoxPaperMarginUnit.setSelectedIndex(0);
		comboBoxPaperMarginUnit_itemStateChanged();

		textFieldOutputFileName.setText(String.format("{pwd}%s%s_{input}_{date}.pdf", File.separator, Lib.getMessage("App.LowerTitle")));

		simulateOutput();
	}

	public MainFrame() {
		super();
		mainFrame_initializeComponents();
		mainFrame_initialize();
	}
}
