/**
	Copyright (c) 2008, hedachi ( http://blog.pettan.jp/ )
	All rights reserved.

	Redistribution and use in source and binary forms, with or without modification,
	are permitted provided that the following conditions are met:

	1. Redistributions of source code must retain the above copyright notice,
	   this list of conditions and the following disclaimer.
	2. Redistributions in binary form must reproduce the above copyright notice,
	   this list of conditions and the following disclaimer in the documentation
	   and/or other materials provided with the distribution.
	3. Neither the name of the pettan nor the names of its contributors may be used
	   to endorse or promote products derived from this software without specific
	   prior written permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
	OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
	SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
	TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
	BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
	ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
	DAMAGE.
*/

package filters;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.sql.Timestamp;
import java.util.Enumeration;
import java.util.Locale;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

public final class RequestDumperFilter implements Filter {
    private FilterConfig filterConfig = null;

    public void destroy() {
        this.filterConfig = null;
    }

    public void doFilter(ServletRequest request, ServletResponse response,
            FilterChain chain) throws IOException, ServletException {
        if (this.filterConfig == null) {
            return;
        }

        StringWriter sw = new StringWriter();
        PrintWriter writer = new PrintWriter(sw);
        writer.println("Request Received at "
                + new Timestamp(System.currentTimeMillis()));
        writer.println(" characterEncoding=" + request.getCharacterEncoding());
        writer.println("     contentLength=" + request.getContentLength());
        writer.println("       contentType=" + request.getContentType());
        writer.println("            locale=" + request.getLocale());
        writer.print("           locales=");
        Enumeration<?> locales = request.getLocales();
        boolean first = true;
        while (locales.hasMoreElements()) {
            Locale locale = (Locale) locales.nextElement();
            if (first)
                first = false;
            else
                writer.print(", ");
            writer.print(locale.toString());
        }
        writer.println();
        Enumeration<?> names = request.getParameterNames();
        while (names.hasMoreElements()) {
            String name = (String) names.nextElement();
            writer.print("         parameter=" + name + "=");
            String[] values = request.getParameterValues(name);
            for (int i = 0; i < values.length; i++) {
                if (i > 0)
                    writer.print(", ");
                writer.print(values[i]);
            }
            writer.println();
        }
        writer.println("          protocol=" + request.getProtocol());
        writer.println("        remoteAddr=" + request.getRemoteAddr());
        writer.println("        remoteHost=" + request.getRemoteHost());
        writer.println("            scheme=" + request.getScheme());
        writer.println("        serverName=" + request.getServerName());
        writer.println("        serverPort=" + request.getServerPort());
        writer.println("          isSecure=" + request.isSecure());

        if ((request instanceof HttpServletRequest)) {
            writer.println("---------------------------------------------");
            HttpServletRequest hrequest = (HttpServletRequest) request;
            writer.println("       contextPath=" + hrequest.getContextPath());
            Cookie[] cookies = hrequest.getCookies();
            if (cookies == null)
                cookies = new Cookie[0];
            for (int i = 0; i < cookies.length; i++) {
                writer.println("            cookie=" + cookies[i].getName()
                        + "=" + cookies[i].getValue());
            }
            names = hrequest.getHeaderNames();
            while (names.hasMoreElements()) {
                String name = (String) names.nextElement();
                String value = hrequest.getHeader(name);
                writer.println("            header=" + name + "=" + value);
            }
            writer.println("            method=" + hrequest.getMethod());
            writer.println("          pathInfo=" + hrequest.getPathInfo());
            writer.println("       queryString=" + hrequest.getQueryString());
            writer.println("        remoteUser=" + hrequest.getRemoteUser());
            writer.println("requestedSessionId="
                    + hrequest.getRequestedSessionId());
            writer.println("        requestURI=" + hrequest.getRequestURI());
            writer.println("       servletPath=" + hrequest.getServletPath());
        }
        writer.println("=============================================");

        writer.flush();
        this.filterConfig.getServletContext().log(sw.getBuffer().toString());

        chain.doFilter(request, response);
    }

    public void init(FilterConfig filterConfig) throws ServletException {
        this.filterConfig = filterConfig;
    }

    public String toString() {
        if (this.filterConfig == null)
            return "RequestDumperFilter()";
        StringBuffer sb = new StringBuffer("RequestDumperFilter(");
        sb.append(this.filterConfig);
        sb.append(")");
        return sb.toString();
    }
}
