unit DataObjects;

// $Id: DataObjects.pas,v 1.5 2002/04/04 18:38:46 takao Exp $

{$ObjExportAll On}

interface

uses
  ActiveX, Classes, Graphics, Windows;

type
  HGLOBAL = type LongInt;
  {$EXTERNALSYM HGLOBAL}

  TEnumFormatEtcProc = procedure(const aFormatEtc: TFormatEtc) of Object;

  { TDataObjectReader }
  TDataObjectReader = class
  private
    FDataObject: IDataObject;

    storageMedium_: TStgMedium;
    storageMediumValid_: Boolean;

  protected
    procedure ReleaseStorageMedium;

    procedure SetDataObject(aDataObject: IDataObject);

  public
    constructor Create(aDataObject: IDataObject = Nil);
    destructor Destroy; override;

    procedure GetFormats(aProc: TEnumFormatEtcProc);
    function HasFormat(aFormat: Word): Boolean;

    // deprecated functions
    function GetAsHandle(aFormat: Word; aTymed: Longint = TYMED_HGLOBAL): HGLOBAL;
    procedure GetAsFileList(files: TStrings);
    procedure GetAsBitmap(aBitmap: Graphics.TBitmap);

    // new style functions
    function GetHandle(aFormat: Word; aTymed: Longint = TYMED_HGLOBAL): HGLOBAL;
    function GetText: string;
    procedure GetFileList(files: TStrings);
    procedure GetBitmap(aBitmap: Graphics.TBitmap);
    procedure GetRichText(lines: TStrings);

    property AsText: string read GetText;
    property DataObject: IDataObject read FDataObject write FDataObject;
  end;

function GetStgMedium(aDataObject: IDataObject;
  var aStorageMedium: TStgMedium; aFormat: Word;
  aTypeOfMedium: Integer = TYMED_HGLOBAL
): Longint;
function GetText(aDataObject: IDataObject): string;
procedure GetFileList(aDataObject: IDataObject; files: TStrings);
procedure GetBitmap(aDataObject: IDataObject; aBitmap: Graphics.TBitmap);
procedure GetRichText(aDataObject: IDataObject; lines: TStrings);


implementation

uses
  ComObj, ShellApi, OleDnDUtils;

{ TDataObjectReader }

constructor TDataObjectReader.Create(aDataObject: IDataObject);
begin
  FDataObject := aDataObject;
end;

destructor TDataObjectReader.Destroy;
begin
  inherited;
end;

function TDataObjectReader.GetHandle(aFormat: Word;
  aTymed: Integer): HGLOBAL;
var
  formatEtc: TFormatEtc;
begin
  Result := 0;
  if FDataObject = Nil then Exit;

  formatEtc.cfFormat := aFormat;
  formatEtc.dwAspect := DVASPECT_CONTENT;
  formatEtc.lindex := -1;
  formatEtc.ptd := Nil;
  formatEtc.tymed := aTymed;

  ReleaseStorageMedium;

  if S_OK <> FDataObject.GetData(formatEtc, storageMedium_) then Exit;

  storageMediumValid_ := True;
  Result := storageMedium_.hGlobal;
end;

procedure TDataObjectReader.GetBitmap(aBitmap: Graphics.TBitmap);
var
  handle: THandle;
begin
  handle := GetHandle(CF_BITMAP, TYMED_GDI);  // type of medium : gdi
  if handle = 0 then Exit
  else aBitmap.Handle := handle;
end;

procedure TDataObjectReader.GetFileList(files: TStrings);
var
  handle: THandle;
  i, n: Integer;
  buff: array[0..255] of Char;
begin
  handle := GetHandle(CF_HDROP);
  if handle = 0 then Exit;

  n := DragQueryFile(handle, $ffffffff, Nil, 0);
  for i := 0 to n - 1 do begin
    DragQueryFile(handle, i, buff, 255);
    files.Add(buff);
  end;
  DragFinish(handle);
end;

procedure TDataObjectReader.GetFormats(aProc: TEnumFormatEtcProc);
var
  enum: IEnumFORMATETC;
  i, count: Integer;
  formats: array[0..99] of TFormatEtc;
begin
  enum := Nil;
  if SUCCEEDED(FDataObject.EnumFormatEtc(DATADIR_GET, enum)) then begin
    count := 100;
    enum.Next(count, formats, @count);
    for i := 0 to count - 1 do aProc(formats[i]);
  end;
end;

function TDataObjectReader.GetText: string;
var
  handle: THandle;
  mem: Pointer;
  size: Integer;
  text: string;
begin
  Result := '';

  handle := GetHandle(CF_TEXT);
  if handle = 0 then Exit;

  size := GlobalSize(handle);
  mem := GlobalLock(handle);
  try
    SetLength(text, size);
    CopyMemory(PChar(text), mem, size);
  finally
    GlobalUnlock(handle);
  end;

  Result := text;
end;

function TDataObjectReader.HasFormat(aFormat: Word): Boolean;
var
  fe: TFormatEtc;
begin
  fe.cfFormat := aFormat;
  fe.dwAspect := DVASPECT_CONTENT;
  fe.lindex := -1;
  fe.ptd := Nil;
  fe.tymed := TYMED_HGLOBAL;

  Result := S_OK = FDataObject.QueryGetData(fe);
end;

procedure TDataObjectReader.ReleaseStorageMedium;
begin
  if storageMediumValid_ then begin
    ReleaseStgMedium(storageMedium_);
    storageMediumValid_ := False;
  end;
end;

procedure TDataObjectReader.SetDataObject(aDataObject: IDataObject);
begin
  FDataobject := aDataObject;
end;

procedure TDataObjectReader.GetAsBitmap(aBitmap: Graphics.TBitmap);
begin
  GetBitmap(aBitmap);
end;

procedure TDataObjectReader.GetAsFileList(files: TStrings);
begin
  GetFileList(files);
end;

function TDataObjectReader.GetAsHandle(aFormat: Word;
  aTymed: Integer): HGLOBAL;
begin
  Result := GetHandle(aFormat, aTymed);
end;

procedure TDataObjectReader.GetRichText(lines: TStrings);
var
  handle: THandle;
  mem: Pointer;
  stream: TMemoryStream;
begin
  stream := TMemoryStream.Create;
  try
    handle := GetHandle(GetCFValue('Rich Text Format'));
    mem := GlobalLock(handle);
    stream.Write(mem^, GlobalSize(handle));
    stream.Position := 0;
    lines.LoadFromStream(stream);
  finally
    stream.Free;
  end;
end;

{ global functions }

function GetStgMedium(aDataObject: IDataObject;
  var aStorageMedium: TStgMedium; aFormat: Word; aTypeOfMedium: Integer
  ): Longint;
var
  fe: TFormatEtc;
begin
  fe.cfFormat := aFormat;
  fe.dwAspect := DVASPECT_CONTENT;
  fe.lindex := -1;
  fe.ptd := Nil;
  fe.tymed := aTypeOfMedium;
  Result := aDataObject.GetData(fe, aStorageMedium);
  OleCheck(Result);
end;

function GetText(aDataObject: IDataObject): string;
var
  stm: TStgMedium;
begin
  GetStgMedium(aDataObject, stm, CF_TEXT);
  SetString(Result, PChar(GlobalLock(stm.hGlobal)), GlobalSize(stm.hGlobal));
  ReleaseStgMedium(stm);
end;

procedure GetFileList(aDataObject: IDataObject; files: TStrings);
var
  stm: TStgMedium;
  i, n: Integer;
  buff: array[0..255] of Char;
begin
  GetStgMedium(aDataObject, stm, CF_HDROP);
  n := DragQueryFile(stm.hGlobal, $ffffffff, Nil, 0);
  for i := 0 to n - 1 do begin
    DragQueryFile(stm.hGlobal, i, buff, 255);
    files.Add(buff);
  end;
  DragFinish(stm.hGlobal);
  ReleaseStgMedium(stm);
end;

function HasFormat(aDataObject: IDataObject; aFormat: Word): Boolean;
var
  fe: TFormatEtc;
begin
  fe.cfFormat := aFormat;
  fe.dwAspect := DVASPECT_CONTENT;
  fe.lindex := -1;
  fe.ptd := Nil;
  fe.tymed := TYMED_HGLOBAL;

  Result := S_OK = aDataObject.QueryGetData(fe);
end;

procedure SetPalette(aDataObject: IDataObject; aBitmap: Graphics.TBitmap);
var
  stm: TStgMedium;
begin
  if HasFormat(aDataObject, CF_PALETTE) then begin
    GetStgMedium(aDataObject, stm, CF_PALETTE);
    aBitmap.Palette := stm.hGlobal;
    ReleaseStgMedium(stm);
  end;
end;

procedure GetBitmap(aDataObject: IDataObject; aBitmap: Graphics.TBitmap);
var
  stm: TStgMedium;
begin
  SetPalette(aDataObject, aBitmap);
  GetStgMedium(aDataObject, stm, CF_BITMAP, TYMED_GDI);
  aBitmap.Handle := stm.hBitmap;
  ReleaseStgMedium(stm);
end;

procedure GetRichText(aDataObject: IDataObject; lines: TStrings);
var
  stm: TStgMedium;
  stream: TMemoryStream;
begin
  stream := TMemoryStream.Create;
  try
    GetStgMedium(aDataObject, stm, GetCFValue('Rich Text Format'));
    stream.Write(GlobalLock(stm.hGlobal)^, GlobalSize(stm.hGlobal));
    GlobalFree(stm.hGlobal);
    ReleaseStgMedium(stm);

    stream.Position := 0;
    lines.LoadFromStream(stream);
  finally
    stream.Free;
  end;
end;


end.

