/*
 * Copyright (C) 2009 - 2010 Funambol, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY FUNAMBOL, FUNAMBOL DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301 USA.
 *
 * You can contact Funambol, Inc. headquarters at 643 Bair Island Road, Suite
 * 305, Redwood City, CA 94063, USA, or at email address info@funambol.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by Funambol" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by Funambol".
 */

/* $Id$ */

#include "executionqueue/CommandsQueue.h"
#include "executionqueue/ICommand.h"

#include "lock.h"
#include "TimeLog.h"

#include "executionqueue/CommandInvokeThread.h"

using namespace NS_DM_Client;
using namespace NS_DM_Client::NS_ExecutionQueue;
using namespace NS_DM_Client::NS_Common;

CommandsQueue::CommandsQueue(NS_Logging::Logger& log)
: m_CmdInvokeThread(0),
m_logger(log)
{
}

CommandsQueue::~CommandsQueue(void)
{
    if (m_CmdInvokeThread)
    {
        if (m_CmdInvokeThread->running())
        {
			LOG_WARNING_(m_logger, "CmdInvokeThread is still running. ");
            m_CmdInvokeThread->softTerminate();
        }
        delete m_CmdInvokeThread;
        m_CmdInvokeThread = 0;
    }

    discardWaitingCommands();

}


bool CommandsQueue::Init()
{
    m_CmdInvokeThread = new CommandInvokeThread(m_resEvent, *this, m_logger);
    if (m_CmdInvokeThread)
    {
        m_CmdInvokeThread->start();
		LOG_DEBUG_(m_logger, "CmdInvokeThread (%p) of CmdsQ (%p) is running=%d. ", m_CmdInvokeThread, this, m_CmdInvokeThread->running());
    }

    return m_CmdInvokeThread != NULL && m_CmdInvokeThread->running();
}


bool CommandsQueue::Add(ICommand& cmd)
{
    bool brc = false;
    if (m_CmdInvokeThread && m_CmdInvokeThread->running())
    {
    	Lock lock(m_CmdsLocker);

        bool noCommands = m_Cmds.empty();
        m_Cmds.push_back(&cmd);

		LOG_DEBUG_(m_logger, "CmdsQ (%p) is empty=%d. ", this, noCommands);

        if (noCommands)
        {
			LOG_DEBUG_(m_logger, "issue NEWCOMMAND signal. ");
            m_resEvent.signal(e_NewCommand);
        }

        brc = true;
    }
    else
    {
        LOG_WARNING_(m_logger, "CmdInvokeThread is not running. ");
    }

    return brc;
}


bool CommandsQueue::StopCommandExecution()
{
    m_CmdInvokeThread->softTerminate();

    const unsigned long waitThreadStopMilisec = 500;
    TimeLog timeLog;

    if (timeLog.Snap() != 0)
    {
        LOG_DEBUG_(m_logger, "clock_gettime failed. ");
    }

    bool waitResult = m_CmdInvokeThread->wait(waitThreadStopMilisec);
    if (timeLog.Snap() != 0)
    {
        LOG_DEBUG_(m_logger, "clock_gettime failed. ");
    }
    else
    {
        LOG_DEBUG_(m_logger, "Thread ending wait time: %d nsec. ", timeLog.NanoSeconds());
    }

    if (!waitResult)
    {
        LOG_WARNING_(m_logger, "CmdInvokeThread did not stop. ");
    }
    else
    {
		discardWaitingCommands();
    }

    return m_CmdInvokeThread->finished();
}


ICommand*   CommandsQueue::getCommand()
{
	Lock lock(m_CmdsLocker);
    ICommand* cmd = NULL;
    if (!m_Cmds.empty())
    {
        cmd = m_Cmds.front();
    }
    else
    {
        LOG_INFO_(m_logger, "CmdsQ is empty - can not get a command. ");
    }

    return cmd;
}


ICommand*   CommandsQueue::popCommand()
{
	Lock lock(m_CmdsLocker);
    ICommand* cmd = NULL;
    if (!m_Cmds.empty())
    {
        cmd = m_Cmds.front();
        m_Cmds.pop_front();
    }
    else
    {
        LOG_WARNING_(m_logger, "CmdsQ is empty - can not pop a command. ");
	}

    return cmd;
}


void		CommandsQueue::discardWaitingCommands()
{
	Lock lock(m_CmdsLocker);
    for (Commands::iterator i = m_Cmds.begin(), end = m_Cmds.end();
		i != end; ++i)
	{
        LOG_DEBUG_(m_logger, "Command %p. ", *i);
		delete *i;
	}

	m_Cmds.clear();
}


