/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.nano;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public class OutputPort extends Datum {

	//
	private PrintStream lineWriter;
	private boolean standard;
	private boolean closed = false;
	private LispMessage msg;

	//
	private OutputPort(OutputStream wr, boolean std, LispMessage msg) {
		if(wr instanceof PrintStream) {
			this.lineWriter = (PrintStream)wr;
		} else {
			this.lineWriter = new PrintStream(wr);
		}
		this.standard   = std;
		this.msg        = msg;
	}

	/**
	 * 
	 * @param wr
	 * @param msg
	 */
	public OutputPort(OutputStream wr, LispMessage msg) {
		this(wr, false, msg);
	}

	/**
	 * 
	 * @param fname
	 * @param msg
	 */
	public OutputPort(File fname, LispMessage msg) {
		try {
			OutputStream wr;

			wr = new FileOutputStream(fname);
			this.lineWriter = new PrintStream(wr);
			this.standard   = false;
		} catch (FileNotFoundException e) {
			throw new LispIOException(e);
		}

	}

	/**
	 * 
	 * @param fname
	 * @param msg
	 * @throws UnsupportedEncodingException 
	 */
	public OutputPort(File fname, String encoding,
			LispMessage msg) throws UnsupportedEncodingException {
		try {
			OutputStream wr;

			wr = new FileOutputStream(fname);
			this.lineWriter = new PrintStream(wr, true, encoding);
			this.standard   = false;
		} catch (FileNotFoundException e) {
			throw new LispIOException(e);
		}

	}

	/**
	 * 
	 * @param msg
	 * @return
	 */
	public static OutputPort getStandard(LispMessage msg) {
		return new OutputPort(System.out, true, msg);
	}

	/**
	 * 
	 * @param ch
	 */
	public void writeChar(LispCharacter ch) {
		if(closed) {
			throw msg.getError("err.port.closed");
		}

		if(standard) {
			System.out.print(ch.getCharacter());
		} else {
			lineWriter.print(ch.getCharacter());
		}
	}

	/**
	 * 
	 * @param d
	 */
	public void write(Datum d) {
		String s;

		if(closed) {
			throw msg.getError("err.port.closed");
		}

		s = LispUtils.toWriteWithoutSS(d);
		if(standard) {
			System.out.print(s);
		} else {
			lineWriter.print(s);
		}
	}

	/**
	 * 
	 * @param d
	 */
	public void display(Datum d) {
		String s;

		if(closed) {
			throw msg.getError("err.port.closed");
		}

		s = LispUtils.toDisplay(d);
		if(standard) {
			System.out.print(s);
		} else {
			lineWriter.print(s);
		}
	}

	/**
	 * 
	 */
	public void newline() {
		if(closed) {
			throw msg.getError("err.port.closed");
		}

		if(standard) {
			System.out.println();
		} else {
			lineWriter.println();
		}
	}

	/**
	 * 
	 */
	public void close() {
		if(!closed && !standard) {
			lineWriter.close();
			closed = true;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#isTypePort()
	 */
	public boolean isTypePort() {
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.nano.Datum#toDisplay()
	 */
	@Override
	public String toDisplay() {
		return "#<oport>";
	}

}
