/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.nano.namespace;

import java.util.Map;

import net.morilib.awk.nano.value.AwkFunction;
import net.morilib.awk.nano.value.AwkValue;

/**
 * awkiumの名前空間を表すインターフェースです。<br />
 * awkiumの名前空間は関数の名称と変数の名称を別々に管理します。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public interface AwkNamespace {

	/**
	 * 変数を探します。
	 * 
	 * @param s 変数名
	 * @return 変数の値、ないときはnull
	 */
	public AwkValue find(String s);

	/**
	 * 関数の名前空間から関数を探します。
	 * 
	 * @param s 関数名
	 * @return 関数、ないときはnull
	 */
	public AwkFunction findFunction(String s);

	/**
	 * 変数を割り当てます。
	 * 
	 * @param s 変数名
	 * @param v 割り当てる変数の値
	 */
	public void assign(String s, AwkValue v);

	/**
	 * 変数を引数として割り当てます。
	 * 
	 * @param s 変数名
	 * @param v 割り当てる変数の値
	 */
	public void assignArgument(String s, AwkValue v);

	/**
	 * 関数を割り当てます
	 * 
	 * @param s 関数名
	 * @param v 割り当てる関数
	 */
	public void bindFunction(String s, AwkFunction v);

	/**
	 * 変数が読み込みのみのときtrueを得ます。
	 * 
	 * @param s 変数名
	 */
	public boolean isReadonly(String s);

	/**
	 * この名前空間の下にある名前空間を得ます。<br />
	 * 名前空間が存在しないときは新規に名前空間を割り当てます。
	 * 
	 * @param s 名前空間名
	 * @return 名前空間
	 */
	public AwkNamespace subnamespace(String s);

	/**
	 * この名前空間の下にある名前空間を得ます。
	 * 
	 * @param s 名前空間名
	 * @return 名前空間、ないときはnull
	 */
	public AwkNamespace referNamespace(String s);

	/**
	 * n番目のフィールドを取得します。
	 * 
	 * @param n フィールド番号
	 * @return フィールド
	 */
	public AwkValue referField(int n);

	/**
	 * フィールドのサイズを取得します。
	 */
	public int sizeOfField();

	/**
	 * 行を設定し、フィールドを割り当てなおします。
	 * 
	 * @param s 設定する行
	 */
	public void setField(String s);

	/**
	 * フィールドを設定します。
	 * 
	 * @param n フィールド番号
	 * @param s フィールド
	 */
	public void setField(int n, String s);

	/**
	 * 組み込み変数FSの値を取得します。
	 */
	public String getFS();

	/**
	 * 組み込み変数RSの値を取得します。
	 */
	public String getRS();

	/**
	 * 組み込み変数OFSの値を取得します。
	 */
	public String getOFS();

	/**
	 * 組み込み変数ORSの値を取得します。
	 */
	public String getORS();

	/**
	 * 親の名前空間を取得します。
	 */
	public AwkNamespace getRoot();

	/**
	 * この名前空間がルートであるときtrueを得ます。
	 */
	public boolean isRoot();

	/**
	 * 新しい名前空間を生成します。
	 */
	public AwkNamespace newInstance();

	/**
	 * 大文字小文字を無視するかを得ます。
	 * 
	 * @return 大文字小文字を無視するときtrue
	 */
	public boolean isIgnoreCase();

	/**
	 * 
	 * @param s
	 * @return
	 */
	public AwkValue get(String s);

	/**
	 * 
	 * @return
	 */
	public Map<String, AwkValue> getValueMap();

	/**
	 * 
	 * @param s
	 * @return
	 */
	public AwkValue remove(String s);

}
