/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import net.morilib.automata.DFAState;
import net.morilib.nina.DFABuilder;
import net.morilib.nina.NinaState;
import net.morilib.nina.translate.sh.ReplaceStrangeChar;
import net.morilib.range.Interval;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/23
 */
public class NinaTranslatorCSharp extends AbstractNinaTranslator {

	//
	private static final Pattern RET_CONST = Pattern.compile(
			"return +([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*;");
	private static final Pattern E_CONST = Pattern.compile(
			"([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*");
	private static final String LOK = "__this__.LOOKAHEAD($c);";

	/**
	 * 
	 */
	public NinaTranslatorCSharp() {
		parent = this;
	}

	//
	private String tostr(int c) {
		String x;

		if(c == '\\') {
			x = "'\\\\'";
		} else if(c == '\n') {
			x = "'\\n'";
		} else if(c == '\'') {
			x = "'\\''";
		} else if(Character.isISOControl(c)) {
			x = Integer.toString(c);
		} else if(c < Character.MAX_VALUE) {
			x = String.format("'%c'", (char)c);
		} else {
			x = Integer.toString(c);
		}
		return x;
	}

	//
	private void printintv(PrintStream out, Interval v, boolean els,
			Object o) {
		String s, t, a, x, y, z;
		int c, d;

		s = v.isInfimumClosed() ? ">=" : ">";
		t = v.isSupremumClosed() ? "<=" : "<";
		a = els ? "\t\t\t} else if" : "\t\t\tif";
		c = ((Integer)v.getInfimumBound()).intValue();
		d = ((Integer)v.getSupremumBound()).intValue();
		x = tostr(c);
		y = tostr(d);
		z = o != null && o.toString().startsWith(LOK) ?
				"__l__ && " : "";
		if(v.isClosed() && c == d) {
			out.format("%s(%s__c__ == %s) {\n", a, z, x);
		} else {
			out.format("%s(%s__c__ %s %s && __c__ %s %s) {\n",
					a, z, s, x, t, y);
		}
	}

	//
	private void printobj(PrintStream out, Object o, boolean els,
			Object m) {
		String a, z;

		a = els ? "\t\t\t} else if" : "\t\t\tif";
		z = m != null && m.toString().startsWith(LOK) ?
				"__l__ && " : "";
		out.format("%s(%s__c__ == \"%s\") {\n", a, z, o.toString());
	}

	//
	private void printclass(PrintStream out, Object o, boolean els,
			Object m) {
		String a, s, z;
		char c;

		a = els ? "\t\t\t} else if" : "\t\t\tif";
		z = m != null && m.toString().startsWith(LOK) ?
				"__l__ && " : "";
		if(o instanceof Character) {
			if((c = ((Character)o).charValue()) == '\\' || c == '\'') {
				s = String.format("Character.valueOf('\\%c')", c);
			} else {
				s = String.format("Character.valueOf('%c')", c);
			}
			out.format("%s(%s%s == __c__) {\n", a, z, s);
		} else if(E_CONST.matcher(s = o.toString()).matches()) {
			out.format("%s(%s%s == __c__) {\n", a, z, s);
		} else {
			out.format("%s(%s__c__ is %s) {\n", a, z, s);
		}
	}

	//
	private String outln(boolean els, PrintStream out) {
		if(els) {
			out.println("\t\t\t} else {");
			return "\t";
		} else {
			return "";
		}
	}

	private boolean putend(String s, PrintStream out,
			DFAState<Object, ?, Void> dfa, boolean els) {
		DFABuilder.DBS b;
		Object o;

		if(dfa instanceof DFABuilder.DBS &&
				(b = ((DFABuilder.DBS)dfa).getEnd()) != null) {
			if(els) {
				out.printf("\t\t\t} else if(%s) {\n", s);
			} else {
				out.printf("\t\t\tif(%s) {\n", s);
			}

			if((o = ((DFABuilder.DBS)dfa).getMealyEnd()) != null) {
				out.format("\t\t\t\t%s\n", ReplaceActionCSharp.replace(
						o.toString()));
			}
			out.format("\t\t\t\t__this__.STATE = %d;\n", getStateNo(b));
			out.format("\t\t\t\treturn 1;\n");
			els = true;
		}
		return els;
	}

	private boolean putuseredges(PrintStream out,
			DFAState<Object, ?, Void> dfa, boolean els) {
		DFABuilder.DBS b;
		Set<String> k;
		Object o;

		if(dfa instanceof DFABuilder.DBS &&
				(k = ((DFABuilder.DBS)dfa).getUserEdges()) != null) {
			for(String x : k) {
				b = ((DFABuilder.DBS)dfa).getUserEdge(x);
				o = ((DFABuilder.DBS)dfa).getUserMealyEdge(x);
				if(els) {
					out.printf("\t\t\t} else if(__g && (%s)) {\n", x);
				} else {
					out.printf("\t\t\tif(__g && (%s)) {\n", x);
				}

				if(o != null) {
					out.format("\t\t\t\t%s\n", ReplaceActionCSharp.replace(
							o.toString()));
				}
				out.format("\t\t\t\t__this__.STATE = %d;\n", getStateNo(b));
				out.format("\t\t\t\treturn 1;\n");
				els = true;
			}
		}
		return els;
	}

	private boolean putothers(PrintStream out,
			DFAState<Object, ?, Void> dfa, boolean els) {
		boolean el2 = true;
		DFABuilder.DBS a, b;
		String s, t;
		Object o;

		if(!(dfa instanceof DFABuilder.DBS)) {
			// do nothing
		} else if((b = (a = (DFABuilder.DBS)dfa).getOthers()) != null) {
			s = outln(els, out);
			if((o = a.getMealyOthers()) != null) {
				out.format("%s\t\t\t%s\n", s, ReplaceActionCSharp.replace(
						o.toString()));
			}
			out.format("%s\t\t\t__this__.STATE = %d;\n", s, getStateNo(b));
			out.format("%s\t\t\treturn 1;\n", s);
			el2 = false;
		} else if((b = a.getRecursive()) != null) {
			s = outln(els, out);
			t = ReplaceStrangeChar.replace(a.getRecursiveName());
			out.format("%s\t\t\t__this__.__Stkpush(%d, __this__.ENGINE_%s);\n", s,
					getStateNo(b), t);
			out.format("%s\t\t\t__this__.STATE = 0;\n", s);
			out.format("%s\t\t\treturn -1;\n", s);
			el2 = false;
		}
		return el2;
	}

	//
	private void printState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		boolean els = false, el2 = true;
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa), c;
		Object o;

		out.format("\t\tcase %d:\n", sn);
		for(Interval v : dfa.getAlphabetRanges()) {
			c = ((Integer)v.getInfimumBound()).intValue();
			if(v.isInfimumOpen())  c++;
			printintv(out, v, els, o = dfa.getLabelInt(c));
			if(o != null) {
				out.format("\t\t\t\t%s\n", ReplaceActionCSharp.replace(
						o.toString()));
			}

			// print next step
			d = dfa.goInt(c);
			out.format("\t\t\t\t__this__.STATE = %d;\n", getStateNo(d));
			out.println("\t\t\t\treturn 1;");
			els = true;
		}
		els = putend("$c < 0", out, dfa, els);
		els = putuseredges(out, dfa, els);
		el2 = putothers(out, dfa, els);
		if(els)  out.println("\t\t\t}");
		if(el2)  out.println("\t\t\treturn 0;");
	}

	//
	private void printObjectState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		boolean els = false, el2 = true;
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa);
		Object o;

		out.format("\t\tcase %d:\n", sn);
		for(Object p : dfa.getAlphabets()) {
			printobj(out, p, els, o = dfa.getLabel(p));
			if(o != null) {
				out.format("\t\t\t\t%s\n", ReplaceActionCSharp.replace(
						o.toString()));
			}

			// print next step
			d = dfa.go(p);
			out.format("\t\t\t\t__this__.STATE = %d;\n", getStateNo(d));
			out.println("\t\t\t\treturn 1;");
			els = true;
		}
		els = putend("$c == null", out, dfa, els);
		els = putuseredges(out, dfa, els);
		el2 = putothers(out, dfa, els);
		if(els)  out.println("\t\t\t}");
		if(el2)  out.println("\t\t\treturn 0;");
	}

	//
	private void printClassState(PrintStream out,
			DFAState<Object, ?, Void> dfa) {
		boolean els = false, el2 = true;
		DFAState<Object, ?, Void> d;
		int sn = getStateNo(dfa);
		Object o;

		out.format("\t\tcase %d:\n", sn);
		for(Object p : dfa.getAlphabets()) {
			printclass(out, p, els, o = dfa.getLabel(p));
			if(o != null) {
				out.format("\t\t\t\t%s\n", ReplaceActionCSharp.replace(
						o.toString()));
			}

			// print next step
			d = dfa.go(p);
			out.format("\t\t\t\t__this__.STATE = %d;\n", getStateNo(d));
			out.println("\t\t\t\treturn 1;");
			els = true;
		}
		els = putend("$c == null", out, dfa, els);
		els = putuseredges(out, dfa, els);
		el2 = putothers(out, dfa, els);
		if(els)  out.println("\t\t\t}");
		if(el2)  out.println("\t\t\treturn 0;");
	}

	//
	private boolean isProcessed(DFAState<Object, ?, Void> state) {
		return containsState(state);
	}

	//
	private int getDeadStateNo() {
		Object s;

		s = builder.getDeadState();
		return containsState(s) ? stateNo(s) : -1;
	}

	//
	private int getStateNoByLabel(String l) {
		Object s;

		s = builder.getStateByLabel(l);
		return containsState(s) ? stateNo(s) : -1;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printStates(java.io.PrintStream)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void printStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		getStateNo(dfa.getInitialState());
		while(!isStackEmpty()) {
			printState(out, popStack());
		}

		for(String t : builder.getLabels()) {
			s = (DFAState<Object, ?, Void>)builder.getStateByLabel(t);
			if(s != null && !isProcessed(s)) {
				getStateNo(s);
				while(!isStackEmpty()) {
					printState(out, popStack());
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printStates(java.io.PrintStream)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void printObjectStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		getStateNo(dfa.getInitialState());
		while(!isStackEmpty()) {
			printObjectState(out, popStack());
		}

		for(String t : builder.getLabels()) {
			s = (DFAState<Object, ?, Void>)builder.getStateByLabel(t);
			if(s != null && !isProcessed(s)) {
				getStateNo(s);
				while(!isStackEmpty()) {
					printObjectState(out, popStack());
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printObjectStates(java.io.PrintStream)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void printClassStates(PrintStream out) {
		DFAState<Object, ?, Void> s;

		getStateNo(dfa.getInitialState());
		while(!isStackEmpty()) {
			printClassState(out, popStack());
		}

		for(String t : builder.getLabels()) {
			s = (DFAState<Object, ?, Void>)builder.getStateByLabel(t);
			if(s != null && !isProcessed(s)) {
				getStateNo(s);
				while(!isStackEmpty()) {
					printClassState(out, popStack());
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printAcceptStates(PrintStream out) {
		String d = "\t\treturn (";

		if(acceptsSize() == 0) {
			out.println("\t\treturn false;");
		} else {
			for(Integer i : acceptsIterable()) {
				out.print(d);
				out.format("__this__.STATE == %d", i);
				d = " ||\n\t\t\t\t";
			}
			out.println(");");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptToken(java.io.PrintStream)
	 */
	@Override
	public void printAcceptToken(PrintStream out) {
		boolean[] z = new boolean[1];
		int m = -1, q;
		NinaState n;
		String x, p;

		for(DFAState<Object, ?, Void> s : stateKeys()) {
			if(!s.isAccepted())  continue;
			p = null;  m = -1;
			out.format("\t\tcase %d:\n", stateNo(s));
			for(Object a : s.getAccepted()) {
				if(a == null || !(a instanceof NinaState)) {
					// do nothing
				} else if((q = ((NinaState)a).getPriority()) < 0) {
					// do nothing
				} else if(q > m) {
					m = q;
				}
			}

			for(Object a : s.getAccepted()) {
				if(a == null || !(a instanceof NinaState)) {
					// do nothing
				} else if((x = (n = (NinaState)a).getLabel()) == null) {
					// do nothing
				} else if((x = x.trim()).equals("")) {
					// do nothing
				} else if(m >= 0) {
					if(n.getPriority() == m)  p = x;
				} else if(RET_CONST.matcher(x).matches()) {
					p = x;  break;
				} else {
					if(p != null) {
						getOptions().pwarn("ambiguousaccept");
					}
					p = x;
				}
			}

			if(p != null) {
				p = ReplaceAction.replace(p, z, this, getStateNo(s),
						builder.getLabelByState(s));
			} else {
				p = "return __b__;";
			}
			out.format("\t\t\t%s\n", p);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptToken(java.io.PrintStream)
	 */
	@Override
	public void printActions(PrintStream out) {
		boolean[] a = new boolean[1];
		String x, p;

		for(DFAState<Object, ?, Void> s : stateKeys()) {
			p = null;
			out.format("\t\tcase %d:\n", stateNo(s));
			if((x = s.toString()) == null) {
				// do nothing
			} else if((x = x.trim()).equals("")) {
				// do nothing
			} else {
				p = x;
			}

			if(p == null) {
				out.println("\t\t\tbreak;");
			} else {
				p = ReplaceAction.replace(p, a, this, stateNo(s),
						builder.getLabelByState(s));
				out.format("\t\t\t%s\n", p);
				if(!a[0]) {
					out.println("\t\t\tbreak;");
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printImports(java.io.PrintStream)
	 */
	@Override
	public void printImports(List<String> imp, PrintStream out) {
		for(String s : imp) {
			out.format("import %s;\n", s);
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printIsEnd(PrintStream out) {
		Set<Integer> t = new HashSet<Integer>();
		String d = "\t\treturn (";

		for(DFAState<Object, ?, Void> s : stateKeys()) {
			if(!(s instanceof DFABuilder.DBS)) {
				// do nothing
			} else if(((DFABuilder.DBS)s).getEnd() != null) {
				t.add(stateNo(s));
			}
		}

		if(t.size() == 0) {
			out.println("\t\treturn false;");
		} else {
			for(Integer i : t) {
				out.print(d);
				out.format("__this__.STATE == %d", i);
				d = " ||\n\t\t\t\t";
			}
			out.println(");");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printDeadState(String n, PrintStream out) {
		out.printf("\t\t\treturn %d;\n", getDeadStateNo());
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printRecover(java.io.PrintStream)
	 */
	@SuppressWarnings("unchecked")
	@Override
	protected void printRecover(PrintStream out) {
		DFAState<Object, ?, Void> s;
		String d = "";

		for(String x : builder.getLabels()) {
			if(x.endsWith("Exception")) {
				s = (DFAState<Object, ?, Void>)builder.getStateByLabel(
						x);
				out.printf("\t\t\t%sif(e is %s) {\n", d, x);
				out.printf("\t\t\t\treturn %d;\n", getStateNo(s));
				d = "} else ";
			}
		}

		if(d.equals("")) {
			out.println("\t\t\treturn -1;");
		} else {
			out.println("\t\t\t} else {");
			out.println("\t\t\t\treturn -1;");
			out.println("\t\t\t}");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAcceptStates(java.io.PrintStream)
	 */
	@Override
	public void printFinallyState(PrintStream out) {
		out.printf("\t\t\treturn %d;\n", getStateNoByLabel("finally"));
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printInitTrap(java.io.PrintStream)
	 */
	@Override
	protected void printInitTrap(PrintStream out) {
		// do nothing
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printNFADeadState(java.io.PrintStream)
	 */
	@Override
	protected void printNFADeadState(PrintStream out) {
		out.print("return (false");
		for(DFAState<Object, ?, Void> s : stateKeys()) {
			if(s.isDead()) {
				out.print(" || ");
				out.print("__this__.STATE == ");
				out.print(stateNo(s));
			}
		}
		out.println(");");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printAttrs(java.io.PrintStream)
	 */
	@Override
	protected void printAttrs(PrintStream out) {
		// do nothing
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#printConstants(java.io.PrintStream)
	 */
	@Override
	protected void printConstants(PrintStream out) {
		for(Map.Entry<String, Integer> t : quadro.getConstantMap().entrySet()) {
			out.printf("\t\tpublic const int %s = %d;\n",
					t.getKey(), t.getValue().intValue());
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#reportStatistics(java.io.PrintStream)
	 */
	@Override
	public void reportStatistics(PrintStream std) {
		getOptions().print("statheader");
		getOptions().print("statstates", stateSize());
		getOptions().print("stataccept", acceptsSize());
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#openScript()
	 */
	@Override
	protected InputStream openScript() {
		return NinaTranslator.class.getResourceAsStream(
				"/net/morilib/nina/translate/nina_template." +
				getMachine() +
				".cs.sh");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#openOutput()
	 */
	@Override
	protected PrintStream openOutput() throws IOException {
		return new PrintStream(new FileOutputStream(
				getOptions().getTitleFile(".cs")), true);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#newPrototype()
	 */
	@Override
	protected AbstractNinaTranslator newPrototype() {
		NinaTranslatorCSharp r;

		r = new NinaTranslatorCSharp();
		r.quadro = quadro;
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#appendValue(java.lang.StringBuffer, java.lang.StringBuffer)
	 */
	protected void appendValue(StringBuffer ot, StringBuffer b1) {
		String l = b1.toString();
		Object o = builder.getStateByLabel(l);
		String t = builder.getTypeByLabel(l);

		if(o == null) {
			ot.append("__").append(b1).append("__");
		} else if(t == null) {
			ot.append("(__this__.__stv[__this__.__slen - 1][");
			ot.append(stateNo(o));
			ot.append("])");
		} else {
			ot.append("((");
			ot.append(t);
			ot.append(")(__this__.__stv[__this__.__slen - 1][");
			ot.append(stateNo(o));
			ot.append("]))");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#appendLvalue(java.lang.StringBuffer, java.lang.StringBuffer)
	 */
	protected void appendLvalue(StringBuffer ot, StringBuffer b1) {
		Object o = builder.getStateByLabel(b1.toString());

		if(o == null) {
			ot.append('@');
			ot.append(b1);
		} else {
			ot.append("(__this__.__stv[__this__.__slen - 1][");
			ot.append(stateNo(o));
			ot.append("])");
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#appendMyPosition(java.lang.StringBuffer, int)
	 */
	protected void appendMyPosition(StringBuffer ot,  String ln,
			int cn) {
		ot.append("(__this__.__stv[__this__.__slen - 1][");
		ot.append(cn);
		ot.append("])");
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.translate.AbstractNinaTranslator#appendReturn(java.lang.StringBuffer)
	 */
	@Override
	protected void appendReturn(StringBuffer ot) {
		ot.append("__r__");
	}

	@Override
	protected String getConstName(String t) {
		if(t.equals("EXIT")) {
			return "NinaAccept";
		} else if(t.equals("FAIL")) {
			return "NinaFail";
		} else if(t.equals("ACCEPT")) {
			return "NinaHaltAccept";
		} else if(t.equals("REJECT")) {
			return "NinaHaltReject";
		} else {
			return null;
		}
	}

	@Override
	protected void appendYield(StringBuffer ot, String b) {
		ot.append("__this__.yieldObject = (");
		ot.append(ReplaceActionCSharp.replace(b)).append(");");
		ot.append("return NinaYield;");
	}

}
