﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NDac.Keys
{
	/// <summary>
	/// キーワード検索の一致タイプを表します。
	/// </summary>
	public class MatchingType
	{
		/// <summary>
		/// キーワード検索の一致タイプの列挙型を表します。
		/// </summary>
		private enum MatchingTypeConstants
		{
			/// <summary>
			/// 前方一致を表します。
			/// </summary>
			Prefix,

			/// <summary>
			/// 後方一致を表します。
			/// </summary>
			Suffix,

			/// <summary>
			/// 前後方一致を表します。
			/// </summary>
			PreSuffix,

			/// <summary>
			/// 完全一致を表します。
			/// </summary>
			Perfect,

			/// <summary>
			/// 未定義の一致を表します。
			/// </summary>
			Unknown,
		}

		private MatchingTypeConstants	_type;
		private string					_toString;
		private static MatchingType	_prefix		= new MatchingType( MatchingTypeConstants.Prefix,		"前方一致" );
		private static MatchingType	_suffix		= new MatchingType( MatchingTypeConstants.Suffix,		"後方一致" );
		private static MatchingType	_presuffix	= new MatchingType( MatchingTypeConstants.PreSuffix,	"前後方一致" );
		private static MatchingType	_perfect	= new MatchingType( MatchingTypeConstants.Perfect,		"完全一致" );
		private static MatchingType	_unknown	= new MatchingType( MatchingTypeConstants.Unknown,		"未定義" );

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="type">キーワード検索の一致タイプ</param>
		/// <param name="toString">ToStringメソッドで変換される値</param>
		private MatchingType( MatchingTypeConstants type, string toString )
		{
			this._type		= type;

			this._toString	= toString;
		}

		/// <summary>
		/// 前方一致を表します。
		/// </summary>
		public static MatchingType Prefix
		{
			get
			{
				return( _prefix );
			}
		}

		/// <summary>
		/// 後方一致を表します。
		/// </summary>
		public static MatchingType Suffix
		{
			get
			{
				return( _suffix );
			}
		}

		/// <summary>
		/// 前後方一致を表します。
		/// </summary>
		public static MatchingType PreSuffix
		{
			get
			{
				return( _presuffix );
			}
		}

		/// <summary>
		/// 完全一致を表します。
		/// </summary>
		public static MatchingType Perfect
		{
			get
			{
				return( _perfect );
			}
		}

		/// <summary>
		/// 未定義を表します。
		/// </summary>
		public static MatchingType Unknown
		{
			get
			{
				return( _unknown );
			}
		}

		/// <summary>
		/// 一致を表す文字列に変換します。
		/// </summary>
		/// <returns>文字列</returns>
		public override string ToString()
		{
			return( this._toString );
		}

		/// <summary>
		/// 検索キーワードに変換します。
		/// </summary>
		/// <param name="originalKeyword">オリジナルキーワード</param>
		/// <returns>検索キーワード</returns>
		public string ToKeyword( string originalKeyword )
		{
			if( this._type == MatchingTypeConstants.Prefix )
			{
				return( originalKeyword + "%" );
			}
			else if( this._type == MatchingTypeConstants.Suffix )
			{
				return( "%" + originalKeyword );
			}
			else if( this._type == MatchingTypeConstants.PreSuffix )
			{
				return( "%" + originalKeyword + "%" );
			}
			else if( this._type == MatchingTypeConstants.Perfect )
			{
				return( originalKeyword );
			}
			else
			{
				return( originalKeyword );
			}
		}
	}
}
