using System;
using System.Collections;
using System.Collections.Generic;
using NDac.Keys;
using NDac.Modules.DataTransferObjects;
using NDac.Modules.Entities;

namespace NDac.Modules
{
	/// <summary>
	/// テーブルモジュールの列挙可能クラスを表します。
	/// </summary>
	class TableModuleEnumerable : TableModuleEnumerable< IKey, IEntity, IDataTransferObject >
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		public TableModuleEnumerable( TableModule module, FilteringCondition filteringCondition )
			: base( ( TableModule< IKey, IEntity, IDataTransferObject > )module, filteringCondition )
		{
		}
	}

	/// <summary>
	/// テーブルモジュールの列挙可能クラスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	/// <typeparam name="TDataTransferObject">データ変換オブジェクトの型</typeparam>
	class TableModuleEnumerable< TKey, TEntity, TDataTransferObject > : IEnumerable< TEntity >
		where TKey : class, IKey
		where TEntity : class, IEntity
		where TDataTransferObject : class, IDataTransferObject
	{
		private TableModule< TKey, TEntity, TDataTransferObject >	_module;
		private FilteringCondition									_filteringCondition;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		public TableModuleEnumerable( TableModule< TKey, TEntity, TDataTransferObject > module, FilteringCondition filteringCondition )
		{
			this._module = module;

			this._filteringCondition = filteringCondition;
		}

		#region IEnumerable<TEntity> メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		public IEnumerator< TEntity > GetEnumerator()
		{
			return( new TableModuleEnumerator< TKey, TEntity, TDataTransferObject >( this._module, this._filteringCondition ) );
		}

		#endregion

		#region IEnumerable メンバ

		/// <summary>
		/// コレクションを反復処理する列挙子を取得します。
		/// </summary>
		/// <returns>列挙子</returns>
		IEnumerator IEnumerable.GetEnumerator()
		{
			return( this.GetEnumerator() );
		}

		#endregion
	}
}
