using System;
using System.Collections.Generic;
using System.Data;
using NDac.Keys;

namespace NDac.Modules.Entities
{
	/// <summary>
	/// エンティティのインターフェイスを表します。
	/// </summary>
	public interface IEntity : IEquatable< IEntity >
	{
		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="columnIndex">列のインデックス</param>
		/// <returns>データ</returns>
		object this[ int columnIndex ]{ get; }

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="columnName">列名</param>
		/// <returns>データ</returns>
		object this[ string columnName ]{ get; }

		/// <summary>
		/// 列名の一覧を表します。
		/// </summary>
		List< string > FieldNames{ get; }

		/// <summary>
		/// エンティティの内容が削除されているか否かを判定します。
		/// </summary>
		/// <returns>内容が削除されている場合trueが返ります。</returns>
		bool IsDeleted();

		/// <summary>
		/// エンティティのデータが未更新かを判定します。データが更新されている場合はfalseが返ります。それ以外の全ての状態はtrueが返ります。
		/// </summary>
		/// <returns>未更新(新規，変更，削除)の場合trueが返ります。</returns>
		bool IsDirty();

		/// <summary>
		/// エンティティの内容が変更されているか否かを判定します。
		/// </summary>
		/// <returns>内容が変更されている場合trueが返ります。</returns>
		bool IsModified();

		/// <summary>
		/// エンティティが新規作成されたデータかを判定します。
		/// </summary>
		/// <returns>新規作成されたデータの場合trueが返ります。</returns>
		bool IsNew();

		/// <summary>
		/// データ行を表します。
		/// </summary>
		DataRow Row{ get; }

		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void Copy( IEntity source );

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		IKey GetPrimaryKey();
	}

	/// <summary>
	/// エンティティのインターフェイスを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	public interface IEntity< TKey > : IEntity, IEquatable< IEntity< TKey > >
		where TKey : class, IKey
	{
		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void Copy( IEntity< TKey > source );

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		new TKey GetPrimaryKey();
	}
}
