using System;
using System.Collections.Generic;
using System.Data;
using NDac.Keys;

namespace NDac.Modules.Entities
{
	/// <summary>
	/// エンティティの抽象クラスを表します。
	/// </summary>
	public abstract partial class Entity : IEntity< IKey >
	{
		protected DataRow _row;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="row">テーブル行</param>
		public Entity( DataRow row )
		{
			this._row = row;
		}

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <typeparam name="T">キーの型</typeparam>
		/// <returns>プライマリキー</returns>
		protected abstract T ToPrimaryKey< T >() where T : class, IKey;

		/// <summary>
		/// ハッシュコードを取得します。
		/// </summary>
		/// <returns>ハッシュコード</returns>
		public override int GetHashCode()
		{
			return( EntityValueVerifier.GetHashCode( this ) );
		}

		/// <summary>
		/// ==演算子を表します。
		/// </summary>
		/// <param name="leftOperand">左辺オペランド</param>
		/// <param name="rightOperand">右辺オペランド</param>
		/// <returns>値が等しい場合trueを返します。</returns>
		public static bool operator ==( Entity leftOperand, Entity rightOperand )
		{
			return( EntityValueVerifier.IsEqualValues( leftOperand, rightOperand ) );
		}

		/// <summary>
		/// !=演算子を表します。
		/// </summary>
		/// <param name="leftOperand">左辺オペランド</param>
		/// <param name="rightOperand">右辺オペランド</param>
		/// <returns>値が等しくない場合trueを返します。</returns>
		public static bool operator !=( Entity leftOperand, Entity rightOperand )
		{
			return( leftOperand == rightOperand ? false : true );
		}

		/// <summary>
		/// インスタンスが等しいかどうかを判断します。
		/// </summary>
		/// <param name="obj">比較対象のインスタンス</param>
		/// <returns>インスタンスが等しい場合trueを返します。</returns>
		public override bool Equals( object obj )
		{
			return( EntityValueVerifier.IsEqualValues( this, obj ) );
		}

		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		protected void Copy( Entity source )
		{
			for( int i = 0; i < source._row.ItemArray.Length; i++ )
			{
				this._row[ i ] = source._row[ i ];
			}
		}

		#region IEntity<IKey> メンバ

		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void IEntity< IKey >.Copy( IEntity< IKey > source )
		{
			this.Copy( ( Entity )source );
		}

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		public IKey GetPrimaryKey()
		{
			return( this.ToPrimaryKey< IKey >() );
		}

		#endregion

		#region IEntity メンバ

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="columnIndex">列のインデックス</param>
		/// <returns>データ</returns>
		public object this[ int columnIndex ]
		{
			get
			{
				return( this._row[ columnIndex ] );
			}
		}

		/// <summary>
		/// インデクサ
		/// </summary>
		/// <param name="columnName">列名</param>
		/// <returns>データ</returns>
		public object this[ string columnName ]
		{
			get
			{
				return( this._row[ columnName ] );
			}
		}

		/// <summary>
		/// 列名の一覧を表します。
		/// </summary>
		List< string > IEntity.FieldNames
		{
			get
			{
				List< string > fieldNames = new List< string >();

				foreach( DataColumn column in this._row.Table.Columns )
				{
					fieldNames.Add( column.ColumnName );
				}

				return( fieldNames );
			}
		}

		/// <summary>
		/// エンティティの内容が削除されているか否かを判定します。
		/// </summary>
		/// <returns>内容が削除されている場合trueが返ります。</returns>
		public bool IsDeleted()
		{
			return( ( this._row.RowState == DataRowState.Deleted ) || ( this._row.RowState == DataRowState.Detached ) ? true : false );
		}

		/// <summary>
		/// エンティティのデータが未更新かを判定します。データが更新されている場合はfalseが返ります。それ以外の全ての状態はtrueが返ります。
		/// </summary>
		/// <returns>未更新(新規，変更，削除)の場合trueが返ります。</returns>
		public bool IsDirty()
		{
			return( this._row.RowState != DataRowState.Unchanged ? true : false );
		}

		/// <summary>
		/// エンティティの内容が変更されているか否かを判定します。
		/// </summary>
		/// <returns>内容が変更されている場合trueが返ります。</returns>
		public bool IsModified()
		{
			return( this._row.RowState == DataRowState.Modified ? true : false );
		}

		/// <summary>
		/// エンティティが新規作成されたデータかを判定します。
		/// </summary>
		/// <returns>新規作成されたデータの場合trueが返ります。</returns>
		public bool IsNew()
		{
			return( this._row.RowState == DataRowState.Added ? true : false );
		}

		/// <summary>
		/// データ行を表します。
		/// </summary>
		DataRow IEntity.Row
		{
			get
			{
				return( this._row );
			}
		}

		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void IEntity.Copy( IEntity source )
		{
			this.Copy( ( Entity )source );
		}

		#endregion

		#region IEquatable<IEntity> メンバ

		/// <summary>
		/// 指定したインスタンスが現在のインスタンスと等しいかどうかを判断します。
		/// </summary>
		/// <param name="other">比較対象のインスタンス</param>
		/// <returns>指定したインスタンスが現在のインスタンスと等しい場合trueを返します。</returns>
		public bool Equals( IEntity other )
		{
			return( EntityValueVerifier.IsEqualValues( this, other ) );
		}

		#endregion

		#region IEquatable<IEntity<IKey>> メンバ

		/// <summary>
		/// 指定したインスタンスが現在のインスタンスと等しいかどうかを判断します。
		/// </summary>
		/// <param name="other">比較対象のインスタンス</param>
		/// <returns>指定したインスタンスが現在のインスタンスと等しい場合trueを返します。</returns>
		public bool Equals( IEntity< IKey > other )
		{
			return( EntityValueVerifier.IsEqualValues( this, other ) );
		}

		#endregion
	}

	/// <summary>
	/// エンティティの抽象クラスを表します。
	/// </summary>
	public abstract partial class Entity< TKey > : Entity, IEntity< TKey >
		where TKey : class, IKey
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="row">テーブル行</param>
		public Entity( DataRow row ) : base( row )
		{
		}

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <typeparam name="T">キーの型</typeparam>
		/// <returns>プライマリキー</returns>
		protected override T ToPrimaryKey< T >()
		{
			IKey primaryKey = this.ToPrimaryKey();

			return( ( T )primaryKey );
		}

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		protected abstract TKey ToPrimaryKey();

		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		protected void Copy( Entity< TKey > source )
		{
			for( int i = 0; i < source._row.ItemArray.Length; i++ )
			{
				this._row[ i ] = source._row[ i ];
			}
		}

		#region IEntity<TKey> メンバ

		/// <summary>
		/// エンティティの内容をコピーします。この操作は値のみのコピーです。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		void IEntity< TKey >.Copy( IEntity< TKey > source )
		{
			this.Copy( ( Entity< TKey > )source );
		}

		/// <summary>
		/// プライマリキーを取得します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		public new TKey GetPrimaryKey()
		{
			return( this.ToPrimaryKey< TKey >() );
		}

		#endregion

		#region IEquatable<IEntity<TKey>> メンバ

		/// <summary>
		/// 指定したインスタンスが現在のインスタンスと等しいかどうかを判断します。
		/// </summary>
		/// <param name="other">比較対象のインスタンス</param>
		/// <returns>指定したインスタンスが現在のインスタンスと等しい場合trueを返します。</returns>
		public bool Equals( IEntity< TKey > other )
		{
			return( EntityValueVerifier.IsEqualValues( this, other ) );
		}

		#endregion
	}
}
