using System;
using NDac.Datas;
using NDac.Gateways;
using NDac.Keys;
using NDac.Modules;
using NDac.Modules.Attributes;
using NDac.Modules.Entities;

namespace NDac.Modules.DataTransferObjects
{
	/// <summary>
	/// Bookテーブルのデータ変換オブジェクトを表します。
	/// </summary>
	[ Serializable ]
	public partial class BookDTO : DataTransferObject< BookKey, BookEntity >
	{
		private long _id;
		private string _title;
		private long _isbn;
		private int _price;
		private int _authorId;
		private int _publisherId;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public BookDTO()
			: base()
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="entity">エンティティ</param>
		public BookDTO( BookEntity entity )
			: base( entity )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public BookDTO( long primaryKey )
		{
			this._id = primaryKey;
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public BookDTO( BookKey primaryKey )
		{
			this._id = primaryKey.Id;
		}

		/// <summary>
		/// Idを表します。
		/// </summary>
		[ PrimaryKey( "Id", typeof( long ) ) ]
		public long Id
		{
			get
			{
				return( this._id );
			}
			set
			{
				this._id = value;
			}
		}

		/// <summary>
		/// Titleを表します。
		/// </summary>
		public string Title
		{
			get
			{
				return( StringCutDowner.Variable( this._title, 50 ) );
			}
			set
			{
				this._title = StringCutDowner.Variable( value, 50 );
			}
		}

		/// <summary>
		/// Isbnを表します。
		/// </summary>
		public long Isbn
		{
			get
			{
				return( this._isbn );
			}
			set
			{
				this._isbn = value;
			}
		}

		/// <summary>
		/// Priceを表します。
		/// </summary>
		public int Price
		{
			get
			{
				return( this._price );
			}
			set
			{
				this._price = value;
			}
		}

		/// <summary>
		/// AuthorIdを表します。
		/// </summary>
		public int AuthorId
		{
			get
			{
				return( this._authorId );
			}
			set
			{
				this._authorId = value;
			}
		}

		/// <summary>
		/// PublisherIdを表します。
		/// </summary>
		public int PublisherId
		{
			get
			{
				return( this._publisherId );
			}
			set
			{
				this._publisherId = value;
			}
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		protected override void Copy( DataTransferObject< BookKey, BookEntity > source )
		{
			this.Copy( ( BookDTO )source );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		public void Copy( BookDTO source )
		{
			this.Id = source.Id;
			this.Title = source.Title;
			this.Isbn = source.Isbn;
			this.Price = source.Price;
			this.AuthorId = source.AuthorId;
			this.PublisherId = source.PublisherId;
		}

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		public override void Copy( BookEntity source )
		{
			this.FillValue( source );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティにコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		/// <param name="destination">コピー先のエンティティ</param>
		public static void Copy( BookDTO source, BookEntity destination )
		{
			source.FillTo( destination );
		}

		/// <summary>
		/// エンティティの内容をデータ変換オブジェクトにコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		/// <param name="destination">コピー先のデータ変換オブジェクト</param>
		public static void Copy( BookEntity source, BookDTO destination )
		{
			destination.FillValue( source );
		}

		/// <summary>
		/// エンティティより値を充填します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillValue( IEntity entity )
		{
			this.FillValue( ( BookEntity )entity );
		}

		/// <summary>
		/// エンティティより値を充填します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillValue( BookEntity entity )
		{
			this._id = entity.Id;
			this._title = entity.Title;
			this._isbn = entity.Isbn;
			this._price = entity.Price;
			this._authorId = entity.AuthorId;
			this._publisherId = entity.PublisherId;
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillTo( IEntity entity )
		{
			this.FillTo( ( BookEntity )entity );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillTo( BookEntity entity )
		{
			entity.Id = this._id;
			entity.Title = this._title;
			entity.Isbn = this._isbn;
			entity.Price = this._price;
			entity.AuthorId = this._authorId;
			entity.PublisherId = this._publisherId;
		}

		/// <summary>
		/// プライマリキーでデータを充填されたテーブルモジュールを取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>テーブルモジュール</returns>
		protected override ITableModule LoadModule( IKey primaryKey )
		{
			return( this.LoadModule( ( BookKey )primaryKey ) );
		}

		/// <summary>
		/// プライマリキーでデータを充填されたテーブルモジュールを取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>テーブルモジュール</returns>
		protected override ITableModule LoadModule( BookKey primaryKey )
		{
			DataSetHolder holder = new DataSetHolder();

			BookGateway gateway = new BookGateway( holder );

			BookModule module = new BookModule( holder );

			gateway.LoadBy( primaryKey );

			return( module );
		}

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		protected override BookKey ToPrimaryKey()
		{
			return( new BookKey( this.Id ) );
		}
	}

	/// <summary>
	/// Bookテーブルエンティティの拡張メソッド定義クラスを表します。
	/// </summary>
	public static class BookEntityExtension
	{
		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		public static void Copy( this BookEntity entity, BookDTO source )
		{
			BookDTO.Copy( source, entity );
		}

		/// <summary>
		/// データ変換オブジェクトに変換します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <returns>データ変換オブジェクト</returns>
		public static BookDTO ToDataTransferObject( this BookEntity entity )
		{
			return( new BookDTO( entity ) );
		}
	}
}
