using System.Collections.Generic;
using NDac.Keys;
using NDac.Modules.Entities;

namespace NDac.IdentityMappings
{
	/// <summary>
	/// 一意マッピングを表します。
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	class IdentityMap< TKey, TEntity > : Dictionary< TKey, TEntity >
		where TKey : IKey
		where TEntity : IEntity
	{
		/// <summary>
		/// 一意マッピングにエンティティを追加した後に呼ばれるイベントのハンドラを表します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <param name="entity">エンティティ</param>
		public delegate void AddedHandler( TKey key, TEntity entity );

		/// <summary>
		/// 一意マッピングから全てのエンティティをクリアした後に呼ばれるイベントのハンドラを表します。
		/// </summary>
		public delegate void ClearedHandler();

		/// <summary>
		/// 一意マッピングにエンティティを追加した後に呼ばれるイベントを表します。
		/// </summary>
		public event AddedHandler Added;

		/// <summary>
		/// 一意マッピングから全てのエンティティをクリアした後に呼ばれるイベントを表します。
		/// </summary>
		public event ClearedHandler Cleared;

		/// <summary>
		/// 一意マッピングにエンティティを追加します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <param name="entity">エンティティ</param>
		public new void Add( TKey key, TEntity entity )
		{
			base.Add( key, entity );

			if( this.Added != null )
			{
				this.Added( key, entity );
			}
		}

		/// <summary>
		/// 一意マッピングから全てのエンティティをクリアします。
		/// </summary>
		public new void Clear()
		{
			base.Clear();

			if( this.Cleared != null )
			{
				this.Cleared();
			}
		}
	}
}
