using System;
using NDac.Keys;
using NDac.Modules.Entities;
using NDac.RemoteFacades;

namespace NDac.IdentityMappings
{
	/// <summary>
	/// Bookテーブルのファインダを表します。
	/// </summary>
	public partial class BookFinder
	{
		private static BookFinder _instance;
		private IdentityMap< BookKey, BookEntity > _identityMap = new IdentityMap< BookKey, BookEntity >();

		/// <summary>
		/// コンストラクタ
		/// </summary>
		private BookFinder()
		{
			this.Constructed();
		}

		/// <summary>
		/// インスタンスを表します。
		/// </summary>
		public static BookFinder Instance
		{
			get
			{
				if( _instance == null )
				{
					_instance = new BookFinder();
				}

				return( _instance );
			}
		}

		/// <summary>
		/// キー情報に一致するBookテーブルのエンティティを取得します。
		/// <para>まず一意マッピングを検索し、一意マッピングに存在する値を返します。</para>
		/// <para>一意マッピングに存在しない場合はFurnaceModelテーブルを検索し、Bookテーブルに存在する値を返します。</para>
		/// <para>一意マッピング，Bookテーブルのいずれにも存在しない場合はnullを返します。</para>
		/// </summary>
		/// <param name="id">Id</param>
		/// <returns>Bookテーブルのエンティティ</returns>
		public BookEntity Find( long id )
		{
			return( this.Find( new BookKey( id ) ) );
		}

		/// <summary>
		/// キー情報に一致するBookテーブルのエンティティを取得します。
		/// <para>まず一意マッピングを検索し、一意マッピングに存在する値を返します。</para>
		/// <para>一意マッピングに存在しない場合はBookテーブルを検索し、Bookテーブルに存在する値を返します。</para>
		/// <para>一意マッピング，Bookテーブルのいずれにも存在しない場合はnullを返します。</para>
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>Bookテーブルのエンティティ</returns>
		public BookEntity Find( BookKey primaryKey )
		{
			if( !this._identityMap.ContainsKey( primaryKey ) )
			{
				BookFacade facade = new BookFacade();

				facade.LoadBy( primaryKey );

				if( 0 < facade.Count )
				{
					this._identityMap.Add( primaryKey, facade[ primaryKey ] );
				}
				else
				{
					this._identityMap.Add( primaryKey, null );
				}
			}

			this.Finded( primaryKey, this._identityMap[ primaryKey ] );

			return( this._identityMap[ primaryKey ] );
		}

		/// <summary>
		/// テーブルより全データを充填し、一意マッピングに登録します。
		/// </summary>
		public void LoadAll()
		{
			BookFacade facade = new BookFacade();

			facade.LoadAll();

			if( this._identityMap.Count != facade.Count )
			{
				foreach( BookEntity entity in facade )
				{
					if( !this._identityMap.ContainsKey( entity.GetPrimaryKey() ) )
					{
						this._identityMap.Add( entity.GetPrimaryKey(), entity );
					}
				}
			}
		}

		/// <summary>
		/// 一意マッピングに存在する全ての値をクリアします。
		/// </summary>
		public void Clear()
		{
			this._identityMap.Clear();
		}

		/// <summary>
		/// コンストラクタが呼び出された後にコールされます。
		/// </summary>
		partial void Constructed();

		/// <summary>
		/// Find終了処理を実行します。
		/// </summary>
		/// <param name="key">キー</param>
		/// <param name="entity">Find処理結果のエンティティ</param>
		partial void Finded( BookKey key, BookEntity entity );
	}
}
