using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;

namespace NDac.Datas
{
	/// <summary>
	/// データアダプタホルダーを表します。
	/// </summary>
	class DataAdapterHolder
	{
		private string _tableName;
		private SqlDataAdapter _adapter;
		private bool? _isMustDbCommand;
		private DataTable _schemaTable;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="adapter">データアダプタ</param>
		public DataAdapterHolder( string tableName, SqlDataAdapter adapter )
		{
			this._tableName			= tableName;

			this._adapter			= adapter;

			this._isMustDbCommand	= null;

			this._schemaTable		= null;
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="tableName">テーブル名</param>
		/// <param name="adapter">データアダプタ</param>
		/// <param name="schemaTable">スキーマ情報の充填されたデータテーブル</param>
		public DataAdapterHolder( string tableName, SqlDataAdapter adapter, DataTable schemaTable )
		{
			this._tableName			= tableName;

			this._adapter			= adapter;

			this._isMustDbCommand	= null;

			this._schemaTable		= schemaTable;
		}

		/// <summary>
		/// テーブル名を表します。
		/// </summary>
		public string TableName
		{
			get
			{
				return( this._tableName );
			}
		}

		/// <summary>
		/// データアダプタを表します。
		/// </summary>
		public SqlDataAdapter DataAdapter
		{
			get
			{
				return( this._adapter );
			}
		}

		/// <summary>
		/// DBコマンドを必要とするかを表します。
		/// </summary>
		public bool IsMustDbCommand
		{
			get
			{
				if( !this._isMustDbCommand.HasValue )
				{
					this._isMustDbCommand = this.HasMultipleTable( this._adapter.SelectCommand.CommandText.ToUpper() );
				}

				return( this._isMustDbCommand.Value );
			}
		}

		/// <summary>
		/// スキーマ情報の充填されたデータテーブルを表します。
		/// </summary>
		public DataTable SchemaTable
		{
			get
			{
				return( this._schemaTable );
			}
			set
			{
				this._schemaTable = value;
			}
		}

		/// <summary>
		/// クエリに複数テーブルを検索しているかを判定します。
		/// </summary>
		/// <param name="query">クエリ</param>
		/// <returns>クエリに複数テーブルを検索している場合trueが返ります。</returns>
		private bool HasMultipleTable( string query )
		{
			List< int > nextIndexes = new List< int >();

			int fromIndex = query.IndexOf( "FROM", 0, query.Length );

			int subIndex = query.IndexOf( "SELECT", fromIndex, query.Length - fromIndex );

			int scanCount = ( 0 < subIndex ) ? ( subIndex - fromIndex ) : ( query.Length - fromIndex );

			nextIndexes.Add( query.IndexOf( "WHERE", fromIndex, scanCount ) );

			nextIndexes.Add( query.IndexOf( "ORDER", fromIndex, scanCount ) );

			nextIndexes.Add( query.IndexOf( "GROUP", fromIndex, scanCount ) );

			nextIndexes.Add( query.Length );

			nextIndexes.RemoveAll( ( x ) => ( x == -1 ) );

			nextIndexes.Sort( ( x, y ) => x.CompareTo( y ) );

			if( 0 < query.IndexOf( ",", fromIndex, nextIndexes[ 0 ] - fromIndex ) )
			{
				return( true );
			}

			if( 0 < query.IndexOf( "JOIN", fromIndex, nextIndexes[ 0 ] - fromIndex ) )
			{
				return( true );
			}

			return( false );
		}
	}
}
