/* $Id: PreferenceManager.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.pref;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;

/**
 * ユーザ環境設定を管理するクラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class PreferenceManager
{
	/** OS依存のファイル区切り */
	public static final String SEP = System.getProperty("file.separator");
	/** ユーザ環境設定ファイル名 */
	private static final String PREF_FILE = SEP + "preferences.xml";
	/** PreferenceManagerのシングルトン */
	private static PreferenceManager singleton;

	/** アプリケーションデータパス */
	private String appDataPath;
	/** ユーザ環境設定ファイルパス */
	private String prefPath;
	/** エンジン情報 */
	private ArrayList<EngineInfo> engineInfos = new ArrayList<EngineInfo>();
	
	/**
	 * PreferenceManagerの生成と初期化を行う。
	 *
	 * @throws Exception 任意の例外
	 */
	public static void create() throws Exception
	{
		singleton = new PreferenceManager();
		singleton.init();
	}

	/**
	 * PreferenceManagerのインスタンスを返す。
	 *
	 * @return PreferenceManager
	 */
	public static PreferenceManager getInstance()
	{
		return singleton;
	}
	
	/**
	 * PreferenceManagerを生成する。
	 */
	private PreferenceManager()
	{
	}
	
	/**
	 * PreferenceManagerを初期化する。
	 *
	 * @throws Exception 任意の例外
	 */
	private void init() throws Exception
	{
		initPath();
		loadPreferences();
	}
	
	/**
	 * アプリケーションデータ格納用のディレクトリのパス名を返す。
	 * 場所は、ユーザのホームディレクトリ配下の以下の場所とする。
	 * <pre>
	 * 
	 * Windows
	 * ${HOME}/Application Data/Nautica/WorkflowAdmin
	 * 
	 * Linux
	 * ${HOME}/.nautica/WorkflowAdmin
	 * 
	 * </pre>
	 * これらの場所がない場合は、カレントを返す。
	 * @return パス名
	 */
	public String getApplicationDataPath()
	{
		return appDataPath;
	}
	
	/**
	 * 現在のエンジン情報一覧を返す。
	 *
	 * @return エンジン情報一覧
	 */
	public List<EngineInfo> getAllEngineInfo()
	{
		return engineInfos;
	}

	/**
	 * エンジン情報名に対応したエンジン情報を返す。
	 * なければnullを返す。
	 *
	 * @param engineName エンジン情報名
	 * @return エンジン情報
	 */
	public EngineInfo getEngineInfo(String engineName)
	{
		if (engineName == null || engineName.trim().equals("")) return null;

		for (EngineInfo info : engineInfos) {
			String name = info.getName();
			if (name.equals(engineName)) {
				return info;
			}
		}
		return null;
	}

	/**
	 * インデクスに対応したエンジン情報を返す。
	 * なければnullを返す。
	 *
	 * @param index インデクス
	 * @return エンジン情報
	 */
	public EngineInfo getEngineInfo(int index)
	{
		if (index < 0 || index >= engineInfos.size()) return null;

		EngineInfo info = (EngineInfo)engineInfos.get(index);
		return info;
	}
	
	/**
	 * エンジン情報を追加する。
	 * 追加に成功した場合は、trueを返す。
	 *
	 * @param info エンジン情報
	 * @return 実行結果
	 */
	public boolean addEngineInfo(EngineInfo info)
	{
		if (info == null) return false;
		String name = info.getName();
		EngineInfo ei = getEngineInfo(name);
		if (ei != null) return false;

		engineInfos.add(info);
		return true;
	}
	
	/**
	 * エンジン情報を更新する。
	 * 更新に成功した場合は、trueを返す。
	 *
	 * @param info エンジン情報
	 * @return 実行結果
	 */
	public boolean  updateEngineInfo(EngineInfo info)
	{
		if (info == null) return false;
		String name = info.getName();
		EngineInfo ei = getEngineInfo(name);
		if (ei == null) return false;

		int ix = engineInfos.indexOf(ei);
		engineInfos.set(ix, info);
		return true;
	}
	
	/**
	 * インデクスに対応したエンジン情報を更新する。
	 * 更新に成功した場合は、trueを返す。
	 *
	 * @param index インデクス
	 * @param info エンジン情報
	 * @return 実行結果
	 */
	public boolean  updateEngineInfo(int index, EngineInfo info)
	{
		if (index < 0 || index >= engineInfos.size()) return false;
		if (info == null) return false;
		String name = info.getName();
		EngineInfo ei1 = getEngineInfo(name);
		EngineInfo ei2 = getEngineInfo(index);
		if (ei1 != null) {
			if (ei1 != ei2) return false;
		}

		engineInfos.set(index, info);
		return true;
	}
	
	/**
	 * エンジン情報を削除する。
	 * 削除に成功した場合は、trueを返す。
	 *
	 * @param engineName エンジン情報名
	 * @return 実行結果
	 */
	public boolean removeEngineInfo(String engineName)
	{
		EngineInfo ei = getEngineInfo(engineName);
		if (ei == null) return false;

		int ix = engineInfos.indexOf(ei);
		engineInfos.remove(ix);
		return true;
	}
	
	/**
	 * インデクスに対応したエンジン情報を削除する。
	 * 削除に成功した場合は、trueを返す。
	 *
	 * @param index インデクス
	 * @return 実行結果
	 */
	public boolean removeEngineInfo(int index)
	{
		if (index < 0 || index >= engineInfos.size()) return false;
		engineInfos.remove(index);
		return true;
	}
	
	/**
	 * 管理されているエンジン情報の個数を返す。
	 *
	 * @return エンジン情報の個数
	 */
	public int getEngineInfoCount()
	{
		return engineInfos.size();
	}
	
	/**
	 * 現在のユーザ環境設定をファイルからロードする。
	 */
	public void load()
	throws IOException, XmlException
	{
		loadPreferences();
	}
	
	/**
	 * 現在のユーザ環境設定をファイルに保存する。
	 */
	public void save()
	throws IOException, XmlException
	{
		savePreferences();
	}
	
	/**
	 * ユーザ環境設定ロード処理。
	 *
	 * @throws IOException IO例外
	 * @throws XmlException XML例外
	 */
	private void loadPreferences()
	throws IOException, XmlException
	{
		engineInfos.clear();

		File f = new File(prefPath);
		if (f.exists()) {
			InputStream in = new FileInputStream(f);
			XmlOptions opts = new XmlOptions();
			opts.setCharacterEncoding("utf-8");
			AdminPreferencesDocument doc = AdminPreferencesDocument.Factory.parse(in, opts);
			in.close();
			
			AdminPreferencesDocument.AdminPreferences prefs =
				doc.getAdminPreferences();
			if (prefs != null) {
				EngineInformationsDocument.EngineInformations einfos = prefs.getEngineInformations();
				loadEngineInformations(einfos);
			}
		}
	}
	
	/**
	 * XMLオブジェクトからエンジン情報をロードする。
	 *
	 * @param einfos XMLオブジェクト
	 */
	private void loadEngineInformations(EngineInformationsDocument.EngineInformations einfos)
	{
		if (einfos == null) return;
		
		EngineInformationDocument.EngineInformation[] einfo =
			einfos.getEngineInformationArray();
		if (einfo == null) return;
		
		for (int i = 0; i < einfo.length; i++) {
			loadEngineInformation(einfo[i]);
		}
	}
	
	/**
	 * XMLオブジェクトからひとつのエンジン情報をロードする。
	 *
	 * @param einfo XMLオブジェクト
	 */
	private void loadEngineInformation(EngineInformationDocument.EngineInformation einfo)
	{
		if (einfo == null) return;
		
		EngineInfo info = new EngineInfo(einfo);
		engineInfos.add(info);
	}
	
	
	/**
	 * ユーザ環境設定保存処理。
	 *
	 * @throws IOException IO例外
	 * @throws XmlException XML例外
	 */
	private void savePreferences()
	throws IOException, XmlException
	{
		AdminPreferencesDocument doc = AdminPreferencesDocument.Factory.newInstance();
		AdminPreferencesDocument.AdminPreferences prefs =
			doc.getAdminPreferences();
		if (prefs == null) {
			prefs = doc.addNewAdminPreferences();
		}
		
		EngineInformationsDocument.EngineInformations einfos = prefs.getEngineInformations();
		if (einfos == null) {
			einfos = prefs.addNewEngineInformations();
			saveEngineInformations(einfos);
		}
		
		OutputStream out = new FileOutputStream(prefPath);
		XmlOptions opts = new XmlOptions();
		opts.setSavePrettyPrint();
		opts.setSavePrettyPrintIndent(2);
		opts.setCharacterEncoding("utf-8");
		doc.save(out, opts);
		out.close();
	}
	
	/**
	 * エンジン情報をXMLに格納する。
	 *
	 * @param einfos XMLオブジェクト
	 */
	private void saveEngineInformations(EngineInformationsDocument.EngineInformations einfos)
	{
		for (EngineInfo info : engineInfos) {
			if (info != null) {
				EngineInformationDocument.EngineInformation einfo =
					einfos.addNewEngineInformation();
				einfo.setName(info.getName());
				einfo.setUrl(info.getURL());
				einfo.setScope(info.getScope());
				einfo.setUser(info.getUser());
				einfo.setPassword(info.getPassword());
				einfo.setAdminKey(info.getAdminKey());
			}
		}
	}
	
	/**
	 * アプリケーションデータ格納用のディレクトリのパス名を初期化する。
	 * 場所は、ユーザのホームディレクトリ配下の以下の場所とする。
	 * <pre>
	 * 
	 * [Windows]
	 * $${HOME}/Application Data/Nautica/WorkflowAdmin
	 * 
	 * [Linux]
	 * $${HOME}/.nautica/WorkflowAdmin
	 * 
	 * </pre>
	 * これらの場所がない場合は、カレントをパスとする。
	 */
	private void initPath()
	{
		String path = System.getProperty("user.home", ".");
		String osname = System.getProperty("os.name");

		//OSごとにパスを生成
		if (osname.indexOf("Windows") >= 0) {
			// Windows
			path += SEP + "Application Data" + SEP + "Nautica" + SEP + "WorkflowAdmin";
		} else {
			// その他の環境(Linux, Solaris, Mac OS X...these are un*x os.)
			path += SEP + ".nautica" + SEP + "WorkflowAdmin";
		}
		
		//パスの存在チェックと作成
		File dir = new File(path);
		if (dir.exists() == false) {
			boolean created = dir.mkdirs();
			if (created == false) {
				//生成できなかった場合は、カレントを返す。
				path = ".";
			}
		}
		
		appDataPath = path;
		prefPath = path + PREF_FILE;
	}
}
