/* $Id: EngineInfoComponent.java,v 1.2 2007/10/24 09:53:00 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.pref;

import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;
import javax.swing.border.Border;
import java.util.regex.Pattern;

import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.DialogComponent;

/**
 * エンジン情報編集用画面コンポーネント
 * 
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class EngineInfoComponent
extends DialogComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	private static final String INITIAL_URL = "ejbd://127.0.0.1:4201";
	
	/** エンジン情報名入力エリア */
	private JTextField nameField;
	/** URL入力エリア */
	private JTextField urlField;
	/** スコープ入力エリア */
	private JTextField scopeField;
	/** ユーザ名入力エリア */
	private JTextField userField;
	/** パスワード入力エリア */
	private JPasswordField pass1Field;
	/** パスワード入力エリア(確認用) */
	private JPasswordField pass2Field;
	/** 管理者キー入力エリア */
	private JPasswordField key1Field;
	/** 管理者キー入力エリア(確認用) */
	private JPasswordField key2Field;

	/** 編集対象となっているエンジン情報 */
	private EngineInfo engineInfo;
	/** 新規作成かどうかのフラグ */
	private boolean isNew;
	
	/**
	 * エンジン情報編集用画面コンポーネントを生成する。
	 */
	public EngineInfoComponent()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		JPanel panel = createBasicInfoPanel(rm);
		addTab(rm.getResource("EngineInfoComponent.tab.0"), panel);

		validate();
	}
	
	/**
	 * 基本情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createBasicInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 260);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("EngineInfoComponent.label.0"));
		JLabel l1 = new JLabel(rm.getResource("EngineInfoComponent.label.1"));
		JLabel l2 = new JLabel(rm.getResource("EngineInfoComponent.label.2"));
		JLabel l3 = new JLabel(rm.getResource("EngineInfoComponent.label.3"));
		JLabel l4 = new JLabel(rm.getResource("EngineInfoComponent.label.4"));
		JLabel l5 = new JLabel(rm.getResource("EngineInfoComponent.label.5"));
		JLabel l6 = new JLabel(rm.getResource("EngineInfoComponent.label.6"));
		JLabel l7 = new JLabel(rm.getResource("EngineInfoComponent.label.7"));

		nameField = new JTextField();
		nameField.setDocument(new ByteLimitedDocument(16));

		urlField = new JTextField(INITIAL_URL);
		urlField.setDocument(new ByteLimitedDocument(64));

		scopeField = new JTextField();
		scopeField.setDocument(new ByteLimitedDocument(32));

		userField = new JTextField();
		userField.setDocument(new ByteLimitedDocument(32));

		pass1Field = new JPasswordField();
		pass1Field.setEchoChar('*');
		pass1Field.setDocument(new ByteLimitedDocument(32));

		pass2Field = new JPasswordField();
		pass2Field.setEchoChar('*');
		pass2Field.setDocument(new ByteLimitedDocument(32));

		key1Field = new JPasswordField();
		key1Field.setEchoChar('*');
		key1Field.setDocument(new ByteLimitedDocument(32));

		key2Field = new JPasswordField();
		key2Field.setEchoChar('*');
		key2Field.setDocument(new ByteLimitedDocument(32));

		panel.add(locate(l0,          10,  10, 100, 25));
		panel.add(locate(l1,          10,  40, 100, 25));
		panel.add(locate(l2,          10,  70, 100, 25));
		panel.add(locate(l3,          10, 100, 100, 25));
		panel.add(locate(l4,          10, 130, 100, 25));
		panel.add(locate(l5,          10, 160, 100, 25));
		panel.add(locate(l6,          10, 190, 100, 25));
		panel.add(locate(l7,          10, 220, 100, 25));
		panel.add(locate(nameField,  110,  10, 190, 25));
		panel.add(locate(urlField,   110,  40, 190, 25));
		panel.add(locate(scopeField, 110,  70, 190, 25));
		panel.add(locate(userField,  110, 100, 190, 25));
		panel.add(locate(pass1Field, 110, 130, 190, 25));
		panel.add(locate(pass2Field, 110, 160, 190, 25));
		panel.add(locate(key1Field,  110, 190, 190, 25));
		panel.add(locate(key2Field,  110, 220, 190, 25));

		validate();
		
		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#init()
	 */
	public void init()
	{
		nameField.setText("");
		urlField.setText(INITIAL_URL);
		scopeField.setText("");
		userField.setText("");
		pass1Field.setText("");
		pass2Field.setText("");
		key1Field.setText("");
		key2Field.setText("");
	}

	/**
	 * エンジン情報を設定する。
	 * 
	 * @param engineInfo	エンジン情報
	 */	
	public void setEngineInfo(EngineInfo info)
	{
		init();

		if (info == null) {
			info = new EngineInfo();
			info.setURL(INITIAL_URL);
			isNew = true;
			nameField.setEditable(true);
		} else {
			isNew = false;
			nameField.setEditable(false);
		}
		engineInfo = info;
		nameField.setText(info.getName());
		urlField.setText(info.getURL());
		scopeField.setText(info.getScope());
		userField.setText(info.getUser());
		pass1Field.setText(info.getPassword());
		pass2Field.setText(info.getPassword());
		key1Field.setText(info.getAdminKey());
		key2Field.setText(info.getAdminKey());

		setSelectedIndex(0);
		repaint();
	}

	/**
	 * 設定値をコミットする。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#commit()
	 */
	public boolean commit()
	{
		if (check()) {
			engineInfo.setName(nameField.getText());
			engineInfo.setURL(urlField.getText());
			engineInfo.setScope(scopeField.getText());
			engineInfo.setUser(userField.getText());
			engineInfo.setPassword(new String(pass1Field.getPassword()));
			engineInfo.setAdminKey(new String(key1Field.getPassword()));
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * 保持されているエンジン情報を取り出す。エンジン情報がない場合は、
	 * nullを返す。
	 *
	 * @return エンジン情報
	 */
	public EngineInfo getEngineInfo()
	{
		EngineInfo ei = engineInfo;
		engineInfo = null;
		init();
		return ei;
	}

	/**
	 * 設定値をチェックする。
	 * 
	 * @return	チェック結果
	 */
	private boolean check()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		if (checkEngineName(rm) == false) return false;
		if (checkURL(rm) == false)        return false;
		if (checkScope(rm) == false)      return false;
		if (checkUser(rm) == false)       return false;
		if (checkPassword(rm) == false)   return false;
		if (checkAdminKey(rm) == false)   return false;
		return true;
	}
	
	/**
	 * エンジン情報名をチェックする。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean checkEngineName(ResourceManager rm)
	{
		String s = nameField.getText();
		String pat = "[A-Za-z0-9]*";
		String msg = null;
		if (s == null)                              msg = rm.getResource("E9005");
		else if (s.trim().equals(""))               msg = rm.getResource("E9001");
		else if (Pattern.matches(pat, s) == false) msg = rm.getResource("E9017");
		else if (s.trim().equals(EngineInfo.UNSET)) msg = rm.getResource("E9001");

		PreferenceManager pm = PreferenceManager.getInstance();
		if (isNew && pm.getEngineInfo(s) != null)   msg = rm.getResource("E9002");

		if (msg != null) {
			DialogUtilities.showError(msg);
			return false;
		}
		return true;
	}
	
	/**
	 * URLをチェックする。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean checkURL(ResourceManager rm)
	{
		String s = urlField.getText();
		String pat = "ejbd://[A-Za-z0-9_]+[\\.A-Za-z0-9_]*[A-Za-z0-9_]+:[0-9]*";
		String msg = null;
		if (s == null)                              msg = rm.getResource("E9003");
		else if (s.trim().equals(""))               msg = rm.getResource("E9003");
		else if (s.trim().equals(EngineInfo.UNSET)) msg = rm.getResource("E9003");
		else if (Pattern.matches(pat, s) == false) msg = rm.getResource("E9004");

		return isValid(msg);
	}
	
	/**
	 * スコープをチェックする。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean checkScope(ResourceManager rm)
	{
		String s = scopeField.getText();
		String pat = "[A-Za-z0-9_]*";
		String msg = null;
		if (s == null)                              msg = rm.getResource("E9005");
		else if (Pattern.matches(pat, s) == false) msg = rm.getResource("E9006");

		return isValid(msg);
	}
	
	/**
	 * ユーザIDをチェックする。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean checkUser(ResourceManager rm)
	{
		String s = userField.getText();
		String pat = "[A-Za-z_][A-Za-z0-9_]*";
		String msg = null;
		if (s == null)                              msg = rm.getResource("E9007");
		else if (s.trim().equals(""))               msg = rm.getResource("E9007");
		else if (s.trim().equals(EngineInfo.UNSET)) msg = rm.getResource("E9007");
		else if (Pattern.matches(pat, s) == false) msg = rm.getResource("E9008");

		return isValid(msg);
	}
	
	/**
	 * パスワードをチェックする。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean checkPassword(ResourceManager rm)
	{
		String s1 = new String(pass1Field.getPassword());
		String s2 = new String(pass2Field.getPassword());
		String pat = "[A-Za-z0-9_]*";
		String msg = null;
		if (s1 == null)                              msg = rm.getResource("E9009");
		else if (s1.trim().equals(""))               msg = rm.getResource("E9009");
		else if (Pattern.matches(pat, s1) == false) msg = rm.getResource("E9009");
		else if (s1.equals(s2) == false)            msg = rm.getResource("E9010");

		return isValid(msg);
	}
	
	/**
	 * 管理者キーをチェックする。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean checkAdminKey(ResourceManager rm)
	{
		String s1 = new String(key1Field.getPassword());
		String s2 = new String(key2Field.getPassword());
		String pat = "[A-Za-z_][A-Za-z0-9_]*";
		String msg = null;
		if (s1 == null)                              msg = rm.getResource("E9011");
		else if (s1.trim().equals(""))               msg = rm.getResource("E9011");
		else if (Pattern.matches(pat, s1) == false) msg = rm.getResource("E9011");
		else if (s1.equals(s2) == false)            msg = rm.getResource("E9012");
		
		return isValid(msg);
	}
	
	/**
	 * チェックエラーがなかったかどうかを返す。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean isValid(String msg)
	{
		if (msg != null) {
			DialogUtilities.showError(msg);
			return false;
		}
		return true;
	}
}
