/* $Id: EngineInfo.java,v 1.2 2007/10/29 02:26:50 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.pref;


import javax.naming.NamingException;

import jp.co.argo21.nautica.workflow.client.ConnectionInfo;
import jp.co.argo21.nautica.workflow.client.WorkflowServiceAccessor;
import jp.co.argo21.nautica.workflow.engine.RepositoryLoader;
import jp.co.argo21.nautica.workflow.util.GeronimoUtils;

/**
 * エンジン接続情報を表す。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class EngineInfo
{
	private static final String REPOSITORY_LOADER_REMOTE_IF =
		"nautica-workflow/RepositoryLoaderBean/jp.co.argo21.nautica.workflow.engine.RepositoryLoader";

	/** OS依存のファイル区切り */
	public static final String UNSET = "(unset)";

	/** エンジン情報名 */
	private String name;
	/** URL */
	private String url;
	/** スコープ */
	private String scope;
	/** ユーザ名 */
	private String user;
	/** パスワード */
	private String password;
	/** 管理者キー */
	private String adminKey;
	/** ワークフローサービスアクセッサ */
	private WorkflowServiceAccessor wsa;

	/**
	 * 空のエンジン情報を生成する。
	 */
	public EngineInfo()
	{
	}

	/**
	 * 引数に指定された情報からエンジン情報を生成する。
	 *
	 * @param name エンジン情報名
	 * @param url URL
	 * @param scope スコープ
	 * @param user ユーザ名
	 * @param password パスワード
	 * @param adminKey 管理者キー
	 */
	public EngineInfo(String name, String url, String scope, String user,
			String password, String adminKey)
	{
		this.name     = name;
		this.url      = url;
		this.scope    = scope;
		this.user     = user;
		this.password = password;
		this.adminKey = adminKey;
	}

	/**
	 * ユーザ設定ファイルの内容からエンジン情報を生成する。
	 *
	 * @param doc エンジン情報XMLドキュメント
	 */
	EngineInfo(EngineInformationDocument.EngineInformation info)
	{
		this.name     = info.getName();
		this.url      = info.getUrl();
		this.scope    = info.getScope();
		this.user     = info.getUser();
		this.password = info.getPassword();
		this.adminKey = info.getAdminKey();
	}
	
	/**
	 * nameを返す。
	 *
	 * @return nameの値
	 */
	public String getName()
	{
		if (name == null || name.trim().equals("")) return UNSET;
		return name;
	}

	/**
	 * urlを返す。
	 *
	 * @return urlの値
	 */
	public String getURL()
	{
		if (url == null || url.trim().equals("")) return UNSET;
		return url;
	}

	/**
	 * scopeを返す。
	 *
	 * @return scopeの値
	 */
	public String getScope()
	{
		if (scope == null || scope.trim().equals("")) return "";
		return scope;
	}

	/**
	 * userを返す。
	 *
	 * @return userの値
	 */
	public String getUser()
	{
		if (user == null || user.trim().equals("")) return UNSET;
		return user;
	}

	/**
	 * passwordを返す。
	 *
	 * @return passwordの値
	 */
	public String getPassword()
	{
		if (password == null || password.trim().equals("")) return "";
		return password;
	}

	/**
	 * adminKeyを返す。
	 *
	 * @return adminKeyの値
	 */
	public String getAdminKey()
	{
		if (adminKey == null || adminKey.trim().equals("")) return "";
		return adminKey;
	}

	/**
	 * nameを設定する。
	 * 
	 * @param name nameへの設定値
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * urlを設定する。
	 * 
	 * @param url urlへの設定値
	 */
	public void setURL(String url)
	{
		init();
		this.url = url;
	}

	/**
	 * scopeを設定する。
	 * 
	 * @param scope scopeへの設定値
	 */
	public void setScope(String scope)
	{
		init();
		this.scope = scope;
	}

	/**
	 * userを設定する。
	 * 
	 * @param user userへの設定値
	 */
	public void setUser(String user)
	{
		init();
		this.user = user;
	}

	/**
	 * passwordを設定する。
	 * 
	 * @param password passwordへの設定値
	 */
	public void setPassword(String password)
	{
		init();
		this.password = password;
	}
	
	/**
	 * adminKeyを設定する。
	 * 
	 * @param adminKey adminKeyへの設定値
	 */
	public void setAdminKey(String adminKey)
	{
		init();
		this.adminKey = adminKey;
	}
	
	/**
	 * エンジン情報に対応したワークフローサービスアクセッサを返す。
	 *
	 * @return ワークフローサービスアクセッサ
	 */
	public WorkflowServiceAccessor getWorkflowServiceAccessor()
	{
		if (wsa == null) {
			ConnectionInfo cinfo = getConnectionInfo();
			wsa = new WorkflowServiceAccessor(cinfo);
		}
		return wsa;
	}
	
	public RepositoryLoader getRepositoryLoader()
	{
		try {
			ConnectionInfo cinfo = getConnectionInfo();
			RepositoryLoader loader =
				(RepositoryLoader)GeronimoUtils.getEJB(
					cinfo.getEngineName(),
					REPOSITORY_LOADER_REMOTE_IF);
			return loader;
		} catch (NamingException ex) {
			return null;
		} 
	}

	/**
	 * 現在の値から、接続情報を生成して返す。
	 *
	 * @return 接続情報
	 */
	private ConnectionInfo getConnectionInfo()
	{
		ConnectionInfo cinfo = new ConnectionInfo(url, scope, user, password);
		return cinfo;
	}
	
	/**
	 * 内部状態を初期化する。
	 */
	private void init()
	{
		wsa = null;
	}
}
