/* $Id: ProcessDefinitionFilteringComponent.java,v 1.1 2007/10/19 07:29:57 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.Dimension;
import java.util.regex.Pattern;

import javax.swing.BorderFactory;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;

import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.FilteringDialogComponent;
import jp.co.argo21.nautica.workflow.filter.GroupFilter;
import jp.co.argo21.nautica.workflow.filter.NameFilter;
import jp.co.argo21.nautica.workflow.filter.StateFilter;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.ProcessDefinitionState;

/**
 * プロセス定義取得用のフィルターを設定する画面コンポーネント
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessDefinitionFilteringComponent extends FilteringDialogComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 演算子選択名 */
	private static String[] opeNames;
	/** 状態選択名 */
	private static String[] stateNames;
	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** フィルタリング文字列入力エリア */
	private JTextField filterStringField;
	/** 演算子選択 */
	private JComboBox opeBox;
	/** 状態選択 */
	private JComboBox stateBox;
	
	/**
	 * 定数の初期化
	 */
	static
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		opeNames = new String[3];
		opeNames[0] = rm.getResource("ProcessDefinitionFilteringComponent.opeName.0");
		opeNames[1] = rm.getResource("ProcessDefinitionFilteringComponent.opeName.1");
		opeNames[2] = rm.getResource("ProcessDefinitionFilteringComponent.opeName.2");

		stateNames = new String[3];
		stateNames[0] = rm.getResource("ProcessDefinitionFilteringComponent.stateName.0");
		stateNames[1] = rm.getResource("ProcessDefinitionFilteringComponent.stateName.1");
		stateNames[2] = rm.getResource("ProcessDefinitionFilteringComponent.stateName.2");
	}

	/**
	 * プロセス定義取得用フィルター設定画面を生成する。
	 */
	public ProcessDefinitionFilteringComponent()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		JPanel panel = createFilteringPanel(rm);
		addTab(rm.getResource("ProcessDefinitionFilteringComponent.tab.0"), panel);

		validate();
	}
	
	/**
	 * フィルター設定のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createFilteringPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(370, 140);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("ProcessDefinitionFilteringComponent.label.0"));
		JLabel l1 = new JLabel(rm.getResource("ProcessDefinitionFilteringComponent.label.1"));

		filterStringField = new JTextField();
		filterStringField.setDocument(new ByteLimitedDocument(64));
		opeBox = new JComboBox(opeNames);
		stateBox = new JComboBox(stateNames);

		panel.add(locate(l0,                 10,  10, 150, 25));
		panel.add(locate(opeBox,             10,  40, 150, 25));
		panel.add(locate(filterStringField, 160,  40, 200, 25));
		panel.add(locate(l1,                 10,  70, 150, 25));
		panel.add(locate(stateBox,           10, 100, 350, 25));

		validate();
		
		return panel;
	}

	/**
	 * コンポーネントを初期化する。
	 *
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#init()
	 */
	public void init()
	{
		filterStringField.setText("");
		opeBox.setSelectedIndex(0);
		stateBox.setSelectedIndex(0);
	}

	/**
	 * コンポーネントの設定内容を検証する。
	 *
	 * @return 検証OKならtrue
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#commit()
	 */
	public boolean commit()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		String s = filterStringField.getText();
		if (s == null)                return true;
		else if (s.trim().equals("")) return true;

		int ope = opeBox.getSelectedIndex();
		if (ope == 0)                  return true;

		String pat = "[^\\s\\\"\\\']+";
		String msg = null;
		if (Pattern.matches(pat, s) == false) msg = rm.getResource("E9015");

		return isValid(msg);
	}

	/**
	 * 設定されたフィルターを返す。
	 *
	 * @return フィルター
	 * @see jp.co.argo21.nautica.tool.wfadm.common.FilteringDialogComponent#getFilter()
	 */
	public Filter getFilter()
	{
		int op = opeBox.getSelectedIndex();
		int st = stateBox.getSelectedIndex();
		String s = filterStringField.getText();

		NameFilter nfilter = null;
		StateFilter sfilter = null;
		if (op != 0) {
			int comparison = 0;
			if (op == 2) comparison = NameFilter.LIKE;
			else         comparison = NameFilter.EQUAL;
			if (s == null || s.trim().equals("")) {
				//NOP
			} else {
				nfilter = new NameFilter(s, comparison);
			}
		}

		if (st == 1) sfilter = new StateFilter(ProcessDefinitionState.ENABLE);
		else if (st == 2) sfilter = new StateFilter(ProcessDefinitionState.DISABLE);
		
		Filter filter = null; 
		if (nfilter != null && sfilter != null) {
			Filter[] filters = new Filter[2];
			filters[0] = nfilter;
			filters[1] = sfilter;
			filter = new GroupFilter(filters);
		} else if (nfilter != null) {
			filter = nfilter;
		} else if (sfilter != null) {
			filter = sfilter;
		} else {
			Filter[] filters = new Filter[0];
			filter = new GroupFilter(filters);
		}
		return filter;
	}
	
	/**
	 * チェックエラーがなかったかどうかを返す。
	 *
	 * @param rm リソースマネージャ
	 * @return	チェック結果
	 */
	private boolean isValid(String msg)
	{
		if (msg != null) {
			DialogUtilities.showError(msg);
			return false;
		}
		return true;
	}
}
