/* $Id: EngineViewFrame.java,v 1.1 2007/10/19 07:29:57 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Properties;

import javax.swing.JInternalFrame;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.plaf.TabbedPaneUI;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.ExpandVetoException;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.swing.ExceptionViewer;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.BusyTask;
import jp.co.argo21.nautica.tool.wfadm.common.FilteringDialog;
import jp.co.argo21.nautica.tool.wfadm.common.PropertyDialog;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.ProcDef;
import jp.co.argo21.nautica.workflow.wfmc.Process;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;

/**
 * エンジンビューの内部フレームである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class EngineViewFrame
extends JInternalFrame
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 背景色 */
	private static final Color BACKGROUND                 = Color.white;
	/** プロセス背景色(完了) */
	private static final Color PROC_COMPLETED_BACKGROUND  = new Color(230, 230, 230);
	/** プロセス背景色(中断) */
	private static final Color PROC_TERMINATED_BACKGROUND = new Color(255, 255, 230);
	/** プロセス背景色(中止) */
	private static final Color PROC_ABORTED_BACKGROUND    = new Color(255, 230, 230);

	/** エンジンビューコントローラ */
	private EngineViewController engineViewController;
	/** スプリットペイン */
	private JSplitPane splitPane;
	/** プロセス情報ツリー */
	private JTree tree;
	/** フロー表示タブ */
	private JTabbedPane tabbedPane;
	/** エンジンノード用ポップアップメニュー */
	private JPopupMenu enginePopup;
	/** プロセス定義ノード用ポップアップメニュー */
	private JPopupMenu procDefPopup;
	/** プロセスノード用ポップアップメニュー */
	private JPopupMenu procPopup;

	/**
	 * 指定したタイトルでエンジンビューを生成する。
	 *
	 * @param title
	 */
	public EngineViewFrame(String title, EngineViewController evc)
	{
		super(title, true, true, true, true);
		
		engineViewController = evc;

		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		setFrameIcon(jp.co.argo21.nautica.tool.wfadm.IconManager.getIcon("process-frameicon"));

		createEnginePopup(rm);
		createProcessDefinitionPopup(rm);
		createProcessPopup(rm);
		
		EngineTreeListner listener = new EngineTreeListner();
		EngineNode root = new EngineNode(this);
		TreeModel model = new DefaultTreeModel(root);
		tree = new JTree(model);
		tree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
		tree.addTreeWillExpandListener(listener);
		tree.addMouseListener(listener);
		tree.setCellRenderer(new EngineTreeCellRenderer());

		JScrollPane sp = new JScrollPane(tree);
		sp.setPreferredSize(new Dimension(200, 200));
		
		tabbedPane = new JTabbedPane();
		tabbedPane.addMouseListener(new TabMouseAdapter());

		splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		splitPane.setLeftComponent(sp);
		splitPane.setRightComponent(tabbedPane);
		
		Container contentPane = this.getContentPane();
		BorderLayout layout = new BorderLayout();
		contentPane.setLayout(layout);
		contentPane.add(splitPane, BorderLayout.CENTER);

		setSize(new Dimension(800, 600));
		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
		addInternalFrameListener(
			new InternalFrameAdapter()
			{
				public void internalFrameOpened(InternalFrameEvent e)
				{
					opened();
				}

				public void internalFrameActivated(InternalFrameEvent e)
				{
					activated();
				}

				public void internalFrameDeactivated(InternalFrameEvent e)
				{
					deactivated();
				}

				public void internalFrameClosing(InternalFrameEvent e)
				{
					close();
				}
			}
		);
		this.validate();
	}
	
	/**
	 * 開いた時の処理
	 */
	void opened()
	{
	}
	
	/**
	 * アクティブ状態になった時の処理
	 */
	void activated()
	{
	}
	
	/**
	 * 非アクティブ状態になった時の処理
	 */
	void deactivated()
	{
	}
	
	/**
	 * 閉じる時の処理
	 */
	void close()
	{
		setVisible(false);
		dispose();
	}

	/**
	 * エンジンビューコントローラを返す。
	 *
	 * @return エンジンビューコントローラ
	 */
	EngineViewController getEngineViewController()
	{
		return engineViewController;
	}
	
	/**
	 * 指定されたプロセスの状態に応じた背景色を返す。
	 *
	 * @param proc プロセス
	 * @return 背景色
	 */
	private Color getBackgroundColor(Process proc)
	{
		if (proc == null) return BACKGROUND;
		
		int state = proc.getProcessState();
		if (state == ProcessState.CLOSED_COMPLETED.toInt())       return PROC_COMPLETED_BACKGROUND;
		else if (state == ProcessState.CLOSED_TERMINATED.toInt()) return PROC_TERMINATED_BACKGROUND;
		else if (state == ProcessState.CLOSED_ABORTED.toInt())    return PROC_ABORTED_BACKGROUND;
		else                                                      return BACKGROUND;
	}
	
	/**
	 * エンジンノード用ポップアップメニューを作成する。
	 */
	private void createEnginePopup(ResourceManager rm)
	{
		enginePopup = new JPopupMenu();

		JMenuItem item1 = new JMenuItem(rm.getResource("EngineViewFrame.popup.3"));
		item1.addActionListener(
			new ActionListener()
			{
				/**
				 * フィルタリング処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					EngineNode node = (EngineNode)path.getLastPathComponent();
					if (node == null) return;
					
					ProcessDefinitionFilteringComponent c = new ProcessDefinitionFilteringComponent();
					Filter filter = FilteringDialog.showDialog(EngineViewFrame.this, c);

					BusyTask task = new NodeReloadTask(node, filter);
					WorkflowAdminManager.runBusyTask(task);

					updateEngineTree();
				}
			}
		);

		JMenuItem item2 = new JMenuItem(rm.getResource("EngineViewFrame.popup.0"));
		item2.addActionListener(
			new ActionListener()
			{
				/**
				 * プロパティ表示処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					ResourceManager rm = WorkflowAdminManager.getResourceManager();

					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					EngineNode node = (EngineNode)path.getLastPathComponent();
					if (node == null) return;
					
					Properties prop = node.getProperties(rm);
					PropertyDialog.showDialog(EngineViewFrame.this, node.toString(), prop);

					updateEngineTree();
				}
			}
		);

		enginePopup.add(item1);
		enginePopup.add(item2);
	}
	
	/**
	 * プロセス定義ノード用ポップアップメニューを作成する。
	 */
	private void createProcessDefinitionPopup(ResourceManager rm)
	{
		procDefPopup = new JPopupMenu();

		JMenuItem item1 = new JMenuItem(rm.getResource("EngineViewFrame.popup.1"));
		item1.addActionListener(
			new ActionListener() {
				/**
				 * プロセス定義表示処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					ProcessDefinitionNode node = (ProcessDefinitionNode)path.getLastPathComponent();
					if (node == null) return;

					ShowProcessDefinitionFlowTask task =
						new ShowProcessDefinitionFlowTask(node);
					WorkflowAdminManager.runBusyTask(task);

					updateEngineTree();
				}
			}
		);

		JMenuItem item2 = new JMenuItem(rm.getResource("EngineViewFrame.popup.3"));
		item2.addActionListener(
			new ActionListener()
			{
				/**
				 * フィルタリング処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					ProcessDefinitionNode node = (ProcessDefinitionNode)path.getLastPathComponent();
					if (node == null) return;
					
					ProcessFilteringComponent c = new ProcessFilteringComponent();
					Filter filter = FilteringDialog.showDialog(EngineViewFrame.this, c);

					BusyTask task = new NodeReloadTask(node, filter);
					WorkflowAdminManager.runBusyTask(task);

					updateEngineTree();
				}
			}
		);

		JMenuItem item3 = new JMenuItem(rm.getResource("EngineViewFrame.popup.0"));
		item3.addActionListener(
			new ActionListener() {
				/**
				 * プロパティ表示処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					ResourceManager rm = WorkflowAdminManager.getResourceManager();

					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					ProcessDefinitionNode node = (ProcessDefinitionNode)path.getLastPathComponent();
					if (node == null) return;
					
					Properties prop = node.getProperties(rm);
					PropertyDialog.showDialog(EngineViewFrame.this, node.toString(), prop);

					updateEngineTree();
				}
			}
		);

		procDefPopup.add(item1);
		procDefPopup.add(item2);
		procDefPopup.add(item3);
	}
	
	/**
	 * プロセスノード用ポップアップメニューを作成する。
	 */
	private void createProcessPopup(ResourceManager rm)
	{
		procPopup = new JPopupMenu();

		JMenuItem item1 = new JMenuItem(rm.getResource("EngineViewFrame.popup.2"));
		item1.addActionListener(
			new ActionListener() {
				/**
				 * プロセス状況表示処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					ProcessNode node = (ProcessNode)path.getLastPathComponent();
					if (node == null) return;

					ShowProcessFlowTask task =
						new ShowProcessFlowTask(node);
					WorkflowAdminManager.runBusyTask(task);

					updateEngineTree();
				}
			}
		);

		JMenuItem item2 = new JMenuItem(rm.getResource("EngineViewFrame.popup.0"));
		item2.addActionListener(
			new ActionListener() {
				/**
				 * プロパティ表示処理
				 *
				 * @param e アクションイベント
				 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
				 */
				public void actionPerformed(ActionEvent e)
				{
					ResourceManager rm = WorkflowAdminManager.getResourceManager();

					TreePath path = tree.getSelectionPath();
					if (path == null) return;
					ProcessNode node = (ProcessNode)path.getLastPathComponent();
					if (node == null) return;
					
					Process p = node.getProcess();
					ProcessLoadTask task = new ProcessLoadTask(p);
					WorkflowAdminManager.runBusyTask(task);
					Process rp = task.getProcess();
					node.setProcess(rp);

					Properties prop = node.getProperties(rm);
					PropertyDialog.showDialog(EngineViewFrame.this, node.toString(), prop);

					updateEngineTree();
				}
			}
		);

		procPopup.add(item1);
		procPopup.add(item2);
	}
	
	/**
	 * ツリーの更新を行う。1.4のJTreeのWindows UIの場合、updateUI()が内部で例外を起こすので、
	 * 緊急的にinstallUI()を使って対応。
	 */
	private void updateEngineTree()
	{
		Runnable doUpdate = new Runnable() {
			public void run()
			{
				tree.updateUI();
			}
		};
		SwingUtilities.invokeLater(doUpdate);
	}
	
	/**
	 * 引数に指定されたプロセス定義名に対応するタブが存在していれば、
	 * そのタブのインデクスを返す。
	 * なければ、-1を返す。
	 *
	 * @param pdname プロセス定義名
	 * @return タブインデクス
	 */
	private int getProcessDefinitionTabIndex(String pdname)
	{
		Component[] comps = tabbedPane.getComponents();
		for (int i = 0; i < comps.length; i++) {
			if (comps[i] instanceof ProcessDefinitionFlowPanel) {
				String n = ((ProcessDefinitionFlowPanel)comps[i]).getProcessDefinitionName();
				if (n.equals(pdname)) return i;
			}
		}
		return -1;
	}
	
	/**
	 * 引数に指定されたプロセスIDに対応するタブが存在していれば、
	 * そのタブのインデクスを返す。
	 * なければ、-1を返す。
	 *
	 * @param pid プロセスID
	 * @return タブインデクス
	 */
	private int getProcessTabIndex(String pid)
	{
		Component[] comps = tabbedPane.getComponents();
		for (int i = 0; i < comps.length; i++) {
			if (comps[i] instanceof ProcessFlowPanel) {
				String n = ((ProcessFlowPanel)comps[i]).getProcessID();
				if (n.equals(pid)) return i;
			}
		}
		return -1;
	}
	
	/**
	 * エンジンツリー用のリスナーオブジェクト。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class EngineTreeListner
	extends MouseAdapter
	implements TreeWillExpandListener
	{
		/**
		 * コンポーネント上でマウスボタンが離されると呼び出されます。
		 * ここではポップアップの表示イベントとして使用します。
		 *
		 * @param	e		マウスイベント
		 */
		public void mouseReleased(MouseEvent e)
		{
			if (e.isPopupTrigger()
					|| e.getModifiers() == MouseEvent.BUTTON2_MASK
					|| e.getModifiers() == MouseEvent.BUTTON3_MASK) {
	
				TreePath path = tree.getPathForLocation(e.getX(), e.getY());
				if (path == null) return; // if not selected anything
	
				tree.setSelectionPath(path);
				TreeNode node = (TreeNode)path.getLastPathComponent();
				if (node == null) return;
	
				JPopupMenu popup = null;
				if (node instanceof EngineNode) popup = enginePopup;
				else if (node instanceof ProcessDefinitionNode) popup = procDefPopup;
				else if (node instanceof ProcessNode) popup = procPopup;
				else return;
	
				// ポップアップを開く
				popup.show(e.getComponent(), e.getX(), e.getY());
			}
		}

		/**
		 * 
		 *
		 * @param event
		 * @throws javax.swing.tree.ExpandVetoException
		 * @see javax.swing.event.TreeWillExpandListener#treeWillCollapse(javax.swing.event.TreeExpansionEvent)
		 */
		public void treeWillCollapse(TreeExpansionEvent event)
		throws ExpandVetoException
		{
		}

		/**
		 * 
		 *
		 * @param event
		 * @throws javax.swing.tree.ExpandVetoException
		 * @see javax.swing.event.TreeWillExpandListener#treeWillExpand(javax.swing.event.TreeExpansionEvent)
		 */
		public void treeWillExpand(TreeExpansionEvent event)
		throws ExpandVetoException
		{
			TreePath tpath = event.getPath();
			
			if (tpath == null) return;

			EngineTreeNode node = (EngineTreeNode)tpath.getLastPathComponent();
			
			if (node == null) return;
			if (node.isLeaf()) return;
			
			BusyTask task = new NodeReloadTask(node, null);
			WorkflowAdminManager.runBusyTask(task);
			updateEngineTree();
		}
	}

	/**
	 * ツリーの要素のレンダリングオブジェクトです。
	 */
	public class EngineTreeCellRenderer
	extends DefaultTreeCellRenderer
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 項目をレンダリングします。
		 *
		 * @param		tree			JTreeコンポーネント
		 * @param		value			項目値
		 * @param		selected		選択状態
		 * @param		expanded		開閉状態
		 * @param		leaf			末端かどうか
		 * @param		row				列
		 * @param		hasFocus		フォーカスの有無
		 * @return		レンダリング済みのコンポーネント
		 */
		public Component getTreeCellRendererComponent(
			JTree tree,
			Object value,
			boolean selected,
			boolean expanded,
			boolean leaf,
			int row,
			boolean hasFocus)
		{
			if (value instanceof ProcessNode) {
				ProcessNode node = (ProcessNode)value;
				Process p = node.getProcess();
				Color c = getBackgroundColor(p);
				setBackgroundNonSelectionColor(c);
			} else {
				setBackgroundNonSelectionColor(BACKGROUND);
			}
			super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);

			return this;
		}
	}

	/**
	 * タブペインでのマウスを制御するアダプタ
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	private class TabMouseAdapter extends MouseAdapter
	{
		
		/**
		 * タブペインでのクリック操作を制御する
		 *
		 * @param e マウスイベント
		 * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
		 */
		public void mouseClicked(MouseEvent e)
		{
			if (e.isControlDown() == false) return;

			int mx = e.getX();
			int my = e.getY();
			TabbedPaneUI ui = tabbedPane.getUI();
			int index = ui.tabForCoordinate(tabbedPane, mx, my);
			
			if (index < 0) return;
			
			tabbedPane.remove(index);
			tabbedPane.repaint();
		}
	}
	
	/**
	 * ノードの再読み込みタスク。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class NodeReloadTask implements BusyTask
	{
		/** エンジンツリーのノード */
		private EngineTreeNode node;
		/** フィルター */
		private Filter filter;
		/** 処理状態 */
		private int state = INIT;
		
		/**
		 * ノードの再読み込みタスクを生成する。
		 *
		 * @param node エンジンツリーのノード
		 * @param filter フィルター
		 */
		NodeReloadTask(EngineTreeNode node, Filter filter)
		{
			this.node = node;
			this.filter = filter;
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}
		/**
		 * ノードの再読み込みを行う。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			try {
				node.reload(filter);
			} catch (Exception ex) {
				ResourceManager rm = WorkflowAdminManager.getResourceManager();
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9014"));
			} finally {
				state = COMPLETE;
			}
		}
	}
	
	/**
	 * プロセス定義フロー表示タスク。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class ShowProcessDefinitionFlowTask implements BusyTask
	{
		/** エンジンツリーのノード */
		private ProcessDefinitionNode node;
		/** 処理状態 */
		private int state = INIT;
		
		/**
		 * プロセス定義フロー表示タスクを生成する。
		 *
		 * @param node エンジンツリーのノード
		 * @param filter フィルター
		 */
		ShowProcessDefinitionFlowTask(ProcessDefinitionNode node)
		{
			this.node = node;
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}
		/**
		 * プロセス定義フロー表示を行う。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			ProcDef pd = node.getProcDef();
			String pdname = pd.getID();
			try {
				int ix = getProcessDefinitionTabIndex(pdname);
				if (ix < 0) {
					PackageDefinition def = engineViewController.getPackgeDefinition(pdname);
					if (def == null) {
						DialogUtilities.showError(rm.getResource("E9016") + "(" + pdname + ")");
						return;
					}
					ProcessDefinitionFlowPanel panel =
						new ProcessDefinitionFlowPanel(EngineViewFrame.this, pd, def);
					tabbedPane.add(pdname, panel);
					tabbedPane.setSelectedComponent(panel);
				} else {
					tabbedPane.setSelectedIndex(ix);
				}
			} catch (Exception ex) {
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9016") + "(" + pdname + ")");
			} finally {
				state = COMPLETE;
			}
		}
	}
	
	/**
	 * プロセスフロー表示タスク。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class ShowProcessFlowTask implements BusyTask
	{
		/** エンジンツリーのノード */
		private ProcessNode node;
		/** 処理状態 */
		private int state = INIT;
		
		/**
		 * プロセスフロー表示タスクを生成する。
		 *
		 * @param node エンジンツリーのノード
		 * @param filter フィルター
		 */
		ShowProcessFlowTask(ProcessNode node)
		{
			this.node = node;
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}
		/**
		 * プロセスフロー表示を行う。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			Process proc = node.getProcess();
			String pname = proc.getName();
			String pid = proc.getID();
			String label = pname + "(" + pid + ")";
			String pdname = proc.getProcDefinitionID();
			try {
				int ix = getProcessTabIndex(pid);
				if (ix < 0) {
					PackageDefinition def = engineViewController.getPackgeDefinition(pdname);
					if (def == null) {
						DialogUtilities.showError(rm.getResource("E9019") + "(" + pname + ")");
						return;
					}
					ProcessFlowPanel panel =
						new ProcessFlowPanel(EngineViewFrame.this, proc, def);
					tabbedPane.add(label, panel);
					tabbedPane.setSelectedComponent(panel);
				} else {
					tabbedPane.setSelectedIndex(ix);
				}
			} catch (Exception ex) {
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9019") + "(" + pname + ")");
			} finally {
				state = COMPLETE;
			}
		}
	}
	
	/**
	 * プロセスインスタンス読み込みタスク。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class ProcessLoadTask implements BusyTask
	{
		/** 処理状態 */
		private int state = INIT;
		
		/** プロセスインスタンス */
		private Process process;
		
		/**
		 * プロセスインスタンス読み込みタスクを生成する。
		 *
		 * @param proc プロセスインスタンス
		 */
		ProcessLoadTask(Process proc)
		{
			this.process = proc;
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}
		
		/**
		 * 保持しているプロセス情報を返す。
		 *
		 * @return プロセス情報
		 */
		Process getProcess()
		{
			return process;
		}

		/**
		 * プロセスインスタンスの読み込みを行う。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			try {
				process = engineViewController.getProcess(process.getID());
			} catch (Exception ex) {
				ResourceManager rm = WorkflowAdminManager.getResourceManager();
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9033"));
			} finally {
				state = COMPLETE;
			}
		}
	}
}
