/* $Id: ConcurrentLayer.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.Color;
import java.awt.Graphics;
import java.util.ArrayList;
import java.util.List;

import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;

/**
 * 並行処理アクティビティで構成される同期階層を管理する。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ConcurrentLayer
{
	//同期階層の開始アクティビティ
	private ActivityDefinition start;
	//同期階層の終了アクティビティ
	private ActivityDefinition end;
	//親の同期階層のリスト
	@SuppressWarnings("unused")
	private ConcurrentLayer parentLayer;
	//内包される同期階層のリスト
	private List<ConcurrentLayer> subLayers = new ArrayList<ConcurrentLayer>();
	//このレイヤーに所属するアクティビティのリスト
	private List<ActivityDefinition> activities = new ArrayList<ActivityDefinition>();

	/**
	 * 同期階層を生成する。
	 * 
	 * @param p	親階層
	 * @param s	並行処理開始アクティビティ
	 * @param e	並行処理終了アクティビティ
	 */
	public ConcurrentLayer(ConcurrentLayer p, ActivityDefinition s, ActivityDefinition e)
	{
		this.parentLayer = p;
		this.start = s;
		this.end = e;
	}

	/**
	 * 同期階層を生成する。
	 */
	public ConcurrentLayer()
	{
	}

	/**
	 * 階層IDを返す。
	 * 
	 * @return	階層ID
	 */
	public String getLayerID()
	{
		return start.getBlockID();
	}

	/**
	 * 階層にアクティビティを追加する。
	 * 
	 * @param def	アクティビティ
	 */	
	public void addLayerDefinition(ActivityDefinition def)
	{
		if (def == null) return;
		activities.add(def);		
	}

	/**
	 * 同期階層を追加する。
	 * 
	 * @param layer	同期階層
	 */	
	public void addLayer(ConcurrentLayer layer)
	{
		if (layer == null) return;
		
		addLayerDefinition(layer.getStartDefinition());
		addLayerDefinition(layer.getEndDefinition());
		subLayers.add(layer);
	}

	/**
	 * 前回層のアクティビティを収集して返す。
	 * 
	 * @param list	収集用リスト
	 */	
	public void collectAllLayerDefinition(List<ActivityDefinition> list)
	{
		for (ConcurrentLayer layer : subLayers) {
			layer.collectAllLayerDefinition(list);
		}

		for (ActivityDefinition act : activities) {
			if (list.contains(act) == false) list.add(act);
		}
		if (list.contains(start) == false) list.add(start);
		if (list.contains(end) == false) list.add(end);
	}

	/**
	 * この階層の並行処理開始アクティビティを返す。
	 * 
	 * @return	並行処理開始アクティビティ
	 */
	public ActivityDefinition getStartDefinition()
	{
		return start;
	}

	/**
	 * この階層の並行処理終了アクティビティを返す。
	 * 
	 * @return	並行処理終了アクティビティ
	 */
	public ActivityDefinition getEndDefinition()
	{
		return end;
	}

	/**
	 * 同期階層の背景を描画する。
	 * 
	 * @param g	描画対象グラフィックス
	 * @param c	描画色
	 * @param grid	単位幅
	 */
	public void paintLayerBackgroud(Graphics g, Color c, int grid)
	{
		int x = start.getLogicalX() * grid;
		int y = start.getLogicalY() * grid;
		int w = end.getLogicalX() * grid + grid - x;
		int h = end.getLogicalY() * grid + grid - y;
		
		g.setColor(c);
		g.fillRect(x, y, w, h);
		
		for  (ConcurrentLayer l : subLayers) {
			l.paintLayerBackgroud(g, c, grid);
		}
	}

	/**
	 * 同期階層の枠を描画する。
	 * 
	 * @param g	描画対象グラフィックス
	 * @param c	描画色
	 * @param grid	単位幅
	 */
	public void paintLayerBorder(Graphics g, Color c, int grid)
	{
		int x = start.getLogicalX() * grid;
		int y = start.getLogicalY() * grid;
		int w = end.getLogicalX() * grid + grid - x;
		int h = end.getLogicalY() * grid + grid - y;

		g.setColor(c);
		g.drawRect(x, y, w, h);
		g.drawRect(x + 1, y + 1, w - 2, h - 2);
		
		for  (ConcurrentLayer l : subLayers) {
			l.paintLayerBorder(g, c, grid);
		}
	}

	/**
	 * この階層の並行処理終了アクティビティを設定する。
	 * 
	 * @param definition	並行処理終了アクティビティ
	 */
	public void setEndDefinition(ActivityDefinition definition)
	{
		end = definition;
	}

	/**
	 * 親の同期階層を設定する。
	 * 
	 * @param layer	同期階層
	 */
	public void setParentLayer(ConcurrentLayer layer)
	{
		parentLayer = layer;
	}

	/**
	 * この階層の並行処理開始アクティビティを設定する。
	 * 
	 * @param definition	並行処理開始アクティビティ
	 */
	public void setStartDefinition(ActivityDefinition definition)
	{
		start = definition;
	}
}
