/* $Id: AttributeComponent.java,v 1.1 2007/10/19 07:29:57 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.text.Document;

import jp.co.argo21.nautica.commons.swing.BooleanDocument;
import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.swing.NumericDocument;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.IconManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.DialogComponent;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;

/**
 * プロセス変数用ダイアログに表示する画面コンポーネント。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class AttributeComponent extends DialogComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 変数アイコン名 */
	private static final String ATTRIBUTE_ICON = "attribute-mini";
	
	/** 変数アイコン */
	private static Icon attributeIcon;
	
	/** テーブルの列名 */
	private static String[] columnNames;

	/** 型名 */
	private static String[] typeNames;

	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** 変数一覧テーブル */
	private JTable attrTable;
	/** 変数地編集エリア */
	private JTextField attrField;
	/** 変更ボタン */
	private JButton updateButton;
	
	/** 文字列入力制御 */
	private Document strDocument = new ByteLimitedDocument(128);
	
	/** 数値入力制御 */
	private Document numDocument = new NumericDocument(9, true);
	
	/** 真理値入力制御 */
	private Document boolDocument = new BooleanDocument();
	
	/** 変数一覧 */
	private List<Attribute> attributes;
	/** 更新変数一覧 */
	private List<Attribute> updateAttributes;
	
	/**
	 * 定数初期化
	 */
	static
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		attributeIcon = IconManager.getIcon(ATTRIBUTE_ICON);

		columnNames = new String[3];
		columnNames[0] = rm.getResource("AttributeComponent.columnName.0");
		columnNames[1] = rm.getResource("AttributeComponent.columnName.1");
		columnNames[2] = rm.getResource("AttributeComponent.columnName.2");

		typeNames = new String[3];
		typeNames[0] = rm.getResource("AttributeComponent.typeName.0");
		typeNames[1] = rm.getResource("AttributeComponent.typeName.1");
		typeNames[2] = rm.getResource("AttributeComponent.typeName.2");
	}
	
	/**
	 * コンポーネントを生成する。
	 */
	public AttributeComponent()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		updateAttributes = new ArrayList<Attribute>();

		JPanel panel = createEditorPanel(rm);
		addTab(rm.getResource("AttributeComponent.tab.0"), panel);

		validate();
	}
	
	/**
	 * プロセス変数パネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createEditorPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.setBorder(border);
		Dimension size = new Dimension(500, 200);
		panel.setSize(size);
		panel.setPreferredSize(size);

		attrTable = new JTable(new AttributeTableModel());
		attrTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		attrTable.setRowHeight(20);
		attrTable.getColumnModel().getColumn(0).setCellRenderer(new AttributeCellRenderer());
		ListSelectionModel lsm = attrTable.getSelectionModel();
		lsm.addListSelectionListener(new SelectAttributeAction());
		JScrollPane sp = new JScrollPane();
		sp.setViewportView(attrTable);
		
		JPanel fieldPanel = new JPanel();
		fieldPanel.setLayout(new BorderLayout());
		attrField = new JTextField();
		attrField.setDocument(strDocument);
		attrField.setEditable(false);

		updateButton = new JButton(rm.getResource("JButton.label.update"));
		updateButton.addActionListener(new UpdateAttributeAction());

		fieldPanel.add(attrField,    BorderLayout.CENTER);
		fieldPanel.add(updateButton, BorderLayout.EAST);
		
		panel.add(sp,          BorderLayout.CENTER);
		panel.add(fieldPanel,  BorderLayout.SOUTH);

		validate();

		return panel;
	}

	/**
	 * 更新された変数一覧を返す。
	 * 
	 * @return	変数一覧
	 */
	public List<Attribute> getUpdatedAttributes()
	{
		return updateAttributes;
	}

	/**
	 * 変数一覧を設定する。
	 * 
	 * @param attrs 変数一覧
	 */
	public void setAttributes(List<Attribute> attrs)
	{
		this.attributes = attrs;
	}

	/**
	 * ビューを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#init()
	 */
	public void init()
	{
	}
	
	/**
	 * 特に処理はない。
	 *
	 * @return 常にtrue
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#commit()
	 */
	public boolean commit()
	{
		return true;
	}
	
	/**
	 * 更新処理。
	 */	
	class UpdateAttributeAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 更新処理。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			int selected = attrTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("W9001"));
				return;
			}
			Attribute attr = (Attribute)attributes.get(selected);
			attr.setValue(attrField.getText());

			//更新リストに追加
			if (updateAttributes.contains(attr)) {
				int ix = updateAttributes.indexOf(attr);
				updateAttributes.set(ix, attr);
			} else {
				updateAttributes.add(attr);
			}
			
			attrTable.clearSelection();
			attrField.setEditable(false);
			attrField.setText("");
			attrTable.updateUI();
		}
	}

	/**
	 * 変数リスト選択時の処理。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class SelectAttributeAction implements ListSelectionListener
	{
		/**
		 * 変数の型に応じて入力エリアを切り替える。
		 * 
		 * @param e	イベント
		 * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
		 */
		public void valueChanged(ListSelectionEvent e)
		{
			if (e.getValueIsAdjusting()) return;

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();

			if (lsm.isSelectionEmpty() == false) {
				int selected = lsm.getMinSelectionIndex();
				Attribute attr = (Attribute)attributes.get(selected);
				String val = attr.getValue();
				int type = attr.getType();

				if (type == Attribute.INTEGER) {
					attrField.setDocument(numDocument);
					attrField.setHorizontalAlignment(SwingUtilities.RIGHT);
				} else if (type == Attribute.BOOLEAN) {
					attrField.setDocument(boolDocument);
					attrField.setHorizontalAlignment(SwingUtilities.LEFT);
				} else {
					attrField.setDocument(strDocument);
					attrField.setHorizontalAlignment(SwingUtilities.LEFT);
				}

				if (val == null) attrField.setText("");
				else attrField.setText(val.trim());

				attrField.setEditable(true);
			}
		}
	}

	/**
	 * プロセス変数テーブル制御モデル
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class AttributeTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return columnNames.length;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			if (attributes == null) return 0;
			return attributes.size();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return columnNames[col];
		}

		/**
		 * カラムの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			Attribute attr = (Attribute)attributes.get(row);

			if (col == 0) {
				return attr;
			} else if (col == 1) {
				int type = attr.getType();

				if (type == Attribute.INTEGER) {
					return typeNames[1];
				} else if (type == Attribute.BOOLEAN) {
					return typeNames[2];
				} else {
					return typeNames[0];
				}
			} else {
				return attr.getValue();
			}
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			if (col == 0) return Attribute.class;
			else return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}

	/**
	 * 変数テーブル用のレンダラー
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	public class AttributeCellRenderer extends DefaultTableCellRenderer
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * レンダリング処理
		 * 
		 * @param table      テーブル
		 * @param o          セル内容
		 * @param isSelected 選択状態
		 * @param hasFocus   フォーカス状態
		 * @param row        行
		 * @param column     列
		 * @return	レンダリング結果
		 * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
		 */		
		public Component
		getTableCellRendererComponent(
				JTable table, 
				Object o, 
				boolean isSelected, 
				boolean hasFocus, 
				int row, 
				int column)
		{
			super.getTableCellRendererComponent(table, o, isSelected, hasFocus, row, column);

			Attribute attr = (Attribute)o;
			setText(attr.getName());

			if (attributeIcon != null) setIcon(attributeIcon);
			return this;
		}
	}
}
