/* $Id: MultiplePropertyComponent.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.common;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.util.Arrays;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.table.AbstractTableModel;

import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;

/**
 * 複数種類のプロパティ用画面コンテンツ
 * 
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class MultiplePropertyComponent
extends DialogComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** テーブルの列名 */
	private static String[] columnNames;
	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/**
	 * 定数の初期化
	 */
	static
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		columnNames = new String[2];
		columnNames[0] = rm.getResource("PropertyComponent.columnName.0");
		columnNames[1] = rm.getResource("PropertyComponent.columnName.1");
	}

	/**
	 * プロパティ用画面を生成する。
	 *
	 * @param prop プロパティ
	 */
	MultiplePropertyComponent(Map<String, Properties> props)
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();
		
		Set<String> keys = props.keySet();
		for (String key : keys) {
			Properties prop = props.get(key);
			JPanel panel = createPropertyPanel(rm, prop);
			String tabName = rm.getResource("PropertyComponent.tab.0") + "(" + key + ")";
			addTab(tabName, panel);
		}

		validate();
	}
	
	/**
	 * プロパティ一覧のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createPropertyPanel(ResourceManager rm, Properties prop)
	{
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.setBorder(border);
		Dimension size = new Dimension(500, 150);
		panel.setSize(size);
		panel.setPreferredSize(size);

		JTable propertyTable = new JTable(new PropertyTableModel(prop));
		propertyTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane sp = new JScrollPane();
		sp.setViewportView(propertyTable);

		panel.add(sp,  BorderLayout.CENTER);

		validate();
		
		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#init()
	 */
	public void init()
	{
	}

	/**
	 * 特に処理はない。
	 *
	 * @return
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#commit()
	 */
	public boolean commit()
	{
		return true;
	}
	
	/**
	 * プロパティテーブル制御モデル
	 */
	class PropertyTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;


		/** プロパティ情報 */
		private Properties properties;

		/**
		 * プロパティテーブル制御モデルを生成する。
		 */
		PropertyTableModel(Properties prop)
		{
			this.properties = prop;
		}

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return columnNames.length;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			if (properties == null) return 0;

			return properties.size();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return columnNames[col];
		}

		/**
		 * セルの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			if (properties == null) return "";

			//キー項目をソートする。
			Set<?> keyset = properties.keySet();
			Object[] keys = keyset.toArray(); 
			Arrays.sort(keys);
			
			if (col == 0) {
				for (int i = 0; i < keys.length; i++) {
					if (i == row) return keys[i].toString();
				}
			} else if (col == 1) {
				for (int i = 0; i < keys.length; i++) {
					if (i == row) {
						return properties.get(keys[i]).toString();
					}
				}
			}
			return "";
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}
}
