/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.utils;

import java.util.Date;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.time.constant.TimeConst;

/**
 * 勤怠管理における有用なメソッドを提供する。<br><br>
 */
public class TimeUtility {
	
	/**
	 * 他クラスからのインスタンス化を防止する。<br>
	 */
	private TimeUtility() {
		// 処理無し
	}
	
	/**
	 * 対象年月及び締日から締期間初日を取得する。<br>
	 * @param cutoffDate  締日
	 * @param targetYear  対象年
	 * @param targetMonth 対象月
	 * @return 締期間初日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	public static Date getCutoffFirstDate(int cutoffDate, int targetYear, int targetMonth) throws MospException {
		// 月末締の場合
		if (cutoffDate == TimeConst.CUTOFF_DATE_LAST_DAY) {
			// 締期間初日設定(対象年月の初日)
			return DateUtility.getFirstDateOfMonth(targetYear, targetMonth);
		}
		// 当月締日判断
		if (cutoffDate > TimeConst.CUTOFF_DATE_THIS_MONTH_MAX) {
			// 当月締の場合
			// 対象年月、締日で日付を取得(締期間初日作成準備)
			Date date = DateUtility.getDate(targetYear, targetMonth, cutoffDate);
			// 月を戻す
			date = DateUtility.addMonth(date, -1);
			// 日を加算(締日の翌日が初日)
			return DateUtility.addDay(date, 1);
		} else {
			// 翌月締の場合
			// 対象年月、締日で日付を取得(締期間初日作成準備)
			Date date = DateUtility.getDate(targetYear, targetMonth, cutoffDate);
			// 締期間初日設定(前月の締日+1日)(翌月締の場合)
			return DateUtility.addDay(date, 1);
		}
	}
	
	/**
	 * 対象年月及び締日から締期間最終日を取得する。<br>
	 * @param cutoffDate  締日
	 * @param targetYear  対象年
	 * @param targetMonth 対象月
	 * @return 締期間最終日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	public static Date getCutoffLastDate(int cutoffDate, int targetYear, int targetMonth) throws MospException {
		// 月末締の場合
		if (cutoffDate == TimeConst.CUTOFF_DATE_LAST_DAY) {
			// 締期間最終日取得(対象年月の最終日)
			return DateUtility.getLastDateOfMonth(targetYear, targetMonth);
		}
		// 当月締日判断
		if (cutoffDate > TimeConst.CUTOFF_DATE_THIS_MONTH_MAX) {
			// 当月締の場合
			// 対象年月、締日で日付を取得(締期間初日作成準備)
			Date date = DateUtility.getDate(targetYear, targetMonth, cutoffDate);
			// 月を戻す
			date = DateUtility.addMonth(date, -1);
			// 日を加算(締日の翌日が初日)
			date = DateUtility.addDay(date, 1);
			// 締期間最終日取得(年月の締日)(当月締の場合)
			return DateUtility.getDate(targetYear, targetMonth, cutoffDate);
		} else {
			// 翌月締の場合
			// 年、月、締日で日付を取得(締期間最終日作成準備)
			Date date = DateUtility.getDate(targetYear, targetMonth, cutoffDate);
			// 締期間最終日設定(翌月の締日)(翌月締の場合)
			return DateUtility.addMonth(date, 1);
		}
	}
	
}
