/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.settings.action;

import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.PaidHolidayInfoReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.PaidHolidayDataDtoInterface;
import jp.mosp.time.settings.vo.PaidHolidayReferenceVo;

/**
 * 従業員別に有給休暇に関する情報を確認する。<br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>

 * </li><li>
 */
public class PaidHolidayReferenceAction extends TimeAction {
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 選択表示コマンド。有給休暇手動付与画面で選択した従業員のサーバ日付時点から1年間の有給休暇に関する詳細情報を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW	= "TM4431";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索コマンド。有効日入力欄に入力された日付を基にその日付から1年間分の有給休暇関連の詳細情報を表示する。<br>
	 */
	public static final String	CMD_SEARCH		= "TM4432";
	
	/**
	 * 一覧結果出力用領域。（残日数(前)）<br>
	 * <br>
	 * 一覧結果に出力する残日数(前)を保存する領域。<br>
	 */
	public static double		formerDate		= 0;
	
	/**
	 * 一覧結果出力用領域。（残時間(前)）<br>
	 * <br>
	 * 一覧結果に出力する残時間(前)を保存する領域。<br>
	 */
	public static int			formerTime		= 0;
	
	/**
	 * 一覧結果出力用領域。（残日数(今)）<br>
	 * <br>
	 * 一覧結果に出力する残日数(今)を保存する領域。<br>
	 */
	public static double		currentDate		= 0;
	
	/**
	 * 一覧結果出力用領域。（残時間(今)）<br>
	 * <br>
	 * 一覧結果に出力する残時間(今)を保存する領域。<br>
	 */
	public static int			currentTime		= 0;
	
	/**
	 * 一覧結果出力用領域。（支給日数）<br>
	 * <br>
	 * 一覧結果に出力する支給日数を保存する領域。<br>
	 */
	public static double		givingDate		= 0;
	
	/**
	 * 一覧結果出力用領域。（支給時間）<br>
	 * <br>
	 * 一覧結果に出力する支給時間を保存する領域。<br>
	 */
	public static int			givingTime		= 0;
	
	/**
	 * 一覧結果出力用領域。（廃棄日数）<br>
	 * <br>
	 * 一覧結果に出力する廃棄日数を保存する領域。<br>
	 */
	public static double		cancelDate		= 0;
	
	/**
	 * 一覧結果出力用領域。（廃棄時間）<br>
	 * <br>
	 * 一覧結果に出力する廃棄時間を保存する領域。<br>
	 */
	public static int			cancelTime		= 0;
	
	/**
	 * 一覧結果出力用領域。（利用日数）<br>
	 * <br>
	 * 一覧結果に出力する利用日数を保存する領域。<br>
	 */
	public static double		useDate			= 0;
	
	/**
	 * 一覧結果出力用領域。（利用時間）<br>
	 * <br>
	 * 一覧結果に出力する利用時間を保存する領域。<br>
	 */
	public static int			useTime			= 0;
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public PaidHolidayReferenceAction() {
		super();
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new PaidHolidayReferenceVo();
	}
	
	@Override
	public void action() throws MospException {
		// VO準備
		prepareVo();
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			search();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void show() throws MospException {
		// 初期値設定
		setDefaultValues();
		// 有給休暇情報表示
		setDefaultList();
		// 年月別表示
		setList();
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void search() throws MospException {
		// 年月別表示
		setList();
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		PaidHolidayReferenceVo vo = (PaidHolidayReferenceVo)mospParams.getVo();
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		// 社員情報の設定
		if (getTransferredCode() != null) {
			setEmployeeParamsForEmployee(getTransferredCode());
		}
		// 入力項目設定
		vo.setTxtEditActivateYear(getFiscalYear(date));
		vo.setTxtLblActivateYear(DateUtility.getStringYear(date));
		vo.setTxtLblActivateMonth(DateUtility.getStringMonth(date));
		vo.setTxtLblActivateDay(DateUtility.getStringDay(date));
		vo.setTxtActiveDate(DateUtility.getStringDate(date));
	}
	
	/**
	 * 初期に表示される有給休暇情報を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setDefaultList() throws MospException {
		// VO準備
		PaidHolidayReferenceVo vo = (PaidHolidayReferenceVo)mospParams.getVo();
		PaidHolidayInfoReferenceBeanInterface getInfo = timeReference().paidHolidayInfo();
		Map<String, Object> map = null;
		if (getTransferredCode() != null) {
			map = getInfo.getPaidHolidayInfo(getPersonalId(getTransferredCode()), DateUtility.getSystemDate());
		} else {
			map = getInfo.getPaidHolidayInfo(getPersonalId(vo.getLblEmployeeCode()), DateUtility.getSystemDate());
		}
		// 休暇情報設定
		// 情報表示欄の設定
		vo.setLblFormerDate(String.valueOf(map.get(TimeConst.CODE_FORMER_YEAR_DATE)));
		vo.setLblFormerTime(String.valueOf(map.get(TimeConst.CODE_FORMER_YEAR_TIME)));
		vo.setLblDate(String.valueOf(map.get(TimeConst.CODE_CURRENT_YEAR_DATE)));
		vo.setLblTime(String.valueOf(map.get(TimeConst.CODE_CURRENT_TIME)));
		vo.setLblGivingDate(String.valueOf(map.get(TimeConst.CODE_GIVING_DATE)));
		vo.setLblGivingTime(String.valueOf(map.get(TimeConst.CODE_GIVING_TIME)));
		vo.setLblCancelDate(String.valueOf(map.get(TimeConst.CODE_CANCEL_DATE)));
		vo.setLblCancelTime(String.valueOf(map.get(TimeConst.CODE_CANCEL_TIME)));
		vo.setLblUseDate(String.valueOf(map.get(TimeConst.CODE_USE_DATE)));
		vo.setLblUseTime(String.valueOf(map.get(TimeConst.CODE_USE_TIME)));
	}
	
	/**
	 * 設定された有効年から取得した年月別の有給休暇情報を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setList() throws MospException {
		// VO準備
		PaidHolidayReferenceVo vo = (PaidHolidayReferenceVo)mospParams.getVo();
		Date date = null;
		// 各タイトルラベル（年月）の設定
		vo.setTxtLblActivateYear(String.valueOf(Integer.valueOf(vo.getTxtEditActivateYear()) + 1));
		vo.setTxtLblPreviousYear(vo.getTxtEditActivateYear());
		// 各月のラベル情報設定（固定値）
		vo.setLblJanuary(TimeConst.CODE_DISPLAY_JANUARY);
		vo.setLblFebruary(TimeConst.CODE_DISPLAY_FEBRUARY);
		vo.setLblMarch(TimeConst.CODE_DISPLAY_MARCH);
		vo.setLblApril(TimeConst.CODE_DISPLAY_APRIL);
		vo.setLblMay(TimeConst.CODE_DISPLAY_MAY);
		vo.setLblJune(TimeConst.CODE_DISPLAY_JUNE);
		vo.setLblJuly(TimeConst.CODE_DISPLAY_JULY);
		vo.setLblAugust(TimeConst.CODE_DISPLAY_AUGUST);
		vo.setLblSeptember(TimeConst.CODE_DISPLAY_SEPTEMBER);
		vo.setLblOctober(TimeConst.CODE_DISPLAY_OCTOBER);
		vo.setLblNovember(TimeConst.CODE_DISPLAY_NOVEMBER);
		vo.setLblDecember(TimeConst.CODE_DISPLAY_DECEMBER);
		// 各月の休暇情報設定
		// 4月分
		int lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_APRIL));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_APRIL, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateApril(String.valueOf(formerDate));
		vo.setLblFormerTimeApril(String.valueOf(formerTime));
		vo.setLblDateApril(String.valueOf(currentDate));
		vo.setLblTimeApril(String.valueOf(currentTime));
		vo.setLblGivingDateApril(String.valueOf(givingDate));
		vo.setLblGivingTimeApril(String.valueOf(givingTime));
		vo.setLblCancelDateApril(String.valueOf(cancelDate));
		vo.setLblCancelTimeApril(String.valueOf(cancelTime));
		vo.setLblUseDateApril(String.valueOf(useDate));
		vo.setLblUseTimeApril(String.valueOf(useTime));
		// 5月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer.valueOf(TimeConst.CODE_DISPLAY_MAY));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_MAY, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateMay(String.valueOf(formerDate));
		vo.setLblFormerTimeMay(String.valueOf(formerTime));
		vo.setLblDateMay(String.valueOf(currentDate));
		vo.setLblTimeMay(String.valueOf(currentTime));
		vo.setLblGivingDateMay(String.valueOf(givingDate));
		vo.setLblGivingTimeMay(String.valueOf(givingTime));
		vo.setLblCancelDateMay(String.valueOf(cancelDate));
		vo.setLblCancelTimeMay(String.valueOf(cancelTime));
		vo.setLblUseDateMay(String.valueOf(useDate));
		vo.setLblUseTimeMay(String.valueOf(useTime));
		// 6月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer.valueOf(TimeConst.CODE_DISPLAY_JUNE));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_JUNE, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateJune(String.valueOf(formerDate));
		vo.setLblFormerTimeJune(String.valueOf(formerTime));
		vo.setLblDateJune(String.valueOf(currentDate));
		vo.setLblTimeJune(String.valueOf(currentTime));
		vo.setLblGivingDateJune(String.valueOf(givingDate));
		vo.setLblGivingTimeJune(String.valueOf(givingTime));
		vo.setLblCancelDateJune(String.valueOf(cancelDate));
		vo.setLblCancelTimeJune(String.valueOf(cancelTime));
		vo.setLblUseDateJune(String.valueOf(useDate));
		vo.setLblUseTimeJune(String.valueOf(useTime));
		// 7月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer.valueOf(TimeConst.CODE_DISPLAY_JULY));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_JULY, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateJuly(String.valueOf(formerDate));
		vo.setLblFormerTimeJuly(String.valueOf(formerTime));
		vo.setLblDateJuly(String.valueOf(currentDate));
		vo.setLblTimeJuly(String.valueOf(currentTime));
		vo.setLblGivingDateJuly(String.valueOf(givingDate));
		vo.setLblGivingTimeJuly(String.valueOf(givingTime));
		vo.setLblCancelDateJuly(String.valueOf(cancelDate));
		vo.setLblCancelTimeJuly(String.valueOf(cancelTime));
		vo.setLblUseDateJuly(String.valueOf(useDate));
		vo.setLblUseTimeJuly(String.valueOf(useTime));
		// 8月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_AUGUST));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_AUGUST, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateAugust(String.valueOf(formerDate));
		vo.setLblFormerTimeAugust(String.valueOf(formerTime));
		vo.setLblDateAugust(String.valueOf(currentDate));
		vo.setLblTimeAugust(String.valueOf(currentTime));
		vo.setLblGivingDateAugust(String.valueOf(givingDate));
		vo.setLblGivingTimeAugust(String.valueOf(givingTime));
		vo.setLblCancelDateAugust(String.valueOf(cancelDate));
		vo.setLblCancelTimeAugust(String.valueOf(cancelTime));
		vo.setLblUseDateAugust(String.valueOf(useDate));
		vo.setLblUseTimeAugust(String.valueOf(useTime));
		// 9月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_SEPTEMBER));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_SEPTEMBER, String
			.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateSeptember(String.valueOf(formerDate));
		vo.setLblFormerTimeSeptember(String.valueOf(formerTime));
		vo.setLblDateSeptember(String.valueOf(currentDate));
		vo.setLblTimeSeptember(String.valueOf(currentTime));
		vo.setLblGivingDateSeptember(String.valueOf(givingDate));
		vo.setLblGivingTimeSeptember(String.valueOf(givingTime));
		vo.setLblCancelDateSeptember(String.valueOf(cancelDate));
		vo.setLblCancelTimeSeptember(String.valueOf(cancelTime));
		vo.setLblUseDateSeptember(String.valueOf(useDate));
		vo.setLblUseTimeSeptember(String.valueOf(useTime));
		// 10月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_OCTOBER));
		date = DateUtility.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_OCTOBER, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateOctober(String.valueOf(formerDate));
		vo.setLblFormerTimeOctober(String.valueOf(formerTime));
		vo.setLblDateOctober(String.valueOf(currentDate));
		vo.setLblTimeOctober(String.valueOf(currentTime));
		vo.setLblGivingDateOctober(String.valueOf(givingDate));
		vo.setLblGivingTimeOctober(String.valueOf(givingTime));
		vo.setLblCancelDateOctober(String.valueOf(cancelDate));
		vo.setLblCancelTimeOctober(String.valueOf(cancelTime));
		vo.setLblUseDateOctober(String.valueOf(useDate));
		vo.setLblUseTimeOctober(String.valueOf(useTime));
		// 11月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_NOVEMBER));
		date = DateUtility
			.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_NOVEMBER, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateNovember(String.valueOf(formerDate));
		vo.setLblFormerTimeNovember(String.valueOf(formerTime));
		vo.setLblDateNovember(String.valueOf(currentDate));
		vo.setLblTimeNovember(String.valueOf(currentTime));
		vo.setLblGivingDateNovember(String.valueOf(givingDate));
		vo.setLblGivingTimeNovember(String.valueOf(givingTime));
		vo.setLblCancelDateNovember(String.valueOf(cancelDate));
		vo.setLblCancelTimeNovember(String.valueOf(cancelTime));
		vo.setLblUseDateNovember(String.valueOf(useDate));
		vo.setLblUseTimeNovember(String.valueOf(useTime));
		// 12月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblPreviousYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_DECEMBER));
		date = DateUtility
			.getDate(vo.getTxtLblPreviousYear(), TimeConst.CODE_DISPLAY_DECEMBER, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateDecember(String.valueOf(formerDate));
		vo.setLblFormerTimeDecember(String.valueOf(formerTime));
		vo.setLblDateDecember(String.valueOf(currentDate));
		vo.setLblTimeDecember(String.valueOf(currentTime));
		vo.setLblGivingDateDecember(String.valueOf(givingDate));
		vo.setLblGivingTimeDecember(String.valueOf(givingTime));
		vo.setLblCancelDateDecember(String.valueOf(cancelDate));
		vo.setLblCancelTimeDecember(String.valueOf(cancelTime));
		vo.setLblUseDateDecember(String.valueOf(useDate));
		vo.setLblUseTimeDecember(String.valueOf(useTime));
		// 1月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblActivateYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_JANUARY));
		date = DateUtility.getDate(vo.getTxtLblActivateYear(), TimeConst.CODE_DISPLAY_JANUARY, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateJanuary(String.valueOf(formerDate));
		vo.setLblFormerTimeJanuary(String.valueOf(formerTime));
		vo.setLblDateJanuary(String.valueOf(currentDate));
		vo.setLblTimeJanuary(String.valueOf(currentTime));
		vo.setLblGivingDateJanuary(String.valueOf(givingDate));
		vo.setLblGivingTimeJanuary(String.valueOf(givingTime));
		vo.setLblCancelDateJanuary(String.valueOf(cancelDate));
		vo.setLblCancelTimeJanuary(String.valueOf(cancelTime));
		vo.setLblUseDateJanuary(String.valueOf(useDate));
		vo.setLblUseTimeJanuary(String.valueOf(useTime));
		// 2月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblActivateYear()), Integer
			.valueOf(TimeConst.CODE_DISPLAY_FEBRUARY));
		date = DateUtility
			.getDate(vo.getTxtLblActivateYear(), TimeConst.CODE_DISPLAY_FEBRUARY, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateFebruary(String.valueOf(formerDate));
		vo.setLblFormerTimeFebruary(String.valueOf(formerTime));
		vo.setLblDateFebruary(String.valueOf(currentDate));
		vo.setLblTimeFebruary(String.valueOf(currentTime));
		vo.setLblGivingDateFebruary(String.valueOf(givingDate));
		vo.setLblGivingTimeFebruary(String.valueOf(givingTime));
		vo.setLblCancelDateFebruary(String.valueOf(cancelDate));
		vo.setLblCancelTimeFebruary(String.valueOf(cancelTime));
		vo.setLblUseDateFebruary(String.valueOf(useDate));
		vo.setLblUseTimeFebruary(String.valueOf(useTime));
		// 3月分
		lastDay = getLastDay(Integer.valueOf(vo.getTxtLblActivateYear()), Integer.valueOf(TimeConst.CODE_DISPLAY_MARCH));
		date = DateUtility.getDate(vo.getTxtLblActivateYear(), TimeConst.CODE_DISPLAY_MARCH, String.valueOf(lastDay));
		setLblDate(date);
		vo.setLblFormerDateMarch(String.valueOf(formerDate));
		vo.setLblFormerTimeMarch(String.valueOf(formerTime));
		vo.setLblDateMarch(String.valueOf(currentDate));
		vo.setLblTimeMarch(String.valueOf(currentTime));
		vo.setLblGivingDateMarch(String.valueOf(givingDate));
		vo.setLblGivingTimeMarch(String.valueOf(givingTime));
		vo.setLblCancelDateMarch(String.valueOf(cancelDate));
		vo.setLblCancelTimeMarch(String.valueOf(cancelTime));
		vo.setLblUseDateMarch(String.valueOf(useDate));
		vo.setLblUseTimeMarch(String.valueOf(useTime));
	}
	
	/**
	 * 設定された有効年から取得した年月別の有給休暇情報を設定する。<br>
	 * @param date 取得対象年月日
	 * @throws MospException 例外発生時
	 */
	public void setLblDate(Date date) throws MospException {
		List<PaidHolidayDataDtoInterface> list = timeReference().paidHolidayData().getPaidHolidayDataInfoList(
				getTransferredCode(), date);
		// 領域の初期化
		formerDate = currentDate = givingDate = cancelDate = useDate = 0;
		formerTime = currentTime = givingTime = cancelTime = useTime = 0;
		for (int i = 0; i < list.size(); i++) {
			PaidHolidayDataDtoInterface dto = list.get(i);
			if (i == 0) {
				// 前年度残日数
				formerDate = dto.getHoldDay();
				// 前年度残時間
				formerTime = dto.getHoldHour();
			} else {
				// 今年度残日数
				currentDate = dto.getHoldDay();
				// 今年度残時間
				currentTime = dto.getHoldHour();
			}
			givingDate = givingDate + dto.getGivingDay();
			givingTime = givingTime + dto.getGivingHour();
			cancelDate = cancelDate + dto.getCancelDay();
			cancelTime = cancelTime + dto.getCancelHour();
			useDate = useDate + dto.getUseDay();
			useTime = useTime + dto.getUseHour();
		}
	}
	
}
