/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.settings.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.base.TimeBeanHandlerInterface;
import jp.mosp.time.bean.PaidHolidayEntranceDateReferenceBeanInterface;
import jp.mosp.time.bean.PaidHolidayEntranceDateRegistBeanInterface;
import jp.mosp.time.bean.PaidHolidayFirstYearRegistBeanInterface;
import jp.mosp.time.bean.PaidHolidayPointDateRegistBeanInterface;
import jp.mosp.time.bean.PaidHolidayRegistBeanInterface;
import jp.mosp.time.bean.StockHolidayRegistBeanInterface;
import jp.mosp.time.dto.settings.PaidHolidayDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayEntranceDateDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayFirstYearDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayPointDateDtoInterface;
import jp.mosp.time.dto.settings.StockHolidayDtoInterface;
import jp.mosp.time.dto.settings.impl.TmmStockHolidayDto;
import jp.mosp.time.settings.base.TimeSettingAction;
import jp.mosp.time.settings.vo.PaidHolidayCardVo;

/**
 * 有給休暇設定情報の個別詳細情報の確認、編集を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_REGIST}
 * </li><li>
 * {@link #CMD_DELETE}
 * </li><li>
 * {@link #CMD_INSERT_MODE}
 * </li><li>
 * {@link #CMD_ADD_MODE}
 * </li><li>
 * {@link #CMD_REPLICATION_MODE}
 * </li><li>
 */
public class PaidHolidayCardAction extends TimeSettingAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 新規登録モードで画面の初期表示を行う。<br>
	 */
	public static final String	CMD_SHOW				= "TM5320";
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 勤務形態一覧画面で選択したレコードの情報を取得し、詳細画面を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW			= "TM5321";
	
	/**
	 * 登録コマンド。<br>
	 * <br>
	 * 各種入力欄に入力されている情報を元に勤務形態情報テーブルに登録する。<br>
	 * 入力チェック時に入力必須項目が入力されていない、または勤務形態コードが
	 * 登録済みのレコードのものと同一であった場合、エラーメッセージを表示する。<br>
	 */
	public static final String	CMD_REGIST				= "TM5325";
	
	/**
	 * 削除コマンド。<br>
	 * <br>
	 * 現在表示している有給休暇情報の論理削除を行う。<br>
	 */
	public static final String	CMD_DELETE				= "TM5327";
	
	/**
	 * 新規登録モード切替コマンド。<br>
	 * <br>
	 * 各種入力欄に表示されている内容をクリアにする。<br>
	 * 登録ボタンクリック時の内容を登録コマンドに切り替え、新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_INSERT_MODE			= "TM5371";
	
	/**
	 * 履歴追加モード切替コマンド。<br>
	 * <br>
	 * 履歴編集モードで読取専用となっていた有効日の年月日入力欄を編集可能にする。<br>
	 * 登録ボタンクリック時のコマンドを履歴追加コマンドに切り替える。<br>
	 * 編集テーブルヘッダに表示されている履歴編集モードリンクを非表示にする。<br>
	 */
	public static final String	CMD_ADD_MODE			= "TM5373";
	
	/**
	 * 複製モード切替コマンド。<br>
	 * <br>
	 * 編集モードで編集不可だった有効日、コードが編集可能となり、登録ボタンクリック時の内容を登録コマンドに切り替える。<br>
	 * モード切替前に現在編集中のレコードのコードを変更することで新たなレコードとして
	 * 登録できる旨を伝える確認ダイアログを表示して利用者が承認をしてからモードを切り替える。<br>
	 */
	public static final String	CMD_REPLICATION_MODE	= "TM5374";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public PaidHolidayCardAction() {
		super();
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new PaidHolidayCardVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			prepareVo();
			editMode();
		} else if (mospParams.getCommand().equals(CMD_REGIST)) {
			// 登録
			prepareVo();
			regist();
		} else if (mospParams.getCommand().equals(CMD_DELETE)) {
			// 削除
			prepareVo();
			delete();
		} else if (mospParams.getCommand().equals(CMD_INSERT_MODE)) {
			// 新規登録モード切替コマンド
			prepareVo();
			insertMode();
		} else if (mospParams.getCommand().equals(CMD_ADD_MODE)) {
			// 履歴追加モード切替コマンド
			prepareVo();
			addMode();
		} else if (mospParams.getCommand().equals(CMD_REPLICATION_MODE)) {
			// 複製モード切替コマンド
			prepareVo();
			replicationMode();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外処理発生
	 */
	protected void show() throws MospException {
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * 登録処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void regist() throws MospException {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 編集モード確認
		if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_INSERT)) {
			// 新規登録
			insert();
		} else if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_ADD)) {
			// 履歴追加
			add();
		} else if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_EDIT)) {
			// 履歴更新
			update();
		}
	}
	
	/**
	 * 新規追加処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void insert() throws MospException {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 登録時に有効フラグにデフォルト値（0）を設定
		vo.setPltEditInactivate("0");
		// 登録クラス取得、DTOの準備
		PaidHolidayEntranceDateDtoInterface dtoEnt = null;
		PaidHolidayPointDateDtoInterface dtoPoint = null;
		// 同コネクションを使用するため最初にtime()を定義する。
		TimeBeanHandlerInterface time = time();
		// 基本情報
		PaidHolidayRegistBeanInterface regist = time.paidHolidayRegist();
		PaidHolidayDtoInterface dto = regist.getInitDto();
		// DTOに値を設定、新規追加処理
		setDtoFields(dto);
		regist.insert(dto);
		// 初年度付与
		PaidHolidayFirstYearRegistBeanInterface registFirst = time.paidHolidayFirstYearRegist();
		PaidHolidayFirstYearDtoInterface dtoFirst = registFirst.getInitDto();
		for (int i = 1; i < 13; i++) {
			dtoFirst.setEntranceMonth(i);
			setDtoFieldsFirst(dtoFirst);
			registFirst.insert(dtoFirst);
		}
		// DTOに値を設定、新規追加処理
		// ストック休暇設定
		StockHolidayRegistBeanInterface registStock = time.stockHolidayRegist();
		StockHolidayDtoInterface dtoStock = registStock.getInitDto();
		setDtoFieldsStock(dtoStock);
		registStock.insert(dtoStock);
		String[] aryPointDateAmount = { vo.getTxtPointDateAmount1(), vo.getTxtPointDateAmount2(),
			vo.getTxtPointDateAmount3(), vo.getTxtPointDateAmount4(), vo.getTxtPointDateAmount5(),
			vo.getTxtPointDateAmount6(), vo.getTxtPointDateAmount7(), vo.getTxtPointDateAmount8(),
			vo.getTxtPointDateAmount9(), vo.getTxtPointDateAmount10(), vo.getTxtPointDateAmount11(),
			vo.getTxtPointDateAmount12() };
		// DTOに値を設定、新規追加処理
		if (vo.getPltPaidHolidayType().equals("0")) {
			// 自動付与【基準日】
			PaidHolidayPointDateRegistBeanInterface registPoint = time.paidHolidayPointDateRegist();
			for (int i = 1; i < 13; i++) {
				if (!aryPointDateAmount[i - 1].isEmpty()) {
					dtoPoint = registPoint.getInitDto();
					dtoPoint.setTimesPointDate(i);
					setDtoFieldsPoint(dtoPoint);
					registPoint.insert(dtoPoint);
				}
			}
		} else if (vo.getPltPaidHolidayType().equals("1") || vo.getPltPaidHolidayType().equals("2")) {
			// 自動付与【入社日】
			int workMonth = 0;
			if (!vo.getTxtWorkYear1().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear1()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth1());
			} else if (!vo.getTxtWorkYear2().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear2()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth2());
			} else if (!vo.getTxtWorkYear3().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear3()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth3());
			} else if (!vo.getTxtWorkYear4().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear4()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth4());
			} else if (!vo.getTxtWorkYear5().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear5()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth5());
			} else if (!vo.getTxtWorkYear6().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear6()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth6());
			} else if (!vo.getTxtWorkYear7().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear7()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth7());
			} else if (!vo.getTxtWorkYear8().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear8()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth8());
			} else if (!vo.getTxtWorkYear9().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear9()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth9());
			} else if (!vo.getTxtWorkYear10().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear10()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth10());
			} else if (!vo.getTxtWorkYear11().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear11()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth11());
			} else if (!vo.getTxtWorkYear12().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear12()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth12());
			}
			if (workMonth != 0) {
				PaidHolidayEntranceDateRegistBeanInterface registEnt = time.paidHolidayEntranceDateRegist();
				dtoEnt = registEnt.getInitDto();
				setDtoFieldsEnt(dtoEnt);
				registEnt.insert(dtoEnt);
			}
		}
		// 新規追加結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 新規追加成功メッセージ設定
		addInsertMessage();
		// 履歴編集モード設定
		setEditUpdateMode(dto.getPaidHolidayCode(), dto.getActivateDate());
	}
	
	/**
	 * 履歴追加処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void add() throws MospException {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 同コネクションを使用するため最初にtime()を定義する。
		TimeBeanHandlerInterface time = time();
		// 登録クラス取得、DTOの準備
		PaidHolidayEntranceDateDtoInterface dtoEnt = null;
		PaidHolidayPointDateDtoInterface dtoPoint = null;
		// 基本情報
		PaidHolidayRegistBeanInterface regist = time.paidHolidayRegist();
		PaidHolidayDtoInterface dto = regist.getInitDto();
		// DTOに値を設定
		setDtoFields(dto);
		// 履歴追加処理
		regist.add(dto);
		// 初年度付与
		PaidHolidayFirstYearRegistBeanInterface registFirst = time.paidHolidayFirstYearRegist();
		PaidHolidayFirstYearDtoInterface dtoFirst = null;
		for (int i = 1; i < 13; i++) {
			dtoFirst = registFirst.getInitDto();
			dtoFirst.setEntranceMonth(i);
			setDtoFieldsFirst(dtoFirst);
			registFirst.add(dtoFirst);
		}
		// DTOに値を設定、履歴追加処理
		// ストック休暇設定
		StockHolidayRegistBeanInterface registStock = time.stockHolidayRegist();
		StockHolidayDtoInterface dtoStock = registStock.getInitDto();
		setDtoFieldsStock(dtoStock);
		registStock.add(dtoStock);
		// DTOに値を設定、履歴追加処理
		if (vo.getPltPaidHolidayType().equals("0")) {
			// 自動付与【基準日】
			PaidHolidayPointDateRegistBeanInterface registPoint = time.paidHolidayPointDateRegist();
			String[] aryPointDateAmount = { vo.getTxtPointDateAmount1(), vo.getTxtPointDateAmount2(),
				vo.getTxtPointDateAmount3(), vo.getTxtPointDateAmount4(), vo.getTxtPointDateAmount5(),
				vo.getTxtPointDateAmount6(), vo.getTxtPointDateAmount7(), vo.getTxtPointDateAmount8(),
				vo.getTxtPointDateAmount9(), vo.getTxtPointDateAmount10(), vo.getTxtPointDateAmount11(),
				vo.getTxtPointDateAmount12() };
			for (int i = 1; i < 13; i++) {
				if (!aryPointDateAmount[i - 1].isEmpty()) {
					dtoPoint = registPoint.getInitDto();
					dtoPoint.setTimesPointDate(i);
					setDtoFieldsPoint(dtoPoint);
					registPoint.add(dtoPoint);
				}
			}
		} else if (vo.getPltPaidHolidayType().equals("1") || vo.getPltPaidHolidayType().equals("2")) {
			// 自動付与【入社日】
			PaidHolidayEntranceDateRegistBeanInterface registEnt = time.paidHolidayEntranceDateRegist();
			PaidHolidayEntranceDateReferenceBeanInterface referenceEnt = timeReference().paidHolidayEntranceDate();
			// システム日付取得
			Date date = DateUtility.getSystemDate();
			int workMonth = getInt(vo.getTxtWorkYear1()) * 12;
			workMonth = workMonth + getInt(vo.getTxtWorkMonth1());
			dtoEnt = referenceEnt.getPaidHolidayEntranceDateInfo(vo.getTxtPaidHolidayCode(), date, workMonth);
			setDtoFieldsEnt(dtoEnt);
			registEnt.add(dtoEnt);
		}
		// 更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addInsertHistoryMessage();
		// 履歴編集モード設定
		setEditUpdateMode(dto.getPaidHolidayCode(), dto.getActivateDate());
	}
	
	/**
	 * 履歴更新処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void update() throws MospException {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 同コネクションを使用するため最初にtime()を定義する。
		TimeBeanHandlerInterface time = time();
		// 登録クラス取得、DTOの準備
		PaidHolidayEntranceDateDtoInterface dtoEnt = null;
		PaidHolidayPointDateDtoInterface dtoPoint = null;
		// 基本情報
		PaidHolidayRegistBeanInterface regist = time.paidHolidayRegist();
		PaidHolidayDtoInterface dto = regist.getInitDto();
		// DTOに値を設定
		setDtoFields(dto);
		// 更新処理
		regist.update(dto);
		// 初年度付与
		PaidHolidayFirstYearRegistBeanInterface registFirst = time.paidHolidayFirstYearRegist();
		PaidHolidayFirstYearDtoInterface dtoFirst = null;
		for (int i = 1; i < 13; i++) {
			dtoFirst = registFirst.getInitDto();
			dtoFirst.setEntranceMonth(i);
			setDtoFieldsFirst(dtoFirst);
			registFirst.update(dtoFirst);
		}
		// DTOに値を設定、履歴更新処理
		// ストック休暇設定
		StockHolidayRegistBeanInterface registStock = time.stockHolidayRegist();
		StockHolidayDtoInterface dtoStock = new TmmStockHolidayDto();
		setDtoFieldsStock(dtoStock);
		registStock.update(dtoStock);
		// DTOに値を設定、履歴更新処理
		if (vo.getPltPaidHolidayType().equals("0")) {
			// 自動付与【基準日】
			// 最初に基準日のデータを全て削除する
			PaidHolidayPointDateRegistBeanInterface registPoint = time.paidHolidayPointDateRegist();
			registPoint.delete(dtoStock.getPaidHolidayCode(), dtoStock.getActivateDate());
			String[] aryPointDateAmount = { vo.getTxtPointDateAmount1(), vo.getTxtPointDateAmount2(),
				vo.getTxtPointDateAmount3(), vo.getTxtPointDateAmount4(), vo.getTxtPointDateAmount5(),
				vo.getTxtPointDateAmount6(), vo.getTxtPointDateAmount7(), vo.getTxtPointDateAmount8(),
				vo.getTxtPointDateAmount9(), vo.getTxtPointDateAmount10(), vo.getTxtPointDateAmount11(),
				vo.getTxtPointDateAmount12() };
			for (int i = 1; i < 13; i++) {
				if (!aryPointDateAmount[i - 1].isEmpty()) {
					dtoPoint = registPoint.getInitDto();
					dtoPoint.setTimesPointDate(i);
					setDtoFieldsPoint(dtoPoint);
					registPoint.update(dtoPoint);
				}
			}
		} else if (vo.getPltPaidHolidayType().equals("1") || vo.getPltPaidHolidayType().equals("2")) {
			// 自動付与【入社日】
			PaidHolidayEntranceDateRegistBeanInterface registEnt = time.paidHolidayEntranceDateRegist();
			PaidHolidayEntranceDateReferenceBeanInterface referenceEnt = timeReference().paidHolidayEntranceDate();
			// システム日付取得
			Date date = DateUtility.getSystemDate();
			int workMonth = getInt(vo.getTxtWorkYear1()) * 12;
			workMonth = workMonth + getInt(vo.getTxtWorkMonth1());
			dtoEnt = referenceEnt.getPaidHolidayEntranceDateInfo(vo.getTxtPaidHolidayCode(), date, workMonth);
			setDtoFieldsEnt(dtoEnt);
			registEnt.update(dtoEnt);
		}
		// 更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addUpdateMessage();
		// 履歴編集モード設定
		setEditUpdateMode(dto.getPaidHolidayCode(), dto.getActivateDate());
	}
	
	/**
	 * 削除処理を行う。<br>
	 * @throws MospException  インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void delete() throws MospException {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 同コネクションを使用するため最初にtime()を定義する。
		TimeBeanHandlerInterface time = time();
		// DTOの準備
		// 基本情報
		PaidHolidayDtoInterface dto = timeReference().paidHoliday().findForKey(vo.getTxtPaidHolidayCode(),
				getEditActivateDate());
		// DTOに値を設定、履歴追加処理
		setDtoFields(dto);
		// 削除処理
		time.paidHolidayRegist().delete(dto);
		// 初年度付与
		// 削除処理
		time.paidHolidayFirstYearRegist().delete(dto.getPaidHolidayCode(), dto.getActivateDate());
		// DTOに値を設定、履歴追加処理
		// ストック休暇設定
		StockHolidayDtoInterface dtoStock = timeReference().stockHoliday().findForKey(vo.getTxtPaidHolidayCode(),
				getEditActivateDate());
		setDtoFieldsStock(dtoStock);
		// 削除処理
		time.stockHolidayRegist().delete(dtoStock);
		// DTOに値を設定、履歴追加処理
		if (vo.getPltPaidHolidayType().equals("0")) {
			// 自動付与【基準日】
			// 削除処理
			time.paidHolidayPointDateRegist().delete(dto.getPaidHolidayCode(), dto.getActivateDate());
		} else if (vo.getPltPaidHolidayType().equals("1") || vo.getPltPaidHolidayType().equals("2")) {
			// 自動付与【入社日】
			int workMonth = 0;
			if (!vo.getTxtWorkYear1().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear1()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth1());
			} else if (!vo.getTxtWorkYear2().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear2()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth2());
			} else if (!vo.getTxtWorkYear3().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear3()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth3());
			} else if (!vo.getTxtWorkYear4().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear4()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth4());
			} else if (!vo.getTxtWorkYear5().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear5()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth5());
			} else if (!vo.getTxtWorkYear6().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear6()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth6());
			} else if (!vo.getTxtWorkYear7().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear7()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth7());
			} else if (!vo.getTxtWorkYear8().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear8()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth8());
			} else if (!vo.getTxtWorkYear9().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear9()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth9());
			} else if (!vo.getTxtWorkYear10().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear10()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth10());
			} else if (!vo.getTxtWorkYear11().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear11()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth11());
			} else if (!vo.getTxtWorkYear12().isEmpty()) {
				workMonth = getInt(vo.getTxtWorkYear12()) * 12;
				workMonth = workMonth + getInt(vo.getTxtWorkMonth12());
			}
			if (workMonth != 0) {
				// 削除処理
				time.paidHolidayEntranceDateRegist().delete(dto.getPaidHolidayCode(), dto.getActivateDate());
			}
		}
		// 削除結果確認
		if (mospParams.hasErrorMessage()) {
			// 削除失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		// 削除成功メッセージ設定
		addDeleteMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
	}
	
	/**
	 * 新規登録モードで画面を表示する。<br>
	 */
	protected void insertMode() {
		// 編集モード設定
		setEditInsertMode();
		// 初期値設定
		setDefaultValues();
	}
	
	/**
	 * 履歴追加モードで画面を表示する。<br>
	 */
	protected void addMode() {
		// 履歴追加モード設定
		setEditAddMode();
	}
	
	/**
	 * 複製モードで画面を表示する。<br>
	 */
	protected void replicationMode() {
		// 複製モード設定
		setEditReplicationMode();
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// コードを空白に設定
		vo.setTxtPaidHolidayCode("");
	}
	
	/**
	 * 履歴編集モードで画面を表示する。<br>
	 * 履歴編集対象は、遷移汎用コード及び有効日で取得する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void editMode() throws MospException {
		// 初期値設定
		setDefaultValues();
		// 遷移汎用コード及び有効日から履歴編集対象を取得し編集モードを設定
		setEditUpdateMode(getTransferredCode(), getDate(getTransferredActivateDate()));
	}
	
	/**
	 * 履歴編集モードを設定する。<br>
	 * 有休コードと有効日で編集対象情報を取得する。<br>
	 * @param paidHolidayCode	勤務形態
	 * @param activateDate		有効日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditUpdateMode(String paidHolidayCode, Date activateDate) throws MospException {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 有給休暇マスタから情報を取得
		PaidHolidayDtoInterface dto = timeReference().paidHoliday().findForKey(paidHolidayCode, activateDate);
		// 有給休暇初年度マスタから情報を取得
		PaidHolidayFirstYearDtoInterface dtoFirst = timeReference().paidHolidayFirstYear().findForKey(paidHolidayCode,
				activateDate, 1);
		// 有給休暇基準日マスタから情報を取得
		List<PaidHolidayPointDateDtoInterface> listPoint = timeReference().paidHolidayPointDate().findForList(
				paidHolidayCode, activateDate);
		// 有給休暇入社日マスタから情報を取得
		List<PaidHolidayEntranceDateDtoInterface> listEnt = timeReference().paidHolidayEntranceDate().findForList(
				paidHolidayCode, activateDate);
		// ストック休暇マスタから情報を取得
		StockHolidayDtoInterface dtoStock = timeReference().stockHoliday().findForKey(paidHolidayCode, activateDate);
		// 存在確認
		checkSelectedDataExist(dto);
		checkSelectedDataExist(dtoFirst);
		if (listPoint.size() != 0) {
			PaidHolidayPointDateDtoInterface dtoPoint = listPoint.get(0);
			checkSelectedDataExist(dtoPoint);
		} else if (listEnt.size() != 0) {
			PaidHolidayEntranceDateDtoInterface dtoEnt = listEnt.get(0);
			checkSelectedDataExist(dtoEnt);
		}
		checkSelectedDataExist(dtoStock);
		// VOにセット
		setVoFields(dto);
		setVoFieldsFirst(dtoFirst);
		setVoFieldsPoint(vo.getTxtPaidHolidayCode(), getEditActivateDate());
		setVoFieldsEnt(vo.getTxtPaidHolidayCode());
		setVoFieldsStock(dtoStock);
		// 編集モード設定
		setEditUpdateMode(timeReference().paidHoliday().getPaidHolidayHistory(paidHolidayCode));
	}
	
	/**
	 * 初期値を設定する。<br>
	 */
	public void setDefaultValues() {
		// VO準備
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// 登録情報欄の初期値設定
		vo.setTxtPaidHolidayCode("");
		vo.setTxtPaidHolidayName("");
		vo.setTxtPaidHolidayAbbr("");
		vo.setTxtWorkRatio("0");
		vo.setPltPaidHolidayType("0");
		vo.setTxtScheduleGiving("0");
		vo.setPltTimelyPaidHoliday("0");
		vo.setPltTimelyPaidHolidayTime("0");
		vo.setPltTimeUnitPaidHoliday("0");
		vo.setTxtMaxCarryOverAmount("");
		vo.setTxtTotalMaxAmount("");
		vo.setPltMaxCarryOverYear("0");
		
		vo.setTxtGivingTimingJanuary("0");
		vo.setTxtGivingTimingFebruary("0");
		vo.setTxtGivingTimingMarch("0");
		vo.setTxtGivingTimingApril("0");
		vo.setTxtGivingTimingMay("0");
		vo.setTxtGivingTimingJune("0");
		vo.setTxtGivingTimingJuly("0");
		vo.setTxtGivingTimingAugust("0");
		vo.setTxtGivingTimingSeptember("0");
		vo.setTxtGivingTimingOctober("0");
		vo.setTxtGivingTimingNovember("0");
		vo.setTxtGivingTimingDecember("0");
		
		vo.setTxtGivingAmountJanuary("0");
		vo.setTxtGivingAmountFebruary("0");
		vo.setTxtGivingAmountMarch("0");
		vo.setTxtGivingAmountApril("0");
		vo.setTxtGivingAmountMay("0");
		vo.setTxtGivingAmountJune("0");
		vo.setTxtGivingAmountJuly("0");
		vo.setTxtGivingAmountAugust("0");
		vo.setTxtGivingAmountSeptember("0");
		vo.setTxtGivingAmountOctober("0");
		vo.setTxtGivingAmountNovember("0");
		vo.setTxtGivingAmountDecember("0");
		
		vo.setTxtGivingLimitJanuary("0");
		vo.setTxtGivingLimitFebruary("0");
		vo.setTxtGivingLimitMarch("0");
		vo.setTxtGivingLimitApril("0");
		vo.setTxtGivingLimitMay("0");
		vo.setTxtGivingLimitJune("0");
		vo.setTxtGivingLimitJuly("0");
		vo.setTxtGivingLimitAugust("0");
		vo.setTxtGivingLimitSeptember("0");
		vo.setTxtGivingLimitOctober("0");
		vo.setTxtGivingLimitNovember("0");
		vo.setTxtGivingLimitDecember("0");
		
		vo.setTxtPointDateMonth("1");
		vo.setTxtPointDateDay("1");
		
		vo.setTxtTimesPointDate1("1");
		vo.setTxtTimesPointDate2("2");
		vo.setTxtTimesPointDate3("3");
		vo.setTxtTimesPointDate4("4");
		vo.setTxtTimesPointDate5("5");
		vo.setTxtTimesPointDate6("6");
		vo.setTxtTimesPointDate7("7");
		vo.setTxtTimesPointDate8("8");
		vo.setTxtTimesPointDate9("9");
		vo.setTxtTimesPointDate10("10");
		vo.setTxtTimesPointDate11("11");
		vo.setTxtTimesPointDate12("12");
		
		vo.setTxtPointDateAmount1("");
		vo.setTxtPointDateAmount2("");
		vo.setTxtPointDateAmount3("");
		vo.setTxtPointDateAmount4("");
		vo.setTxtPointDateAmount5("");
		vo.setTxtPointDateAmount6("");
		vo.setTxtPointDateAmount7("");
		vo.setTxtPointDateAmount8("");
		vo.setTxtPointDateAmount9("");
		vo.setTxtPointDateAmount10("");
		vo.setTxtPointDateAmount11("");
		vo.setTxtPointDateAmount12("");
		
		vo.setTxtGeneralPointAmount("0");
		
		vo.setTxtWorkYear1("");
		vo.setTxtWorkYear2("");
		vo.setTxtWorkYear3("");
		vo.setTxtWorkYear4("");
		vo.setTxtWorkYear5("");
		vo.setTxtWorkYear6("");
		vo.setTxtWorkYear7("");
		vo.setTxtWorkYear8("");
		vo.setTxtWorkYear9("");
		vo.setTxtWorkYear10("");
		vo.setTxtWorkYear11("");
		vo.setTxtWorkYear12("");
		
		vo.setTxtWorkMonth1("");
		vo.setTxtWorkMonth2("");
		vo.setTxtWorkMonth3("");
		vo.setTxtWorkMonth4("");
		vo.setTxtWorkMonth5("");
		vo.setTxtWorkMonth6("");
		vo.setTxtWorkMonth7("");
		vo.setTxtWorkMonth8("");
		vo.setTxtWorkMonth9("");
		vo.setTxtWorkMonth10("");
		vo.setTxtWorkMonth11("");
		vo.setTxtWorkMonth12("");
		
		vo.setTxtJoiningDateAmount1("");
		vo.setTxtJoiningDateAmount2("");
		vo.setTxtJoiningDateAmount3("");
		vo.setTxtJoiningDateAmount4("");
		vo.setTxtJoiningDateAmount5("");
		vo.setTxtJoiningDateAmount6("");
		vo.setTxtJoiningDateAmount7("");
		vo.setTxtJoiningDateAmount8("");
		vo.setTxtJoiningDateAmount9("");
		vo.setTxtJoiningDateAmount10("");
		vo.setTxtJoiningDateAmount11("");
		vo.setTxtJoiningDateAmount12("");
		
		vo.setTxtGeneralJoiningMonth("");
		vo.setTxtGeneralJoiningAmount("");
		
		vo.setTxtStockYearAmount("0");
		vo.setTxtStockTotalAmount("0");
		vo.setTxtStockLimitDate("0");
		
		// 各識別IDの初期化
		long[] aryId = new long[12];
		vo.setTmmPaidHolidayFirstYearId(aryId);
		vo.setTmmPaidHolidayPointDateId(aryId);
		vo.setTmmPaidHolidayEntranceDateId(aryId);
		
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setDtoFields(PaidHolidayDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setTmmPaidHolidayId(vo.getRecordId());
		dto.setActivateDate(getEditActivateDate());
		dto.setPaidHolidayCode(vo.getTxtPaidHolidayCode());
		dto.setPaidHolidayName(vo.getTxtPaidHolidayName());
		dto.setPaidHolidayAbbr(vo.getTxtPaidHolidayAbbr());
		dto.setWorkRatio(getInt(vo.getTxtWorkRatio()));
		dto.setPaidHolidayType(getInt(vo.getPltPaidHolidayType()));
		dto.setScheduleGiving(getInt(vo.getTxtScheduleGiving()));
		dto.setTimelyPaidHolidayFlag(getInt(vo.getPltTimelyPaidHoliday()));
		if (vo.getPltTimelyPaidHoliday().equals("0")) {
			dto.setTimelyPaidHolidayTime(getInt(vo.getPltTimelyPaidHolidayTime()));
			dto.setTimeUnitPaidHoliday(getInt(vo.getPltTimeUnitPaidHoliday()));
		}
		dto.setMaxCarryOverYear(getInt(vo.getPltMaxCarryOverYear()));
		dto.setInactivateFlag(getInt(vo.getPltEditInactivate()));
		dto.setPointDateMonth(getInt(vo.getTxtPointDateMonth()));
		dto.setPointDateDay(getInt(vo.getTxtPointDateDay()));
		if (vo.getTxtGeneralPointAmount().length() != 0) {
			dto.setGeneralPointAmount(getInt(vo.getTxtGeneralPointAmount()));
		}
		if (vo.getTxtGeneralJoiningMonth().length() != 0) {
			dto.setGeneralJoiningMonth(getInt(vo.getTxtGeneralJoiningMonth()));
		}
		if (vo.getTxtGeneralJoiningAmount().length() != 0) {
			dto.setGeneralJoiningAmount(getInt(vo.getTxtGeneralJoiningAmount()));
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setVoFields(PaidHolidayDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setRecordId(dto.getTmmPaidHolidayId());
		vo.setTxtEditActivateYear(getStringYear(dto.getActivateDate()));
		vo.setTxtEditActivateMonth(getStringMonth(dto.getActivateDate()));
		vo.setTxtEditActivateDay(getStringDay(dto.getActivateDate()));
		vo.setTxtPaidHolidayCode(dto.getPaidHolidayCode());
		vo.setTxtPaidHolidayName(dto.getPaidHolidayName());
		vo.setTxtPaidHolidayAbbr(dto.getPaidHolidayAbbr());
		vo.setTxtWorkRatio(String.valueOf(dto.getWorkRatio()));
		vo.setPltPaidHolidayType(String.valueOf(dto.getPaidHolidayType()));
		vo.setTxtScheduleGiving(String.valueOf(dto.getScheduleGiving()));
		vo.setPltTimelyPaidHoliday(String.valueOf(dto.getTimelyPaidHolidayFlag()));
		vo.setPltTimelyPaidHolidayTime(String.valueOf(dto.getTimelyPaidHolidayTime()));
		vo.setPltTimeUnitPaidHoliday(String.valueOf(dto.getTimeUnitPaidHoliday()));
		vo.setTxtMaxCarryOverAmount(String.valueOf(dto.getMaxCarryOverAmount()));
		vo.setTxtTotalMaxAmount(String.valueOf(dto.getTotalMaxAmount()));
		vo.setPltMaxCarryOverYear(String.valueOf(dto.getMaxCarryOverYear()));
		vo.setPltEditInactivate(String.valueOf(dto.getInactivateFlag()));
		vo.setPltEditInactivate(String.valueOf(dto.getInactivateFlag()));
		vo.setTxtGeneralPointAmount(String.valueOf(dto.getGeneralPointAmount()));
		vo.setTxtGeneralJoiningMonth(String.valueOf(dto.getGeneralJoiningMonth()));
		vo.setTxtGeneralJoiningAmount(String.valueOf(dto.getGeneralJoiningAmount()));
		vo.setTxtPointDateMonth(String.valueOf(dto.getPointDateMonth()));
		vo.setTxtPointDateDay(String.valueOf(dto.getPointDateDay()));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setDtoFieldsFirst(PaidHolidayFirstYearDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setActivateDate(getEditActivateDate());
		dto.setPaidHolidayCode(vo.getTxtPaidHolidayCode());
		dto.setInactivateFlag(getInt(vo.getPltEditInactivate()));
		// 値が何も入っていない場合は0を設定する
		// 1月分
		if (dto.getEntranceMonth() == 1) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(0));
			if (!(vo.getTxtGivingTimingJanuary().isEmpty()) || !(vo.getTxtGivingAmountJanuary().isEmpty())
					|| !(vo.getTxtGivingLimitJanuary().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingJanuary()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountJanuary()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitJanuary()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 2月分
		if (dto.getEntranceMonth() == 2) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(1));
			if (!(vo.getTxtGivingTimingFebruary().isEmpty()) || !(vo.getTxtGivingAmountFebruary().isEmpty())
					|| !(vo.getTxtGivingLimitFebruary().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingFebruary()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountFebruary()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitFebruary()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 3月分
		if (dto.getEntranceMonth() == 3) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(2));
			if (!(vo.getTxtGivingTimingMarch().isEmpty()) || !(vo.getTxtGivingAmountMarch().isEmpty())
					|| !(vo.getTxtGivingLimitMarch().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingMarch()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountMarch()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitMarch()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 4月分
		if (dto.getEntranceMonth() == 4) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(3));
			if (!(vo.getTxtGivingTimingApril().isEmpty()) || !(vo.getTxtGivingAmountApril().isEmpty())
					|| !(vo.getTxtGivingLimitApril().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingApril()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountApril()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitApril()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 5月分
		if (dto.getEntranceMonth() == 5) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(4));
			if (!(vo.getTxtGivingTimingMay().isEmpty()) || !(vo.getTxtGivingAmountMay().isEmpty())
					|| !(vo.getTxtGivingLimitMay().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingMay()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountMay()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitMay()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 6月分
		if (dto.getEntranceMonth() == 6) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(5));
			if (!(vo.getTxtGivingTimingJune().isEmpty()) || !(vo.getTxtGivingAmountJune().isEmpty())
					|| !(vo.getTxtGivingLimitJune().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingJune()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountJune()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitJune()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 7月分
		if (dto.getEntranceMonth() == 7) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(6));
			if (!(vo.getTxtGivingTimingJuly().isEmpty()) || !(vo.getTxtGivingAmountJuly().isEmpty())
					|| !(vo.getTxtGivingLimitJuly().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingJuly()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountJuly()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitJuly()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 8月分
		if (dto.getEntranceMonth() == 8) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(7));
			if (!(vo.getTxtGivingTimingAugust().isEmpty()) || !(vo.getTxtGivingAmountAugust().isEmpty())
					|| !(vo.getTxtGivingLimitAugust().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingAugust()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountAugust()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitAugust()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 9月分
		if (dto.getEntranceMonth() == 9) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(8));
			if (!(vo.getTxtGivingTimingSeptember().isEmpty()) || !(vo.getTxtGivingAmountSeptember().isEmpty())
					|| !(vo.getTxtGivingLimitSeptember().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingSeptember()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountSeptember()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitSeptember()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 10月分
		if (dto.getEntranceMonth() == 10) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(9));
			if (!(vo.getTxtGivingTimingOctober().isEmpty()) || !(vo.getTxtGivingAmountOctober().isEmpty())
					|| !(vo.getTxtGivingLimitOctober().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingOctober()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountOctober()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitOctober()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 11月分
		if (dto.getEntranceMonth() == 11) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(10));
			if (!(vo.getTxtGivingTimingNovember().isEmpty()) || !(vo.getTxtGivingAmountNovember().isEmpty())
					|| !(vo.getTxtGivingLimitNovember().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingNovember()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountNovember()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitNovember()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
		// 12月分
		if (dto.getEntranceMonth() == 12) {
			dto.setTmmPaidHolidayFirstYearId(vo.getTmmPaidHolidayFirstYearId(11));
			if (!(vo.getTxtGivingTimingDecember().isEmpty()) || !(vo.getTxtGivingAmountDecember().isEmpty())
					|| !(vo.getTxtGivingLimitDecember().isEmpty())) {
				dto.setGivingMonth(getInt(vo.getTxtGivingTimingDecember()));
				dto.setGivingAmount(getInt(vo.getTxtGivingAmountDecember()));
				dto.setGivingLimit(getInt(vo.getTxtGivingLimitDecember()));
			} else {
				dto.setGivingMonth(0);
				dto.setGivingAmount(0);
				dto.setGivingLimit(0);
			}
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生
	 */
	protected void setVoFieldsFirst(PaidHolidayFirstYearDtoInterface dto) throws MospException {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// DTOの値をVOに設定
		PaidHolidayFirstYearDtoInterface localDto = null;
		// 1月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 1);
		if (localDto == null) {
			return;
		}
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 0);
		vo.setTxtGivingTimingJanuary(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountJanuary(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitJanuary(String.valueOf(localDto.getGivingLimit()));
		// 2月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 2);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 1);
		vo.setTxtGivingTimingFebruary(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountFebruary(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitFebruary(String.valueOf(localDto.getGivingLimit()));
		// 3月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 3);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 2);
		vo.setTxtGivingTimingMarch(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountMarch(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitMarch(String.valueOf(localDto.getGivingLimit()));
		// 4月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 4);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 3);
		vo.setTxtGivingTimingApril(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountApril(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitApril(String.valueOf(localDto.getGivingLimit()));
		// 5月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 5);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 4);
		vo.setTxtGivingTimingMay(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountMay(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitMay(String.valueOf(localDto.getGivingLimit()));
		// 6月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 6);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 5);
		vo.setTxtGivingTimingJune(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountJune(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitJune(String.valueOf(localDto.getGivingLimit()));
		// 7月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 7);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 6);
		vo.setTxtGivingTimingJuly(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountJuly(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitJuly(String.valueOf(localDto.getGivingLimit()));
		// 8月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 8);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 7);
		vo.setTxtGivingTimingAugust(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountAugust(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitAugust(String.valueOf(localDto.getGivingLimit()));
		// 9月分
		localDto = timeReference().paidHolidayFirstYear()
			.findForKey(dto.getPaidHolidayCode(), getEditActivateDate(), 9);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 8);
		vo.setTxtGivingTimingSeptember(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountSeptember(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitSeptember(String.valueOf(localDto.getGivingLimit()));
		// 10月分
		localDto = timeReference().paidHolidayFirstYear().findForKey(dto.getPaidHolidayCode(), getEditActivateDate(),
				10);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 9);
		vo.setTxtGivingTimingOctober(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountOctober(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitOctober(String.valueOf(localDto.getGivingLimit()));
		// 11月分
		localDto = timeReference().paidHolidayFirstYear().findForKey(dto.getPaidHolidayCode(), getEditActivateDate(),
				11);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 10);
		vo.setTxtGivingTimingNovember(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountNovember(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitNovember(String.valueOf(localDto.getGivingLimit()));
		// 12月分
		localDto = timeReference().paidHolidayFirstYear().findForKey(dto.getPaidHolidayCode(), getEditActivateDate(),
				12);
		vo.setTmmPaidHolidayFirstYearId(localDto.getTmmPaidHolidayFirstYearId(), 11);
		vo.setTxtGivingTimingDecember(String.valueOf(localDto.getGivingMonth()));
		vo.setTxtGivingAmountDecember(String.valueOf(localDto.getGivingAmount()));
		vo.setTxtGivingLimitDecember(String.valueOf(localDto.getGivingLimit()));
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setDtoFieldsPoint(PaidHolidayPointDateDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setActivateDate(getEditActivateDate());
		dto.setPaidHolidayCode(vo.getTxtPaidHolidayCode());
		dto.setInactivateFlag(getInt(vo.getPltEditInactivate()));
		if (dto.getTimesPointDate() == 1) {
			if (!(vo.getTxtPointDateAmount1().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(0));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate1()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount1()));
			}
		}
		if (dto.getTimesPointDate() == 2) {
			if (!(vo.getTxtPointDateAmount2().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(1));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate2()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount2()));
			}
		}
		if (dto.getTimesPointDate() == 3) {
			if (!(vo.getTxtPointDateAmount3().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(2));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate3()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount3()));
			}
		}
		if (dto.getTimesPointDate() == 4) {
			if (!(vo.getTxtPointDateAmount4().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(3));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate4()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount4()));
			}
		}
		if (dto.getTimesPointDate() == 5) {
			if (!(vo.getTxtPointDateAmount5().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(4));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate5()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount5()));
			}
		}
		if (dto.getTimesPointDate() == 6) {
			if (!(vo.getTxtPointDateAmount6().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(5));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate6()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount6()));
			}
		}
		if (dto.getTimesPointDate() == 7) {
			if (!(vo.getTxtPointDateAmount7().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(6));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate7()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount7()));
			}
		}
		if (dto.getTimesPointDate() == 8) {
			if (!(vo.getTxtPointDateAmount8().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(7));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate8()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount8()));
			}
		}
		if (dto.getTimesPointDate() == 9) {
			if (!(vo.getTxtPointDateAmount9().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(8));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate9()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount9()));
			}
		}
		if (dto.getTimesPointDate() == 10) {
			if (!(vo.getTxtPointDateAmount10().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(9));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate10()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount10()));
			}
		}
		if (dto.getTimesPointDate() == 11) {
			if (!(vo.getTxtPointDateAmount11().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(10));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate11()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount11()));
			}
		}
		if (dto.getTimesPointDate() == 12) {
			if (!(vo.getTxtPointDateAmount12().isEmpty())) {
				dto.setTmmPaidHolidayPointDateId(vo.getTmmPaidHolidayPointDateId(11));
				dto.setTimesPointDate(getInt(vo.getTxtTimesPointDate12()));
				dto.setPointDateAmount(getInt(vo.getTxtPointDateAmount12()));
			}
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param holidayCode 有給休暇コード
	 * @param activeDate 有効日
	 * @throws MospException 例外処理
	 */
	protected void setVoFieldsPoint(String holidayCode, Date activeDate) throws MospException {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// DTOの値をVOに設定
		List<PaidHolidayPointDateDtoInterface> list = timeReference().paidHolidayPointDate().findForList(holidayCode,
				getEditActivateDate());
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			PaidHolidayPointDateDtoInterface localDto = list.get(i);
			vo.setTmmPaidHolidayPointDateId(localDto.getTmmPaidHolidayPointDateId(), i);
			if (localDto.getTimesPointDate() == 1) {
				// 自動付与1
				vo.setTxtPointDateAmount1(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 2) {
				// 自動付与2
				vo.setTxtPointDateAmount2(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 3) {
				// 自動付与3
				vo.setTxtPointDateAmount3(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 4) {
				// 自動付与4
				vo.setTxtPointDateAmount4(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 5) {
				// 自動付与5
				vo.setTxtPointDateAmount5(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 6) {
				// 自動付与6
				vo.setTxtPointDateAmount6(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 7) {
				// 自動付与7
				vo.setTxtPointDateAmount7(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 8) {
				// 自動付与8
				vo.setTxtPointDateAmount8(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 9) {
				// 自動付与9
				vo.setTxtPointDateAmount9(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 10) {
				// 自動付与10
				vo.setTxtPointDateAmount10(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 11) {
				// 自動付与11
				vo.setTxtPointDateAmount11(String.valueOf(localDto.getPointDateAmount()));
			}
			if (localDto.getTimesPointDate() == 12) {
				// 自動付与12
				vo.setTxtPointDateAmount12(String.valueOf(localDto.getPointDateAmount()));
			}
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setDtoFieldsEnt(PaidHolidayEntranceDateDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setActivateDate(getEditActivateDate());
		dto.setPaidHolidayCode(vo.getTxtPaidHolidayCode());
		if (!(vo.getTxtWorkYear1().isEmpty()) || !(vo.getTxtWorkMonth1().isEmpty())
				|| !(vo.getTxtJoiningDateAmount1().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(0));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear1()) * 12) + (getInt(vo.getTxtWorkMonth1())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount1()));
		}
		if (!(vo.getTxtWorkYear2().isEmpty()) || !(vo.getTxtWorkMonth2().isEmpty())
				|| !(vo.getTxtJoiningDateAmount2().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(1));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear2()) * 12) + (getInt(vo.getTxtWorkMonth2())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount2()));
		}
		if (!(vo.getTxtWorkYear3().isEmpty()) || !(vo.getTxtWorkMonth3().isEmpty())
				|| !(vo.getTxtJoiningDateAmount3().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(2));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear3()) * 12) + (getInt(vo.getTxtWorkMonth3())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount3()));
		}
		if (!(vo.getTxtWorkYear4().isEmpty()) || !(vo.getTxtWorkMonth4().isEmpty())
				|| !(vo.getTxtJoiningDateAmount4().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(3));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear4()) * 12) + (getInt(vo.getTxtWorkMonth4())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount4()));
		}
		if (!(vo.getTxtWorkYear5().isEmpty()) || !(vo.getTxtWorkMonth5().isEmpty())
				|| !(vo.getTxtJoiningDateAmount5().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(4));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear5()) * 12) + (getInt(vo.getTxtWorkMonth5())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount5()));
		}
		if (!(vo.getTxtWorkYear6().isEmpty()) || !(vo.getTxtWorkMonth6().isEmpty())
				|| !(vo.getTxtJoiningDateAmount6().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(5));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear6()) * 12) + (getInt(vo.getTxtWorkMonth6())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount6()));
		}
		if (!(vo.getTxtWorkYear7().isEmpty()) || !(vo.getTxtWorkMonth7().isEmpty())
				|| !(vo.getTxtJoiningDateAmount7().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(6));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear7()) * 12) + (getInt(vo.getTxtWorkMonth7())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount7()));
		}
		if (!(vo.getTxtWorkYear8().isEmpty()) || !(vo.getTxtWorkMonth8().isEmpty())
				|| !(vo.getTxtJoiningDateAmount8().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(7));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear8()) * 12) + (getInt(vo.getTxtWorkMonth8())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount8()));
		}
		if (!(vo.getTxtWorkYear9().isEmpty()) || !(vo.getTxtWorkMonth9().isEmpty())
				|| !(vo.getTxtJoiningDateAmount9().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(8));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear9()) * 12) + (getInt(vo.getTxtWorkMonth9())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount9()));
		}
		if (!(vo.getTxtWorkYear10().isEmpty()) || !(vo.getTxtWorkMonth10().isEmpty())
				|| !(vo.getTxtJoiningDateAmount10().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(9));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear10()) * 12) + (getInt(vo.getTxtWorkMonth10())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount10()));
		}
		if (!(vo.getTxtWorkYear11().isEmpty()) || !(vo.getTxtWorkMonth11().isEmpty())
				|| !(vo.getTxtJoiningDateAmount11().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(10));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear11()) * 12) + (getInt(vo.getTxtWorkMonth11())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount11()));
		}
		if (!(vo.getTxtWorkYear12().isEmpty()) || !(vo.getTxtWorkMonth12().isEmpty())
				|| !(vo.getTxtJoiningDateAmount12().isEmpty())) {
			dto.setTmmPaidHolidayEntranceDateId(vo.getTmmPaidHolidayEntranceDateId(11));
			dto.setWorkMonth((getInt(vo.getTxtWorkYear12()) * 12) + (getInt(vo.getTxtWorkMonth12())));
			dto.setJoiningDateAmount(getInt(vo.getTxtJoiningDateAmount12()));
		}
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param holidayCode 有給休暇コード
	 * @throws MospException 例外発生時
	 */
	protected void setVoFieldsEnt(String holidayCode) throws MospException {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		List<PaidHolidayEntranceDateDtoInterface> list = timeReference().paidHolidayEntranceDate().findForList(
				holidayCode, getEditActivateDate());
		// DTOの値をVOに設定
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			PaidHolidayEntranceDateDtoInterface localDto = list.get(i);
			vo.setTmmPaidHolidayEntranceDateId(localDto.getTmmPaidHolidayEntranceDateId(), i);
			if (i == 0) {
				vo.setTxtWorkYear1(String.valueOf(localDto.getWorkMonth() / 12));
				vo.setTxtWorkMonth1(String.valueOf(localDto.getWorkMonth() % 12));
				vo.setTxtJoiningDateAmount1(String.valueOf(localDto.getJoiningDateAmount()));
			}
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setDtoFieldsStock(StockHolidayDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setTmmStockHolidayId(vo.getTmmStockHolidayId());
		dto.setActivateDate(getEditActivateDate());
		dto.setPaidHolidayCode(vo.getTxtPaidHolidayCode());
		dto.setInactivateFlag(getInt(vo.getPltEditInactivate()));
		dto.setStockYearAmount(getInt(vo.getTxtStockYearAmount()));
		dto.setStockTotalAmount(getInt(vo.getTxtStockTotalAmount()));
		dto.setStockLimitDate(getInt(vo.getTxtStockLimitDate()));
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setVoFieldsStock(StockHolidayDtoInterface dto) {
		// VO取得
		PaidHolidayCardVo vo = (PaidHolidayCardVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setTmmStockHolidayId(dto.getTmmStockHolidayId());
		vo.setTxtStockYearAmount(String.valueOf(dto.getStockYearAmount()));
		vo.setTxtStockTotalAmount(String.valueOf(dto.getStockTotalAmount()));
		vo.setTxtStockLimitDate(String.valueOf(dto.getStockLimitDate()));
	}
	
}
