/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.management.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.property.MospProperties;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.system.SectionReferenceBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.RequestSearchBeanInterface;
import jp.mosp.time.comparator.settings.ManagementRequestRequestDateComparator;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.ManagementRequestListDtoInterface;
import jp.mosp.time.input.action.ApprovalHistoryAction;
import jp.mosp.time.management.vo.RequestListVo;
import jp.mosp.time.settings.base.TimeSettingAction;

/**
 * 勤怠管理権限者が自分の部下の各種申請承認情報の検索と確認を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 */
public class RequestListAction extends TimeSettingAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 初期表示を行う。<br>
	 */
	public static final String	CMD_SHOW				= "TM2200";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力された各種情報項目を基に検索を行い、条件に沿ったアカウント情報の一覧表示を行う。<br>
	 * 一覧表示の際には有効日でソートを行う。<br>
	 */
	public static final String	CMD_SEARCH				= "TM2202";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * パンくずリスト等を用いてこの画面よりも奥の階層から改めて遷移した場合、
	 * 各種情報の登録状況が更新された状態で検索し、再表示を行う。<br>
	 */
	public static final String	CMD_RE_SHOW				= "TM2203";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。<br>
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT				= "TM2208";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE				= "TM2209";
	
	/**
	 * 表示期間決定コマンド。<br>
	 * <br>
	 * 各プルダウン情報についてテキストボックスに入力した表示期間で検索を行って情報を取得する。<br>
	 * それらの情報から選択可能なロールのプルダウンリストを作成し、検索項目にセットする。<br>
	 * 表示期間決定後、表示期間は編集不可になる。<br>
	 */
	public static final String	CMD_SET_ACTIVATION_DATE	= "TM2290";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public RequestListAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new RequestListVo();
	}
	
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ソート
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_ACTIVATION_DATE)) {
			// 表示期間決定
			prepareVo();
			setActivateDate();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// VO取得
		RequestListVo vo = (RequestListVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 表示期間（年）
		vo.setAryPltSearchRequestYear(getYearArray(String.valueOf(DateUtility.getYear(DateUtility.getSystemDate()))));
		// 表示期間（月）
		vo.setAryPltSearchRequestMonth(getMonthArray());
		// 表示期間（日）
		vo.setAryPltSearchRequestDay(getDayArray(31));
		vo.setPltSearchRequestYear(String.valueOf(DateUtility.getYear(DateUtility.getSystemDate())));
		vo.setPltSearchRequestMonth(String.valueOf(DateUtility.getMonth(DateUtility.getSystemDate())));
		vo.setPltSearchRequestDay(String.valueOf(DateUtility.getDay(DateUtility.getSystemDate())));
		// プルダウン設定
		setPulldown();
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// ソートキー設定
		vo.setComparatorName(ManagementRequestRequestDateComparator.class.getName());
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合
	 */
	protected void search() throws MospException {
		// VO準備
		RequestListVo vo = (RequestListVo)mospParams.getVo();
		// 検索クラス取得
		RequestSearchBeanInterface search = timeReference().requestSearch();
		// VOの値を検索クラスへ設定
		int year = Integer.parseInt(vo.getPltSearchRequestYear());
		int month = Integer.parseInt(vo.getPltSearchRequestMonth());
		Date firstDate = DateUtility.getFirstDateOfMonth(year, month);
		search.setTargetDate(firstDate);
		search.setStartDate(firstDate);
		search.setEndDate(DateUtility.getLastDateOfMonth(year, month));
		if (!vo.getPltSearchRequestDay().isEmpty()) {
			Date targetDate = DateUtility.getDate(vo.getPltSearchRequestYear(), vo.getPltSearchRequestMonth(), vo
				.getPltSearchRequestDay());
			search.setTargetDate(targetDate);
			search.setStartDate(targetDate);
			search.setEndDate(targetDate);
		}
		search.setApprovalType(vo.getPltSearchApprovalType());
		search.setState(vo.getPltSearchState());
		search.setEmployeeCode(vo.getTxtSearchEmployeeCode());
		search.setEmployeeName(vo.getTxtSearchEmployeeName());
		search.setWorkPlaceCode(vo.getPltSearchWorkPlace());
		search.setEmploymentContractCode(vo.getPltSearchEmployment());
		search.setSectionCode(vo.getPltSearchSection());
		search.setPositionCode(vo.getPltSearchPosition());
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<ManagementRequestListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(ManagementRequestRequestDateComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.size() == 0) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * 表示期間設定処理を行う。<br>
	 * 保持有効日モードを確認し、モード及びプルダウンの再設定を行う。<br>
	 * @throws MospException プルダウンの取得に失敗した場合
	 */
	protected void setActivateDate() throws MospException {
		// VO取得
		RequestListVo vo = (RequestListVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
			// 勤務地/雇用契約/所属/職位プルダウンの初期化
			vo.setPltSearchWorkPlace("0");
			vo.setPltSearchEmployment("0");
			vo.setPltSearchSection("0");
			vo.setPltSearchPosition("0");
			vo.setPltSearchEmployment("0");
		}
		// プルダウン取得
		setPulldown();
	}
	
	/**
	 * 初期値を設定する。<br>
	 */
	public void setDefaultValues() {
		// VO準備
		RequestListVo vo = (RequestListVo)mospParams.getVo();
		
		vo.setPltSearchRequestYear("");
		vo.setPltSearchRequestMonth("");
		vo.setPltSearchRequestDay("");
		vo.setPltSearchApprovalType("");
		vo.setPltSearchState("");
		vo.setTxtSearchEmployeeCode("");
		vo.setTxtSearchEmployeeName("");
		vo.setPltSearchWorkPlace("");
		vo.setPltSearchEmployment("");
		vo.setPltSearchSection("");
		vo.setPltSearchPosition("");
		// 有効日モード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		RequestListVo vo = (RequestListVo)mospParams.getVo();
		Date date = DateUtility.getFirstDateOfMonth(Integer.parseInt(vo.getPltSearchRequestYear()), Integer.parseInt(vo
			.getPltSearchRequestMonth()));
		if (!vo.getPltSearchRequestDay().isEmpty()) {
			date = DateUtility.getDate(vo.getPltSearchRequestYear(), vo.getPltSearchRequestMonth(), vo
				.getPltSearchRequestDay());
		}
		// プルダウンの設定
		MospProperties properties = mospParams.getProperties();
		String[][] aryApprovalType = properties.getCodeArray(TimeConst.CODE_APPROVAL_TYPE, true);
		String[][] aryState = properties.getCodeArray(TimeConst.CODE_STATE, true);
		// 申請カテゴリ
		vo.setAryPltSearchApprovalType(aryApprovalType);
		// 状態
		vo.setAryPltSearchState(aryState);
		// 勤務地
		String[][] aryWorkPlace = reference().workPlace().getNameSelectArray(date, true, null);
		vo.setAryPltSearchWorkPlace(aryWorkPlace);
		// 雇用契約
		String[][] aryEmployment = reference().employmentContract().getNameSelectArray(date, true, null);
		vo.setAryPltSearchEmployment(aryEmployment);
		// 所属
		String[][] aryPltSection = reference().section().getNameSelectArray(date, true, null);
		vo.setAryPltSearchSection(aryPltSection);
		// 職位
		String[][] aryPltPosition = reference().position().getNameSelectArray(date, true, null);
		vo.setAryPltSearchPosition(aryPltPosition);
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		RequestListVo vo = (RequestListVo)mospParams.getVo();
		// データ配列初期化
		String[] aryLblEmployeeCode = new String[list.size()];
		String[] aryLblEmployeeName = new String[list.size()];
		String[] aryLblSection = new String[list.size()];
		String[] aryLblRequestType = new String[list.size()];
		String[] aryLblRequestDate = new String[list.size()];
		String[] aryLblRequestInfo = new String[list.size()];
		String[] aryLblState = new String[list.size()];
		String[] aryWorkflow = new String[list.size()];
		String[] aryStage = new String[list.size()];
		String[] aryState = new String[list.size()];
		String[] aryLblRequestTypeCmd = new String[list.size()];
		String[] aryLblHistoryCmd = new String[list.size()];
		String[] aryLblRequestFunctionCode = new String[list.size()];
		String[] aryRecordId = new String[list.size()];
		String[] aryBackColor = new String[list.size()];
		SectionReferenceBeanInterface section = reference().section();
		Date date = DateUtility.getSystemDate();
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			ManagementRequestListDtoInterface dto = (ManagementRequestListDtoInterface)list.get(i);
			// 配列に情報を設定
			aryLblEmployeeCode[i] = dto.getEmployeeCode();
			aryLblEmployeeName[i] = dto.getLastName() + dto.getFirstName();
			aryLblSection[i] = section.getSectionAbbr(dto.getSectionCode(), date);
			aryLblRequestType[i] = getRequestTypeValueView(dto.getRequestType());
			// カテゴリを詳細画面のCmdIdに変換する
			String[] aryHistoryCmd = getRequestTypeCmd(aryLblRequestType[i]);
			aryLblRequestTypeCmd[i] = aryHistoryCmd[0];
			aryLblHistoryCmd[i] = aryHistoryCmd[1];
			aryLblRequestFunctionCode[i] = aryHistoryCmd[2];
			aryLblRequestDate[i] = DateUtility.getStringDateAndDay(dto.getRequestDate());
			aryLblRequestInfo[i] = dto.getRequestInfo();
			aryLblState[i] = getStatusStageValueView(dto.getState(), dto.getStage());
			aryWorkflow[i] = String.valueOf(dto.getWorkflow());
			aryStage[i] = String.valueOf(dto.getStage());
			aryState[i] = dto.getState();
			aryRecordId[i] = String.valueOf(dto.getTmdManagementRequestListId());
			aryBackColor[i] = setBackColor(getPersonalId(dto.getEmployeeCode()), dto.getRequestDate(), aryHistoryCmd[2]);
		}
		// データをVOに設定
		vo.setAryLblEmployeeCode(aryLblEmployeeCode);
		vo.setAryLblEmployeeName(aryLblEmployeeName);
		vo.setAryLblSection(aryLblSection);
		vo.setAryLblRequestType(aryLblRequestType);
		vo.setAryLblRequestDate(aryLblRequestDate);
		vo.setAryLblRequestInfo(aryLblRequestInfo);
		vo.setAryLblState(aryLblState);
		vo.setAryRequestTypeCmd(aryLblRequestTypeCmd);
		vo.setAryHistoryCmd(aryLblHistoryCmd);
		vo.setAryRequestFunctionCode(aryLblRequestFunctionCode);
		vo.setAryWorkflow(aryWorkflow);
		vo.setAryStage(aryStage);
		vo.setAryState(aryState);
		vo.setAryRecordId(aryRecordId);
		vo.setAryBackColor(aryBackColor);
	}
	
	/**
	 * カテゴリコードから詳細画面のCmdIdを返す。<br>
	 * @param requestType カテゴリコード
	 * @return 詳細画面へ遷移するためのCmdId
	 */
	protected String[] getRequestTypeCmd(String requestType) {
		String aryRequestTypeCmd[] = new String[3];
		if (requestType.equals(mospParams.getName("WorkManage"))) {
			// 勤怠承認
			aryRequestTypeCmd[0] = ApprovalCardAction.CMD_APPROVAL_CONFIRMATION_ATTENDANCE;
			aryRequestTypeCmd[1] = ApprovalHistoryAction.CMD_TIME_APPROVAL_HISTORY_SELECT_SHOW;
			aryRequestTypeCmd[2] = TimeConst.CODE_FUNCTION_WORK_MANGE;
		} else if (requestType.equals(mospParams.getName("OvertimeWork"))) {
			// 残業承認
			aryRequestTypeCmd[0] = ApprovalCardAction.CMD_APPROVAL_CONFIRMATION_OVERTIME;
			aryRequestTypeCmd[1] = ApprovalHistoryAction.CMD_OVERTIME_APPROVAL_HISTORY_SELECT_SHOW;
			aryRequestTypeCmd[2] = TimeConst.CODE_FUNCTION_OVER_WORK;
		} else if (requestType.equals(mospParams.getName("Vacation"))) {
			// 休暇承認
			aryRequestTypeCmd[0] = ApprovalCardAction.CMD_APPROVAL_CONFIRMATION_HOLIDAY;
			aryRequestTypeCmd[1] = ApprovalHistoryAction.CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW;
			aryRequestTypeCmd[2] = TimeConst.CODE_FUNCTION_VACATION;
		} else if (requestType.equals(mospParams.getName("WorkingHoliday"))) {
			// 休日出勤承認
			aryRequestTypeCmd[0] = ApprovalCardAction.CMD_APPROVAL_CONFIRMATION_WORKONHOLIDAY;
			aryRequestTypeCmd[1] = ApprovalHistoryAction.CMD_APPROVAL_HISTORY_HOLIDAY_WORK_SELECT_SHOW;
			aryRequestTypeCmd[2] = TimeConst.CODE_FUNCTION_WORK_HOLIDAY;
		} else if (requestType.equals(mospParams.getName("CompensatoryHoliday"))) {
			// 代休承認
			aryRequestTypeCmd[0] = ApprovalCardAction.CMD_APPROVAL_CONFIRMATION_SUBHOLIDAY;
			aryRequestTypeCmd[1] = ApprovalHistoryAction.CMD_APPROVAL_HISTORY_LIEU_SELECT_SHOW;
			aryRequestTypeCmd[2] = TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY;
		} else if (requestType.equals(mospParams.getName("TimeDifference"))) {
			// 時差出勤承認
			aryRequestTypeCmd[0] = ApprovalCardAction.CMD_APPROVAL_CONFIRMATION_DIFFERENCE;
			aryRequestTypeCmd[1] = ApprovalHistoryAction.CMD_DIFFERENCE_WORK_APPROVAL_HISTORY_SELECT_SHOW;
			aryRequestTypeCmd[2] = TimeConst.CODE_FUNCTION_DIFFERENCE;
		}
		return aryRequestTypeCmd;
	}
}
