/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.OvertimeInfoReferenceBeanInterface;
import jp.mosp.time.bean.OvertimeRequestSearchBeanInterface;
import jp.mosp.time.comparator.settings.OvertimeRequestRequestDateComparator;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestListDtoInterface;
import jp.mosp.time.input.vo.OvertimeRequestVo;

/**
 * 残業申請情報の確認と編集を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_DRAFT}
 * </li><li>
 * {@link #CMD_APPLI}
 * </li><li>
 * {@link #CMD_WITHDRAWN}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_INSERT_MODE}
 * </li><li>
 * {@link #CMD_EDIT_MODE}
 * </li><li>
 */
public class OvertimeRequestAction extends TimeAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 現在ログインしているユーザの残業申請画面を表示する。<br>
	 */
	public static final String	CMD_SHOW				= "TM1400";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力した情報を元に残業申請情報の検索を行う。<br>
	 */
	public static final String	CMD_SEARCH				= "TM1402";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * 一度この画面を表示してからパンくずリスト等で再度遷移する場合は保持していた検索条件で検索を行って再表示する。<br>
	 */
	public static final String	CMD_RE_SHOW				= "TM1403";
	
	/**
	 * 下書きマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を残業情報テーブルに登録し、下書情報として保存する。<br>
	 */
	public static final String	CMD_DRAFT				= "TM1404";
	
	/**
	 * 申請コマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を残業情報テーブルに登録し、残業申請を行う。以降、このレコードは上長が差戻をしない限り編集不可となる。<br>
	 * 残業の申請可能時間を超過している、申請年月日で申請不可な日付が選択されている、<br>
	 * 申請時間が0時間0分のまま、残業理由の項目が入力されていない、<br>
	 * といった状態で申請を行おうとした場合は場合はエラーメッセージにて通知し、申請は実行されない。<br>
	 */
	public static final String	CMD_APPLI				= "TM1405";
	
	/**
	 * 取下コマンド。<br>
	 * <br>
	 * 下書状態または差戻状態で登録されていたレコードの取下を行う。<br>
	 * 取下後、対象の残業申請情報は未申請状態へ戻る。<br>
	 */
	public static final String	CMD_WITHDRAWN			= "TM1407";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。<br>
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT				= "TM1408";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE				= "TM1409";
	
	/**
	 * 残業年月日決定コマンド。<br>
	 * <br>
	 * 残業年月日欄に入力されている日付を基にその日時点で申請者に紐付けられている承認者情報を取得し、
	 * それぞれの承認段階のプルダウンで選択可能な状態とする。<br>
	 * 決定した日付時点で申請者がどの承認者にも紐付けられていない場合はエラーメッセージを表示し、
	 * 処理を中断する。<br>
	 */
	public static final String	CMD_SET_OVERTIME_DATE	= "TM1490";
	
	/**
	 * 新規登録モード切替コマンド。<br>
	 * <br>
	 * 申請テーブルの各入力欄に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_INSERT_MODE			= "TM1491";
	
	/**
	 * 編集モード切替コマンド。<br>
	 * <br>
	 * 選択したレコードの内容を申請テーブルの各入力欄にそれぞれ表示させる。申請テーブルヘッダに新規登録モード切替リンクを表示させる。
	 * に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_EDIT_MODE			= "TM1492";
	
	/**
	 * 一括更新コマンド。<br>
	 * <br>
	 * 一覧にて選択チェックボックスにチェックが入っている下書状態のレコードの申請処理を繰り返し行う。<br>
	 * ひとつもチェックが入っていない状態で一括更新ボタンをクリックした場合はエラーメッセージにて通知し、処理を中断する。<br>
	 */
	public static final String	CMD_BATCH_UPDATE		= "TM1495";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public OvertimeRequestAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new OvertimeRequestVo();
	}
	
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_DRAFT)) {
			// 下書
			prepareVo();
			draft();
		} else if (mospParams.getCommand().equals(CMD_APPLI)) {
			// 申請
			prepareVo();
			appli();
		} else if (mospParams.getCommand().equals(CMD_WITHDRAWN)) {
			// 取下
			prepareVo();
			withdrawn();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート 
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_OVERTIME_DATE)) {
			// 残業年月日決定
			prepareVo();
			setOvertimeDate();
		} else if (mospParams.getCommand().equals(CMD_INSERT_MODE)) {
			// 新規モード切替
			prepareVo();
			insertMode();
		} else if (mospParams.getCommand().equals(CMD_EDIT_MODE)) {
			// 編集モード切替
			prepareVo();
			editMode();
		} else if (mospParams.getCommand().equals(CMD_BATCH_UPDATE)) {
			// 一括更新
			prepareVo();
			batchUpdate();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		setEmployeeParams();
		// 新規登録モード設定
		insertMode();
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void search() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// 検索クラス取得
		OvertimeRequestSearchBeanInterface search = timeReference().overtimeRequestSearch();
		// VOの値を検索クラスへ設定
		search.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		search.setWorkflowStatus(vo.getPltSearchState());
		search.setScheduleOver(vo.getPltSearchScheduleOver());
		search.setOvertimeType(vo.getPltSearchOverTimeType());
		int year = Integer.parseInt(vo.getPltSearchRequestYear());
		int startMonth = 0;
		int endMonth = 0;
		if (vo.getPltSearchRequestMonth().isEmpty()) {
			startMonth = 1;
			endMonth = 12;
		} else {
			startMonth = Integer.parseInt(vo.getPltSearchRequestMonth());
			endMonth = startMonth;
		}
		search.setRequestStartDate(DateUtility.getDate(year, startMonth, 1));
		search.setRequestEndDate(DateUtility.getDate(year, endMonth, getEndDay(year, endMonth)));
		
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<OvertimeRequestListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(OvertimeRequestRequestDateComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.size() == 0) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	* 下書処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void draft() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// DTOの準備
		OvertimeRequestDtoInterface dto = timeReference().overtimeRequest().findForKey(vo.getRecordId());
		if (dto == null) {
			dto = time().overtimeRequestRegist().getInitDto();
		}
		setOvertimeDtoFields(dto);
		// 妥当性チェック
		time().overtimeRequestRegist().validate(dto);
		// 申請の相関チェック
		time().overtimeRequestRegist().checkDraft(dto);
		// 登録クラスの取得
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto == null) {
			workflowDto = workflowRegist.getInitDto();
			workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_OVER_WORK);
		}
		workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
		// 登録後ワークフローの取得
		workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
				PlatformConst.WORKFLOW_TYPE_TIME);
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					dto.getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("WorkPaper") }));
			// ワークフロー番号セット
			dto.setWorkflow(workflowDto.getWorkflow());
			// 登録
			time().overtimeRequestRegist().regist(dto);
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 下書成功メッセージ設定
		addDraftMessage();
		// 検索
		search();
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
	}
	
	/**
	 * 申請処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void appli() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// DTOの準備
		OvertimeRequestDtoInterface dto = timeReference().overtimeRequest().findForKey(vo.getRecordId());
		if (dto == null) {
			dto = time().overtimeRequestRegist().getInitDto();
		}
		setOvertimeDtoFields(dto);
		// 申請の相関チェック
		time().overtimeRequestRegist().checkAppli(dto);
		// 登録クラスの取得
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto == null) {
			workflowDto = workflowRegist.getInitDto();
			workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_OVER_WORK);
		}
		workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
		// 登録後ワークフローの取得
		workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
				PlatformConst.WORKFLOW_TYPE_TIME);
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(workflowDto, dto.getPersonalId(),
					chkApproverSettingCommentName());
			// ワークフロー番号セット
			dto.setWorkflow(workflowDto.getWorkflow());
			// 登録
			time().overtimeRequestRegist().regist(dto);
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 申請成功メッセージ設定
		addAppliMessage();
		// 登録結果確認
		if (!mospParams.hasErrorMessage()) {
			// 登録が成功した場合、初期状態に戻す。
			insertMode();
			search();
		}
	}
	
	/**
	* 取下処理を行う。<br>
	 * @throws MospException 例外発生時
	*/
	protected void withdrawn() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// DTOの準備
		OvertimeRequestDtoInterface dto = timeReference().overtimeRequest().findForKey(vo.getRecordId());
		// 存在確認
		checkSelectedDataExist(dto);
		// 取下の相関チェック
		time().overtimeRequestRegist().checkWithdrawn(dto);
		// ワークフロー取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		// 存在確認
		checkSelectedDataExist(workflowDto);
		// ワークフロー登録
		workflowDto = platform().workflowRegist().withdrawn(workflowDto, dto.getPersonalId());
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					dto.getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("TakeDown") }));
		}
		// 取下結果確認
		if (mospParams.hasErrorMessage()) {
			// 取下失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		// 取り成功メッセージ設定
		addTakeDownMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 */
	protected void page() {
		setVoList(pageList());
	}
	
	/**
	 * 有効日決定処理を行う。<br>
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void setOvertimeDate() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		if (vo.getModeActivateDate().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			// プルダウン初期化
			vo.setAryPltLblApproverSetting(new String[0]);
			// 有効日モード設定
			vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
			return;
		}
		// プルダウン作成
		if (setApprvalPullDown(getPersonalId(vo.getLblEmployeeCode()), getEditRequestDate()) == false) {
			// 決定失敗メッセージ設定
			addFixFailedMessage();
			return;
		}
		// 有効日のモード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void insertMode() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// プルダウン設定
		setPulldown();
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNCLAIMED);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(OvertimeRequestRequestDateComparator.class.getName());
		// 基本情報チェック
//		timeReference().overtimeRequest().chkBasicInfo(vo.getPersonalId(), getEditRequestDate());
	}
	
	/**
	 * 履歴編集モードで画面を表示する。<br>
	 * 履歴編集対象は、遷移汎用コード及び有効日で取得する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void editMode() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// だるま連携用設定
		setPageInfo(CMD_PAGE, getListLength());
		setEmployeeParams();
		insertMode();
		// 遷移汎用コード及び有効日から履歴編集対象を取得し編集モードを設定
		setEditUpdateMode(vo.getLblEmployeeCode(), getDate(getTransferredActivateDate()), Integer
			.valueOf(getTransferredType()));
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
		// 有効日のモード設定
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		setOvertimeDate();
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void batchUpdate() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// 一括更新処理
		time().overtimeRequestRegist().update(getIdArray(vo.getCkbSelect()));
		// 一括更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addBatchUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addUpdateMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		OvertimeInfoReferenceBeanInterface overtimeInfo = timeReference().overtimeInfo();
		Date date = DateUtility.getSystemDate();
		// 編集項目
		vo.setRecordId(0);
		if (getTransferredYear() == null) {
			vo.setPltEditRequestYear(DateUtility.getStringYear(date));
			vo.setPltEditRequestMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditRequestDay(String.valueOf(DateUtility.getDay(date)));
		} else {
			vo.setPltEditRequestYear(getTransferredYear());
			vo.setPltEditRequestMonth(getTransferredMonth());
			vo.setPltEditRequestDay(getTransferredDay());
		}
		vo.setPltEditRequestHour("0");
		vo.setPltEditRequestMinute("0");
		if (getTransferredHour() != null) {
			vo.setPltEditRequestHour(getTransferredHour());
			vo.setPltEditRequestMinute(getTransferredMinute());
		}
		vo.setPltEditOverTimeType(Integer.toString(2));
		vo.setTxtEditRequestReason("");
		// 検索項目
		vo.setPltSearchState("");
		vo.setPltSearchScheduleOver("");
		vo.setPltSearchScheduleOver("");
		vo.setPltSearchRequestYear(DateUtility.getStringYear(date));
		vo.setPltSearchRequestMonth("");
		// 状態
		vo.setLblSection("");
		// 承認者欄の初期化
		String[] aryPltLblApproverSetting = new String[0];
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		vo.setPltApproverSetting1("");
		vo.setPltApproverSetting2("");
		vo.setPltApproverSetting3("");
		vo.setPltApproverSetting4("");
		vo.setPltApproverSetting5("");
		vo.setPltApproverSetting6("");
		vo.setPltApproverSetting7("");
		vo.setPltApproverSetting8("");
		vo.setPltApproverSetting9("");
		vo.setPltApproverSetting10("");
		vo.setModeActivateDate(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		
		vo.setJsBeforeOvertimeFlag("0");
		
		// 申請可能時間表示
		vo.setLblRemainderWeek(getTimeTimeFormat(0));
		vo.setLblRemainderMonth(getTimeTimeFormat(0));
		String remainderWeek = getTimeTimeFormat(overtimeInfo.getPossibleTime1Week(getPersonalId(vo
			.getLblEmployeeCode())));
		if (mospParams.hasErrorMessage()) {
			return;
		}
		String remainderMonth = getTimeTimeFormat(overtimeInfo.getPossibleTime1Month(getPersonalId(vo
			.getLblEmployeeCode())));
		if (mospParams.hasErrorMessage()) {
			return;
		}
		vo.setLblRemainderWeek(remainderWeek);
		vo.setLblRemainderMonth(remainderMonth);
		// 
		Integer beforeOvertimeFlag = timeReference().timeSetting().getBeforeOvertimeFlag(vo.getPersonalId(), date);
		// 勤務前残業のチェック
		if (null != beforeOvertimeFlag && 1 == beforeOvertimeFlag) {
			// 無効に設定
			vo.setJsBeforeOvertimeFlag("1");
		}
	}
	
	/**
	 * プルダウン設定
	 */
	private void setPulldown() {
		// VO準備
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// システム日付取得
		String editRequestYear;
		Date date = DateUtility.getSystemDate();
		// 編集項目設定
		editRequestYear = DateUtility.getStringYear(date);
		vo.setAryPltEditRequestYear(getYearArray(editRequestYear));
		vo.setAryPltEditRequestMonth(getMonthArray());
		// 31日まで取得
		vo.setAryPltEditRequestDay(getDayArray(31));
		vo.setAryPltEditRequestHour(getHourArray());
		vo.setAryPltEditRequestMinute(getMinuteArray());
		vo.setAryPltEditOverTimeType(mospParams.getProperties().getCodeArray(TimeConst.CODE_OVER_TIME_TYPE, false));
		// 検索項目設定
		vo.setAryPltSearchState(mospParams.getProperties().getCodeArray(TimeConst.CODE_APPROVAL_STATE, true));
		vo.setAryPltSearchScheduleOver(mospParams.getProperties().getCodeArray(TimeConst.CODE_SCHEDULE_OVER, true));
		vo.setAryPltSearchOverTimeType(mospParams.getProperties().getCodeArray(TimeConst.CODE_OVER_TIME_TYPE, true));
		vo.setAryPltSearchRequestYear(getYearArray(editRequestYear));
		vo.setAryPltSearchRequestMonth(addArrayFirstRowBlank(getMonthArray()));
	}
	
	/**
	 * 履歴編集モードを設定する。<br>
	 * 雇用契約コードと有効日で編集対象情報を取得する。<br>
	 * @param employmentContractCode 雇用契約コード
	 * @param activateDate 有効日
	 * @param overTimeType 残業区分
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditUpdateMode(String employmentContractCode, Date activateDate, int overTimeType)
			throws MospException {
		// 履歴編集対象取得
		OvertimeRequestDtoInterface dto = timeReference().overtimeRequest().findForKey(
				getPersonalId(employmentContractCode), activateDate, overTimeType);
		// 存在確認
		checkSelectedDataExist(dto);
		// VOにセット
		setVoFields(dto);
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setVoFields(OvertimeRequestDtoInterface dto) throws MospException {
		// VO取得
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// DTOの値をVOに設定
		vo.setRecordId(dto.getTmdOvertimeRequestId());
		vo.setTmdOvertimeRequestId(dto.getTmdOvertimeRequestId());
		vo.setPltEditRequestYear(String.valueOf(DateUtility.getYear(dto.getRequestDate())));
		vo.setPltEditRequestMonth(String.valueOf(DateUtility.getMonth(dto.getRequestDate())));
		vo.setPltEditRequestDay(String.valueOf(DateUtility.getDay(dto.getRequestDate())));
		vo.setPltEditRequestHour(convIntegerTimeToIntegerHour(dto.getRequestTime()));
		vo.setPltEditRequestMinute(convIntegerTimeToIntegerMinutes(dto.getRequestTime()));
		vo.setPltEditOverTimeType(String.valueOf(dto.getOvertimeType()));
		vo.setTxtEditRequestReason(dto.getRequestReason());
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setJsStatusApplication(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) {
		// VO取得
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// データ配列初期化
		String[] aryCkbRecordId = new String[list.size()];
		String[] aryLblRequestDate = new String[list.size()];
		String[] aryLblOvertimeTypeName = new String[list.size()];
		String[] aryLblOvertimeTypeCode = new String[list.size()];
		String[] aryLblRequestTime = new String[list.size()];
		String[] aryLblResultTime = new String[list.size()];
		String[] aryLblRequestReason = new String[list.size()];
		String[] aryLblWorkflowStatus = new String[list.size()];
		String[] aryLblApproverName = new String[list.size()];
		String[] aryLblOnOff = new String[list.size()];
		// データ作成
		int i = 0;
		for (BaseDtoInterface baseDto : list) {
			// リストから情報を取得
			OvertimeRequestListDtoInterface dto = (OvertimeRequestListDtoInterface)baseDto;
			// 配列に情報を設定
			aryCkbRecordId[i] = String.valueOf(dto.getTmdOvertimeRequestId());
			aryLblRequestDate[i] = DateUtility.getStringDateAndDay(dto.getRequestDate());
			aryLblOvertimeTypeName[i] = getOvertimeTypeName(dto.getOvertimeType());
			aryLblOvertimeTypeCode[i] = String.valueOf(dto.getOvertimeType());
			aryLblRequestTime[i] = getTimeDotFormat(dto.getRequestTime());
			aryLblResultTime[i] = dto.getResultsTime();
			aryLblRequestReason[i] = dto.getRequestReason();
			aryLblWorkflowStatus[i] = getStatusStageValueView(dto.getState(), dto.getStage());
			aryLblApproverName[i] = dto.getApproverName();
			aryLblOnOff[i] = getButtonOnOff(dto.getState());
			i++;
		}
		// データをVOに設定
		vo.setAryCkbOvertimeRequestListId(aryCkbRecordId);
		vo.setAryLblDate(aryLblRequestDate);
		vo.setAryLblOverTimeTypeName(aryLblOvertimeTypeName);
		vo.setAryLblOverTimeTypeCode(aryLblOvertimeTypeCode);
		vo.setAryLblScheduleTime(aryLblRequestTime);
		vo.setAryLblResultTime(aryLblResultTime);
		vo.setAryLblRequestReason(aryLblRequestReason);
		vo.setAryLblState(aryLblWorkflowStatus);
		vo.setAryLblApprover(aryLblApproverName);
		vo.setAryLblOnOff(aryLblOnOff);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setOvertimeDtoFields(OvertimeRequestDtoInterface dto) throws MospException {
		// VO取得
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setTmdOvertimeRequestId(vo.getRecordId());
		dto.setPersonalId(reference().human().getPersonalId(vo.getLblEmployeeCode(), getEditRequestDate()));
		dto.setRequestDate(getEditRequestDate());
		int workRequestTime = (Integer.parseInt(vo.getPltEditRequestHour()) * TimeConst.CODE_DEFINITION_HOUR)
				+ Integer.parseInt(vo.getPltEditRequestMinute());
		dto.setRequestTime(workRequestTime);
		dto.setOvertimeType(Integer.parseInt(vo.getPltEditOverTimeType()));
		dto.setRequestReason(vo.getTxtEditRequestReason());
	}
	
	/**
	 * @return 文字列となっているリクエスト年月日をDate型へ変換して返す。
	 * @throws MospException 例外発生時
	 */
	protected Date getEditRequestDate() throws MospException {
		// VO取得
		OvertimeRequestVo vo = (OvertimeRequestVo)mospParams.getVo();
		return DateUtility.getDate(vo.getPltEditRequestYear(), vo.getPltEditRequestMonth(), vo.getPltEditRequestDay());
	}
}
