/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.DifferenceRequestRegistBeanInterface;
import jp.mosp.time.bean.DifferenceRequestSearchBeanInterface;
import jp.mosp.time.comparator.settings.DifferenceRequestRequestDateComparator;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestListDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.WorkTypeDtoInterface;
import jp.mosp.time.dto.settings.WorkTypeItemDtoInterface;
import jp.mosp.time.input.vo.DifferenceRequestVo;

/**
 * 時差出勤申請情報の確認と編集を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_RE_SEARCH}
 * </li><li>
 * {@link #CMD_DRAFT}
 * </li><li>
 * {@link #CMD_APPLI}
 * </li><li>
 * {@link #CMD_WITHDRAWN}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_INSERT_MODE}
 * </li><li>
 * {@link #CMD_EDIT_MODE}
 * </li><li>
 */
public class DifferenceRequestAction extends TimeAction {
	
	/**
	 * 表示コマンド。<br>
	 * <br>
	 * 現在ログインしているユーザの時差出勤申請画面を表示する。<br>
	 */
	public static final String	CMD_SHOW				= "TM1000";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力した情報を元に時差出勤申請情報の検索を行う。
	 */
	public static final String	CMD_SEARCH				= "TM1002";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * 一度この画面を表示した後、パンくずリスト等を用いて他の画面から改めて遷移した場合、<br>
	 * 各種情報の登録状況を更新した上で保持していた検索条件で検索を行って画面を再表示する。<br>
	 */
	public static final String	CMD_RE_SEARCH			= "TM1003";
	
	/**
	 * 下書きコマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を時差出勤情報テーブルに登録し、下書情報として保存する。<br>
	 */
	public static final String	CMD_DRAFT				= "TM1004";
	
	/**
	 * 申請コマンド。<br>
	 * <br>
	 * 申請欄に入力した内容を時差出勤情報テーブルに登録し、時差出勤申請を行う。以降、このレコードは上長が差戻をしない限り編集不可となる。<br>
	 * 開始年月日で申請不可な日付が選択されている、申請時間が0時間0分のまま、申請理由の項目が入力されていない、<br>
	 * 下といった状態で申請を行おうとした場合は場合はエラーメッセージにて通知し、申請は実行されない。<br>
	 */
	public static final String	CMD_APPLI				= "TM1005";
	
	/**
	 * 勤務時刻再計算コマンド。<br>
	 * <br>
	 * 決定した出勤日の勤務形態をカレンダより取得し、時差出勤開始時刻にその勤務形態の勤務時間と休憩時間を加えた時刻を時差出勤終了時刻欄に表示する。<br>
	 */
	public static final String	CMD_WORKTIME_CALC		= "TM1006";
	
	/**
	 * 取下コマンド。<br>
	 * <br>
	 * 下書状態または差戻状態で登録されていたレコードの取下を行う。取下後、対象の時差出勤申請情報は未申請状態へ戻る。<br>
	 */
	public static final String	CMD_WITHDRAWN			= "TM1007";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT				= "TM1008";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE				= "TM1009";
	
	/**
	 * 出勤日決定コマンド。<br>
	 * <br>
	 * 入力した時差出勤日の勤務形態をカレンダより取得して略称をラベルに出力する。<br>
	 * 勤務開始時刻は勤務形態プルダウンで選択したレコードの出勤時刻が入力されている状態にする。<br>
	 */
	public static final String	CMD_SET_ACTIVATION_DATE	= "TM1090";
	
	/**
	 * 新規登録モード切替コマンド。<br>
	 * <br>
	 * 申請テーブルの各入力欄に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_INSERT_MODE			= "TM1091";
	
	/**
	 * 編集モード切替コマンド。<br>
	 * <br>
	 * 選択したレコードの内容を申請テーブルの各入力欄にそれぞれ表示させる。申請テーブルヘッダに新規登録モード切替リンクを表示させる。
	 * に表示されているレコード内容をクリアにする。<br>
	 * 申請テーブルヘッダに表示されている新規登録モード切替リンクを非表示にする。<br>
	 */
	public static final String	CMD_EDIT_MODE			= "TM1092";
	
	/**
	 * 一括更新コマンド。<br>
	 * <br>
	 * 一覧にて選択チェックボックスにチェックが入っている下書状態のレコードの申請処理を繰り返し行う。<br>
	 * ひとつもチェックが入っていない状態で一括更新ボタンをクリックした場合はエラーメッセージにて通知し、処理を中断する。<br>
	 */
	public static final String	CMD_BATCH_UPDATE		= "TM1095";
	
	/**
	 * 表示期間決定コマンド。<br>
	 * <br>
	 * 入力した表示期間時点で有効な勤務形態情報を取得し、その略称と勤務時間を勤務形態プルダウンに表示する。<br>
	 */
	public static final String	CMD_SET_VIEW_PERIOD		= "TM1097";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public DifferenceRequestAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SEARCH;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new DifferenceRequestVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SHOW)) {
			// 表示
			prepareVo(false, false);
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SEARCH)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_DRAFT)) {
			// 下書き
			prepareVo();
			draft();
		} else if (mospParams.getCommand().equals(CMD_APPLI)) {
			// 申請
			prepareVo();
			appli();
		} else if (mospParams.getCommand().equals(CMD_WORKTIME_CALC)) {
			// 勤務時刻再計算
			prepareVo();
			workTimeCalc();
		} else if (mospParams.getCommand().equals(CMD_WITHDRAWN)) {
			// 取下
			prepareVo();
			withdrawn();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_SET_ACTIVATION_DATE)) {
			// 出勤日決定
			prepareVo();
			setActivationDate();
		} else if (mospParams.getCommand().equals(CMD_INSERT_MODE)) {
			// 新規登録モード切替
			prepareVo();
			insertMode();
		} else if (mospParams.getCommand().equals(CMD_EDIT_MODE)) {
			// 編集モード切替
			prepareVo();
			editMode();
		} else if (mospParams.getCommand().equals(CMD_BATCH_UPDATE)) {
			// 一括更新
			prepareVo();
			batchUpdate();
		} else if (mospParams.getCommand().equals(CMD_SET_VIEW_PERIOD)) {
			// 表示期間決定
			prepareVo();
			setViewPeriod();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void show() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// 新規登録モード設定
		insertMode();
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(DifferenceRequestRequestDateComparator.class.getName());
	}
	
	/**
	* 下書処理を行う。<br>
	 * @throws MospException 例外処理発生時
	*/
	protected void draft() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 登録クラス取得
		DifferenceRequestRegistBeanInterface regist = time().differenceRequestRegist();
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		Date startDate = DateUtility.getDate(vo.getPltEditRequestYear(), vo.getPltEditRequestMonth(), vo
			.getPltEditRequestDay());
		Date endDate = startDate;
		if (vo.getCkbEndDate().equals("on")) {
			endDate = DateUtility.getDate(vo.getPltEditEndYear(), vo.getPltEditEndMonth(), vo.getPltEditEndDay());
		}
		if (endDate.before(startDate)) {
			// 終了日が開始日より前である場合
			String[] aryMeassage = { mospParams.getName("Period") };
			mospParams.addMessage(PlatformMessageConst.MSG_INPUT, aryMeassage);
			return;
		}
		Date targetDate = startDate;
		while (!targetDate.after(endDate)) {
			// DTOの準備
			DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			if (targetDate.compareTo(startDate) != 0) {
				dto = regist.getInitDto();
			}
			// DTOに値を設定
			setDifferenceDtoFields(dto, targetDate);
			if (targetDate.compareTo(startDate) != 0) {
				dto.setTmdDifferenceRequestId(0);
			}
			// 妥当性チェック
			regist.validate(dto);
			// 申請の相関チェック
			regist.checkDraft(dto);
			// 登録クラスの取得
			WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_DIFFERENCE);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(
						workflowDto,
						dto.getPersonalId(),
						mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
								new String[]{ mospParams.getName("WorkPaper") }));
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 1日加算
			targetDate = DateUtility.addDay(targetDate, 1);
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 下書成功メッセージ設定
		addDraftMessage();
		// 検索
		search();
		// 履歴取得
		setEditUpdateMode(personalId, startDate);
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
	}
	
	/**
	 * 申請処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void appli() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 登録クラス取得
		DifferenceRequestRegistBeanInterface regist = time().differenceRequestRegist();
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		Date startDate = DateUtility.getDate(vo.getPltEditRequestYear(), vo.getPltEditRequestMonth(), vo
			.getPltEditRequestDay());
		Date endDate = startDate;
		if (vo.getCkbEndDate().equals("on")) {
			endDate = DateUtility.getDate(vo.getPltEditEndYear(), vo.getPltEditEndMonth(), vo.getPltEditEndDay());
		}
		if (endDate.before(startDate)) {
			// 終了日が開始日より前である場合
			String[] aryMeassage = { mospParams.getName("Period") };
			mospParams.addMessage(PlatformMessageConst.MSG_INPUT, aryMeassage);
			return;
		}
		Date targetDate = startDate;
		while (!targetDate.after(endDate)) {
			// DTOの準備
			DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			if (targetDate.compareTo(startDate) != 0) {
				dto = regist.getInitDto();
			}
			// DTOに値を設定
			setDifferenceDtoFields(dto, targetDate);
			if (targetDate.compareTo(startDate) != 0) {
				dto.setTmdDifferenceRequestId(0);
			}
			// 申請の相関チェック
			regist.checkAppli(dto);
			// 登録クラスの取得
			WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
			// ワークフローの設定
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				workflowDto = workflowRegist.getInitDto();
				workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_DIFFERENCE);
			}
			workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
			// 登録後ワークフローの取得
			workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			if (workflowDto != null) {
				// ワークフローコメント登録
				platform().workflowCommentRegist().addComment(workflowDto, dto.getPersonalId(),
						chkApproverSettingCommentName());
				// ワークフロー番号セット
				dto.setWorkflow(workflowDto.getWorkflow());
				// 登録
				regist.regist(dto);
			}
			// 1日加算
			targetDate = DateUtility.addDay(targetDate, 1);
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 申請成功メッセージ設定
		addAppliMessage();
		// 履歴取得
		setEditUpdateMode(personalId, startDate);
		// 登録結果確認
		if (!mospParams.hasErrorMessage()) {
			// 登録が成功した場合、初期状態に戻す。
			insertMode();
			search();
		}
	}
	
	/**
	 * 勤務時刻再計算処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void workTimeCalc() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		if ("a".equals(vo.getPltEditDifferenceType())) {
			vo.setPltEditRequestHour("8");
			vo.setPltEditRequestMinute("0");
			vo.setLblEndTimeHour("16");
			vo.setLblEndTimeMinutes("00");
			vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		} else if ("b".equals(vo.getPltEditDifferenceType())) {
			vo.setPltEditRequestHour("8");
			vo.setPltEditRequestMinute("30");
			vo.setLblEndTimeHour("16");
			vo.setLblEndTimeMinutes("30");
			vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		} else if ("c".equals(vo.getPltEditDifferenceType())) {
			vo.setPltEditRequestHour("9");
			vo.setPltEditRequestMinute("30");
			vo.setLblEndTimeHour("17");
			vo.setLblEndTimeMinutes("30");
			vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		} else if ("d".equals(vo.getPltEditDifferenceType())) {
			vo.setPltEditRequestHour("10");
			vo.setPltEditRequestMinute("0");
			vo.setLblEndTimeHour("18");
			vo.setLblEndTimeMinutes("00");
			vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		} else if ("s".equals(vo.getPltEditDifferenceType())) {
			if (vo.getJsEditDifferenceTypeMode().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
				// 出勤時刻
				WorkTypeItemDtoInterface dtoItemWorkStart = timeReference().workTypeItem().getWorkTypeItemInfo(
						vo.getLblWorkType(), getEditRequestDate(), TimeConst.CODE_WORKSTART);
				if (dtoItemWorkStart == null) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Work")
							+ mospParams.getName("Form"), null);
					// 有効日モード設定
					vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
					return;
				}
				// 退勤時刻
				WorkTypeItemDtoInterface dtoItemWorkEnd = timeReference().workTypeItem().getWorkTypeItemInfo(
						vo.getLblWorkType(), getEditRequestDate(), TimeConst.CODE_WORKEND);
				if (dtoItemWorkEnd == null) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Work")
							+ mospParams.getName("Form"), null);
					// 有効日モード設定
					vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
					return;
				}
				// 開始時刻の取得
				vo.setPltEditRequestHour(String.valueOf(DateUtility.getHour(dtoItemWorkStart.getWorkTypeItemValue())));
				vo.setPltEditRequestMinute(String.valueOf(DateUtility
					.getMinute(dtoItemWorkStart.getWorkTypeItemValue())));
				// 終了時刻の取得
				vo.setLblEndTimeHour(DateUtility.getStringHour(dtoItemWorkEnd.getWorkTypeItemValue()));
				vo.setLblEndTimeMinutes(DateUtility.getStringMinute(dtoItemWorkEnd.getWorkTypeItemValue()));
			} else {
				// 変更された時差出勤開始時刻から時差出勤終了時刻を設定する
				int startHour = Integer.valueOf(vo.getPltEditRequestHour());
				int startMinutes = Integer.valueOf(vo.getPltEditRequestMinute());
				// 勤務時間取得
				WorkTypeItemDtoInterface dtoItemWorkTime = timeReference().workTypeItem().getWorkTypeItemInfo(
						vo.getLblWorkType(), getEditRequestDate(), TimeConst.CODE_WORKTIME);
				if (dtoItemWorkTime == null) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Work")
							+ mospParams.getName("Form"), null);
					// 有効日モード設定
					vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
					return;
				}
				int workTimeHour = DateUtility.getHour(dtoItemWorkTime.getWorkTypeItemValue());
				int workTimeMinutes = DateUtility.getMinute(dtoItemWorkTime.getWorkTypeItemValue());
				// 休憩時間取得
				WorkTypeItemDtoInterface dtoItemRestTime = timeReference().workTypeItem().getWorkTypeItemInfo(
						vo.getLblWorkType(), getEditRequestDate(), TimeConst.CODE_RESTTIME);
				if (dtoItemRestTime == null) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Work")
							+ mospParams.getName("Form"), null);
					// 有効日モード設定
					vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
					return;
				}
				int restTimeHour = DateUtility.getHour(dtoItemRestTime.getWorkTypeItemValue());
				int restTimeMinutes = DateUtility.getMinute(dtoItemRestTime.getWorkTypeItemValue());
				int totalWorkTimeHour = workTimeHour + restTimeHour;
				int totalWorkTimeMinutes = workTimeMinutes + restTimeMinutes;
				if (totalWorkTimeMinutes >= 60) {
					totalWorkTimeHour++;
					totalWorkTimeMinutes = totalWorkTimeMinutes - 60;
				}
				totalWorkTimeHour = startHour + totalWorkTimeHour;
				totalWorkTimeMinutes = startMinutes + totalWorkTimeMinutes;
				if (totalWorkTimeMinutes >= 60) {
					totalWorkTimeHour++;
					totalWorkTimeMinutes = totalWorkTimeMinutes - 60;
				}
				vo.setLblEndTimeHour(String.valueOf(totalWorkTimeHour));
				vo.setLblEndTimeMinutes(String.valueOf(totalWorkTimeMinutes));
			}
			vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		}
	}
	
	/**
	* 取下処理を行う。<br>
	 * @throws MospException 例外処理発生時
	*/
	protected void withdrawn() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// DTOの準備
		DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKey(vo.getRecordId());
		// 存在確認
		checkSelectedDataExist(dto);
		// 取下の相関チェック
		time().differenceRequestRegist().checkWithdrawn(dto);
		// ワークフロー取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		// 存在確認
		checkSelectedDataExist(workflowDto);
		// ワークフロー登録
		workflowDto = platform().workflowRegist().withdrawn(workflowDto, dto.getPersonalId());
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					dto.getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("TakeDown") }));
		}
		// 取下結果確認
		if (mospParams.hasErrorMessage()) {
			// 取下失敗メッセージ設定
			addDeleteHistoryFailedMessage();
			return;
		}
		// コミット
		commit();
		
		// 取下成功メッセージ設定
		addTakeDownMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外処理発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * @throws MospException 例外処理発生時
	 * 
	 */
	protected void setActivationDate() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getJsEditActivateDateMode().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			if (setApprvalPullDown(getPersonalId(vo.getLblEmployeeCode()), DateUtility.getDate(vo
				.getPltEditRequestYear(), vo.getPltEditRequestMonth(), vo.getPltEditRequestDay())) == false) {
				return;
			}
			DifferenceRequestRegistBeanInterface regist = time().differenceRequestRegist();
			DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKey(vo.getRecordId());
			if (dto == null) {
				dto = regist.getInitDto();
			}
			dto.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
			dto.setRequestDate(getEditRequestDate());
			regist.checkTemporaryClosingFinal(dto);
			regist.checkDifferenceOverlap(dto);
			regist.checkRequest(dto);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			// 有効日モード設定
			vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			String[] aryPltLblApproverSetting = new String[0];
			vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
			// 時差出勤区分の初期化
			vo.setPltEditDifferenceType("a");
			vo.setPltEditRequestHour("8");
			vo.setPltEditRequestMinute("0");
			vo.setLblEndTimeHour("16");
			vo.setLblEndTimeMinutes("00");
			// 有効日モード設定
			vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
			vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
		// 出勤日決定
		getEditItems();
	}
	
	/**
	 * @throws MospException 例外処理発生時
	 */
	protected void getEditItems() throws MospException {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 有効日フラグ確認
		if (vo.getJsEditActivateDateMode().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			// 年月日を締め日
			ApplicationDtoInterface applicationDto = timeReference().application().findForPerson(
					getPersonalId(vo.getLblEmployeeCode()), DateUtility.getSystemDate());
			timeReference().application().chkExistApplication(applicationDto, DateUtility.getSystemDate());
			if (mospParams.hasErrorMessage()) {
				// 有効日モード設定
				vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
				return;
			}
			String calendarCode = applicationDto.getScheduleCode();
			// getScheduleDateInfoのメソッドの二つ目の引数のtargetDate（有効日）の仕様が未確定
			ScheduleDateDtoInterface scheduleDateDto = timeReference().scheduleDate().getScheduleDateInfo(calendarCode,
					DateUtility.getSystemDate(), getEditRequestDate());
			timeReference().scheduleDate().chkExistScheduleDate(scheduleDateDto, getEditRequestDate());
			if (mospParams.hasErrorMessage()) {
				// 有効日モード設定
				vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
				return;
			}
			WorkTypeDtoInterface workTypeDto = timeReference().workType().getWorkTypeInfo(
					scheduleDateDto.getWorkTypeCode(), getEditRequestDate());
			if (workTypeDto == null) {
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Work")
						+ mospParams.getName("Form"), null);
				// 有効日モード設定
				vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
				return;
			}
			// 出勤時刻
			WorkTypeItemDtoInterface dtoItemWorkStart = timeReference().workTypeItem().getWorkTypeItemInfo(
					scheduleDateDto.getWorkTypeCode(), getEditRequestDate(), TimeConst.CODE_WORKSTART);
			if (dtoItemWorkStart == null) {
				mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Work")
						+ mospParams.getName("Form"), null);
				// 有効日モード設定
				vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
				return;
			}
			String itemWorkStart = getStringTimeMinutes(DateUtility.getStringHour(dtoItemWorkStart
				.getWorkTypeItemValue()), DateUtility.getStringMinute(dtoItemWorkStart.getWorkTypeItemValue()));
			// 退勤時刻
			WorkTypeItemDtoInterface dtoItemWorkEnd = timeReference().workTypeItem().getWorkTypeItemInfo(
					scheduleDateDto.getWorkTypeCode(), getEditRequestDate(), TimeConst.CODE_WORKEND);
			String itemWorkEnd = getStringTimeMinutes(DateUtility.getStringHour(dtoItemWorkEnd.getWorkTypeItemValue()),
					DateUtility.getStringMinute(dtoItemWorkEnd.getWorkTypeItemValue()));
			// 勤怠形態の取得
			vo.setLblWorkType(workTypeDto.getWorkTypeCode());
			String workTypeName = workTypeDto.getWorkTypeAbbr()
					+ mospParams.getName("FrontWithCornerParentheses")
					+ getTimeWaveFormatNoCheck(dtoItemWorkStart.getWorkTypeItemValue(), dtoItemWorkEnd
						.getWorkTypeItemValue()) + mospParams.getName("BackWithCornerParentheses");
			vo.setLblWorkTypeName(workTypeName);
			// 勤務時間の取得
			vo.setLblWorkTime(getTimeWaveFormat(itemWorkStart, itemWorkEnd));
			// 開始/終了時刻の設定
			workTimeCalc();
			return;
		}
		// 勤怠形態の取得
		vo.setLblWorkType("");
		vo.setLblWorkTypeName("");
		// 勤務時間の取得
		vo.setLblWorkTime("");
		// 開始時刻の取得
		vo.setPltEditRequestHour("");
		vo.setPltEditRequestMinute("");
		// 終了時刻の取得
		vo.setLblEndTimeHour("");
		vo.setLblEndTimeMinutes("");
	}
	
	/**
	 * 初期値を設定する。<br>
	 */
	public void setDefaultValues() {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		// 検索項目設定
		vo.setRecordId(0);
		if (getTransferredYear() == null) {
			vo.setPltEditRequestYear(DateUtility.getStringYear(date));
			vo.setPltEditRequestMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditRequestDay(String.valueOf(DateUtility.getDay(date)));
			vo.setPltEditEndYear(DateUtility.getStringYear(date));
			vo.setPltEditEndMonth(String.valueOf(DateUtility.getMonth(date)));
			vo.setPltEditEndDay(String.valueOf(DateUtility.getDay(date)));
		} else {
			vo.setPltEditRequestYear(getTransferredYear());
			vo.setPltEditRequestMonth(getTransferredMonth());
			vo.setPltEditRequestDay(getTransferredDay());
			vo.setPltEditEndYear(getTransferredYear());
			vo.setPltEditEndMonth(getTransferredMonth());
			vo.setPltEditEndDay(getTransferredDay());
		}
		vo.setPltEditRequestType("");
		vo.setPltEditDifferenceType("a");
		vo.setPltEditRequestHour("0");
		vo.setPltEditRequestMinute("0");
		vo.setLblEndTimeHour("");
		vo.setLblEndTimeMinutes("");
		vo.setTxtEditRequestReason("");
		vo.setLblWorkType("");
		vo.setLblWorkTypeName("");
		vo.setLblWorkTime("");
		vo.setLblEndTimeHour("");
		vo.setLblEndTimeMinutes("");
		vo.setPltSearchState("");
		vo.setPltSearchRequestHour("");
		vo.setPltSearchRequestMinute("");
		vo.setPltSearchRequestYear(DateUtility.getStringYear(date));
		vo.setPltSearchRequestMonth("");
		vo.setCkbEndDate("");
		// 承認者欄の初期化
		String[] aryPltLblApproverSetting = new String[0];
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		vo.setPltApproverSetting1("");
		vo.setPltApproverSetting2("");
		vo.setPltApproverSetting3("");
		vo.setPltApproverSetting4("");
		vo.setPltApproverSetting5("");
		vo.setPltApproverSetting6("");
		vo.setPltApproverSetting7("");
		vo.setPltApproverSetting8("");
		vo.setPltApproverSetting9("");
		vo.setPltApproverSetting10("");
		vo.setJsEditDifferenceTypeMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
	}
	
	/**
	 * プルダウン設定
	 */
	private void setPulldown() {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		vo.setAryPltEditRequestYear(getYearArray(DateUtility.getStringYear(date)));
		vo.setAryPltEditRequestMonth(getMonthArray());
		// 31日まで取得
		vo.setAryPltEditRequestDay(getDayArray(31));
		vo.setAryPltEditRequestHour(getHourArray());
		vo.setAryPltEditRequestMinute(getMinuteArray());
		vo.setAryPltEditEndYear(getYearArray(DateUtility.getStringYear(date)));
		vo.setAryPltEditEndMonth(getMonthArray());
		vo.setAryPltEditEndDay(getDayArray(31));
		vo.setAryPltEditDifferenceType(mospParams.getProperties().getCodeArray(TimeConst.CODE_DIFFERENCE_TYPE, false));
		vo.setAryPltSearchState(mospParams.getProperties().getCodeArray(TimeConst.CODE_APPROVAL_STATE, true));
		vo.setAryPltSearchRequestYear(getYearArray(DateUtility.getStringYear(date)));
		vo.setAryPltSearchRequestMonth(addArrayFirstRowBlank(getMonthArray()));
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void insertMode() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		setEmployeeParams();
		// 初期値設定
		setDefaultValues();
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// プルダウン設定
		setPulldown();
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNCLAIMED);
		// 
		setJsMode();
		// 表示期間決定
		setSearchPulldown();
		// 基本情報チェック
		timeReference().differenceRequest().chkBasicInfo(vo.getPersonalId(), getEditRequestDate());
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void search() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 検索クラス取得
		DifferenceRequestSearchBeanInterface search = timeReference().differenceRequestSearch();
		// VOの値を検索クラスへ設定
		search.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		search.setWorkflowStatus(vo.getPltSearchState());
		search.setWorkTypeCode(vo.getPltSearchWorkType());
		int year = Integer.valueOf(vo.getPltSearchRequestYear());
		int startMonth = 0;
		int endMonth = 0;
		if (vo.getPltSearchRequestMonth().isEmpty()) {
			startMonth = 1;
			endMonth = 12;
		} else {
			startMonth = Integer.valueOf(vo.getPltSearchRequestMonth());
			endMonth = startMonth;
		}
		search.setRequestStartDate(DateUtility.getDate(year, startMonth, 1));
		search.setRequestEndDate(DateUtility.getDate(year, endMonth, getEndDay(year, endMonth)));
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<DifferenceRequestListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(DifferenceRequestRequestDateComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.size() == 0) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
	}
	
	/**
	 * 
	 */
	protected void setJsMode() {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 有効日(編集)モード設定
		vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		// 有効日(検索)モード設定
		vo.setJsSearchActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
	}
	
	/**
	 * @return 有効日
	 */
	protected Date getEditRequestDate() {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 有効日取得
		return getDate(vo.getPltEditRequestYear(), vo.getPltEditRequestMonth(), vo.getPltEditRequestDay());
	}
	
	/**
	 * @return 有効日
	 */
	protected Date getSearchRequestDate() {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		Date date;
		// 有効日取得
		if (vo.getPltSearchRequestMonth().isEmpty()) {
			date = getDate(vo.getPltSearchRequestYear(), "1", "1");
		} else {
			date = getDate(vo.getPltSearchRequestYear(), vo.getPltSearchRequestMonth(), "1");
		}
		return date;
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外処理発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// データ配列初期化
		String[] aryCkbRecordId = new String[list.size()];
		String[] aryLblDate = new String[list.size()];
		String[] aryLblRequestType = new String[list.size()];
		String[] aryLblWorkTime = new String[list.size()];
		String[] aryLblRequestReason = new String[list.size()];
		String[] aryLblState = new String[list.size()];
		String[] aryLblApprover = new String[list.size()];
		String[] aryLblOnOff = new String[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			DifferenceRequestListDtoInterface dto = (DifferenceRequestListDtoInterface)list.get(i);
			// 配列に情報を設定
			aryCkbRecordId[i] = String.valueOf(dto.getTmdDifferenceRequestId());
			aryLblDate[i] = DateUtility.getStringDateAndDay(dto.getRequestDate());
			aryLblRequestType[i] = getDifferenceTypeName(dto.getAroundType());
			aryLblWorkTime[i] = DateUtility.getStringTime(dto.getStartTime()) + mospParams.getName("Wave")
					+ getEndTime(dto.getStartTime(), dto.getEndTime());
			aryLblRequestReason[i] = dto.getRequestReason();
			aryLblState[i] = getStatusStageValueView(dto.getState(), dto.getStage());
			aryLblOnOff[i] = getButtonOnOff(dto.getState());
			// 承認者に関してはdtoから直接取得する（他の申請画面も同様）
			aryLblApprover[i] = dto.getApproverName();
		}
		// データをVOに設定
		vo.setAryCkbDifferenceRequestListId(aryCkbRecordId);
		vo.setAryLblDate(aryLblDate);
		vo.setAryLblRequestType(aryLblRequestType);
		vo.setAryLblWorkTime(aryLblWorkTime);
		vo.setAryLblRequestReason(aryLblRequestReason);
		vo.setAryLblState(aryLblState);
		vo.setAryLblApprover(aryLblApprover);
		vo.setAryLblOnOff(aryLblOnOff);
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void batchUpdate() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		
		// 一括更新処理
		time().differenceRequestRegist().update(getIdArray(vo.getCkbSelect()));
		// 一括更新結果確認
		if (mospParams.hasErrorMessage()) {
			// 更新失敗メッセージ設定
			addBatchUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 更新成功メッセージ設定
		addUpdateMessage();
		// 新規登録モード設定(編集領域をリセット)
		insertMode();
		// 検索
		search();
	}
	
	/**
	 * 一括更新処理を行う。<br>
	 * @throws MospException 例外処理発生時
	 */
	protected void setViewPeriod() throws MospException {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 現在の有効日モードを確認
		if (vo.getJsSearchActivateDateMode().equals(PlatformConst.MODE_ACTIVATE_DATE_CHANGING)) {
			// 有効日モード設定
			vo.setJsSearchActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_FIXED);
		} else {
			// 有効日モード設定
			vo.setJsSearchActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		}
		setSearchPulldown();
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外処理発生時
	 */
	private void setSearchPulldown() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// 有効日フラグ確認
		if (vo.getJsSearchActivateDateMode().equals(PlatformConst.MODE_ACTIVATE_DATE_FIXED)) {
			vo.setAryPltSearchWorkType(timeReference().workType()
				.getStartTimeEndTimeSelectArray(getSearchRequestDate()));
			return;
		}
		vo.setAryPltSearchWorkType(new String[0][0]);
	}
	
	/**
	 * 履歴編集モードで画面を表示する。<br>
	 * 履歴編集対象は、遷移汎用コード及び有効日で取得する。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void editMode() throws MospException {
		// VO準備
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		// だるま連携用設定
		setPageInfo(CMD_PAGE, getListLength());
		insertMode();
		// 遷移汎用コード及び有効日から履歴編集対象を取得し編集モードを設定
		setEditUpdateMode(getPersonalId(vo.getLblEmployeeCode()), getDate(getTransferredActivateDate()));
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
		// 有効日モード設定
		vo.setJsEditActivateDateMode(PlatformConst.MODE_ACTIVATE_DATE_CHANGING);
		setActivationDate();
	}
	
	/**
	 * 履歴編集モードを設定する。<br>
	 * 雇用契約コードと有効日で編集対象情報を取得する。<br>
	 * @param personalId 個人ID
	 * @param activateDate 有効日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void setEditUpdateMode(String personalId, Date activateDate) throws MospException {
		// 履歴編集対象取得
		DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKeyOnWorkflow(personalId,
				activateDate);
		// 存在確認
		checkSelectedDataExist(dto);
		// VOにセット
		setVoFields(dto);
	}
	
	/**
	 * DTOの値をVO(編集項目)に設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外処理発生時
	 */
	protected void setVoFields(DifferenceRequestDtoInterface dto) throws MospException {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		int year = DateUtility.getYear(dto.getRequestDate());
		int month = DateUtility.getMonth(dto.getRequestDate());
		int day = DateUtility.getDay(dto.getRequestDate());
		// DTOの値をVOに設定
		vo.setRecordId(dto.getTmdDifferenceRequestId());
		vo.setPltEditRequestYear(String.valueOf(year));
		vo.setPltEditRequestMonth(String.valueOf(month));
		vo.setPltEditRequestDay(String.valueOf(day));
		vo.setPltEditEndYear(String.valueOf(year));
		vo.setPltEditEndMonth(String.valueOf(month));
		vo.setPltEditEndDay(String.valueOf(day));
		vo.setPltEditRequestHour(String.valueOf(DateUtility.getHour(dto.getRequestStart())));
		vo.setPltEditRequestMinute(String.valueOf(DateUtility.getMinute(dto.getRequestStart())));
		vo.setPltEditDifferenceType(dto.getDifferenceType());
		vo.setLblWorkType(dto.getWorkTypeCode());
		vo.setLblEndTimeHour(getEndTimeHour(dto.getRequestStart(), dto.getRequestEnd()));
		vo.setLblEndTimeMinutes(DateUtility.getStringMinute(dto.getRequestEnd()));
		vo.setTxtEditRequestReason(dto.getRequestReason());
		// 状態設定
//		setStatusApplication(dto.getWorkflow());
		vo.setJsStatusApplication("");
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param date 時差出勤日
	 * @throws MospException 例外処理発生時
	 */
	protected void setDifferenceDtoFields(DifferenceRequestDtoInterface dto, Date date) throws MospException {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		// VOの値をDTOに設定
		dto.setTmdDifferenceRequestId(vo.getRecordId());
		dto.setPersonalId(personalId);
		dto.setRequestDate(date);
		dto.setDifferenceType(vo.getPltEditDifferenceType());
		dto.setWorkTypeCode(vo.getLblWorkType());
		dto.setRequestStart(DateUtility
			.getDateTime(DateUtility.getYear(date), DateUtility.getMonth(date), DateUtility.getDay(date), Integer
				.parseInt(vo.getPltEditRequestHour()), Integer.parseInt(vo.getPltEditRequestMinute())));
		dto.setRequestEnd(getEndTime(date));
		dto.setRequestReason(vo.getTxtEditRequestReason());
		dto.setTimesWork(1);
	}
	
	/**
	 * @param date 対象日
	 * @return 時差出勤時刻の退勤時刻
	 * @throws MospException 例外処理発生時
	 */
	protected Date getEndTime(Date date) throws MospException {
		// VO取得
		DifferenceRequestVo vo = (DifferenceRequestVo)mospParams.getVo();
		int endHour = Integer.parseInt(vo.getLblEndTimeHour());
		int endMinute = Integer.parseInt(vo.getLblEndTimeMinutes());
		// 24時以上の場合、次の日付を設定
		if (24 <= endHour) {
			Date targetDate = DateUtility.addDay(date, 1);
			int year = DateUtility.getYear(targetDate);
			int month = DateUtility.getMonth(targetDate);
			int day = DateUtility.getDay(targetDate);
			int hour = endHour - 24;
			int minute = endMinute;
			return DateUtility.getDateTime(year, month, day, hour, minute);
		}
		return DateUtility.getDateTime(DateUtility.getYear(date), DateUtility.getMonth(date), DateUtility.getDay(date),
				endHour, endMinute);
	}
}
