/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.AttendanceCorrectionReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.AttendanceCorrectionDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeCorrectionDtoInterface;
import jp.mosp.time.input.vo.AttendanceHistoryVo;

/**
 * 選択したレコードの勤怠申請承認情報の修正履歴情報を取得し、表示する。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_ATTENDANCE_INFO_SELECT_SHOW}
 * </li><li>
 */
public class AttendanceHistoryAction extends TimeAction {
	
	/**
	 * 勤怠情報修正履歴の選択表示コマンド。<br>
	 * <br>
	 * 勤怠情報テーブルより取得した勤怠情報を勤怠情報欄に、その修正履歴を承認履歴欄にそれぞれ表示する。<br>
	 */
	public static final String	CMD_ATTENDANCE_INFO_SELECT_SHOW		= "TM1911";
	
	/**
	 * 勤怠修正情報の修正履歴の選択表示コマンド<br>
	 * <br>
	 * 勤怠情報テーブルより取得した勤怠修正情報を勤怠情報欄に、その修正履歴を承認履歴欄にそれぞれ表示する。<br>
	 */
	public static final String	CMD_ATTENDANCE_FIXED_SELECT_SHOW	= "TM1921";
	
	/**
	 * 勤怠修正情報の修正履歴の選択表示コマンド<br>
	 * <br>
	 * この画面よりも奥の階層にあたる画面からこの画面に再び遷移した際に編集内容を反映させる。<br>
	 */
	public static final String	CMD_ATTENDANCE_FIXED_RE_SHOW		= "TM1912";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public AttendanceHistoryAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_ATTENDANCE_FIXED_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new AttendanceHistoryVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_ATTENDANCE_INFO_SELECT_SHOW)) {
			// 選択表示
			prepareVo(false, false);
			attendanceInfo();
			attendanceApprovalHistory();
		} else if (mospParams.getCommand().equals(CMD_ATTENDANCE_FIXED_SELECT_SHOW)) {
			prepareVo(false, false);
			attendanceAggregateApprovalHistory();
		} else if (mospParams.getCommand().equals(CMD_ATTENDANCE_FIXED_RE_SHOW)) {
			prepareVo(true, false);
			attendanceInfo();
			attendanceApprovalHistory();
		}
	}
	
	/**
	 * 
	 * @throws MospException 例外発生時
	 */
	protected void attendanceInfo() throws MospException {
		// VO準備
		AttendanceHistoryVo vo = (AttendanceHistoryVo)mospParams.getVo();
		// 該当社員欄設定
		setEmployeeParams();
		Date targetDate = getTransferredLocalDate();
		String personalId = getTransferredLocalCode();
		// DTO準備		
		AttendanceDtoInterface dto = timeReference().attendance().findForKey(personalId, targetDate, 1);
		AttendanceCorrectionDtoInterface correctionDto = timeReference().attendanceCorrection()
			.getLatestAttendanceCorrectionInfo(personalId, targetDate, 1);
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setLblAttendanceDate(DateUtility.getStringDateAndDay(dto.getWorkDate()));
		vo.setAttendanceDate(DateUtility.getStringDate(dto.getWorkDate()));
		vo.setLblAttendanceWorkType(timeReference().workType().getWorkTypeInfo(dto.getWorkTypeCode(), targetDate)
			.getWorkTypeAbbr());
		vo.setLblAttendanceStartDate(DateUtility.getStringTime(dto.getStartTime()));
		vo.setLblAttendanceEndDate(DateUtility.getStringTime(dto.getEndTime()));
		vo.setLblAttendanceWorkTime(getValueHyphenView(dto.getWorkTime()));
		vo.setLblAttendanceRestTime(getValueHyphenView(dto.getRestTime()));
		vo.setLblAttendanceLateTime(getValueHyphenView(dto.getLateTime()));
		vo.setLblAttendanceLeaveEarly(getValueHyphenView(dto.getLeaveEarlyTime()));
		vo.setLblAttendanceOverTimeIn(getValueHyphenView(dto.getOvertime()));
		vo.setLblAttendanceOverTimeOut(getValueHyphenView(dto.getOvertimeOut()));
		vo.setLblAttendanceWorkOnHoliday(getValueHyphenView(dto.getSpecificWorkTime()));
		vo.setLblAttendanceLateNight(getValueHyphenView(dto.getLateNightTime()));
		vo.setLblAttendanceState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		vo.setFunctionCode(workflowDto.getFunctionCode());
		if (workflowDto.getFunctionCode().equals(TimeConst.CODE_FUNCTION_WORK_MANGE)) {
			vo.setApprovalHistoryCmd(ApprovalHistoryAction.CMD_TIME_APPROVAL_HISTORY_SELECT_SHOW);
		} else if (workflowDto.getFunctionCode().equals(TimeConst.CODE_FUNCTION_OVER_WORK)) {
			vo.setApprovalHistoryCmd(ApprovalHistoryAction.CMD_OVERTIME_APPROVAL_HISTORY_SELECT_SHOW);
		} else if (workflowDto.getFunctionCode().equals(TimeConst.CODE_FUNCTION_VACATION)) {
			vo.setApprovalHistoryCmd(ApprovalHistoryAction.CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW);
		} else if (workflowDto.getFunctionCode().equals(TimeConst.CODE_FUNCTION_WORK_HOLIDAY)) {
			vo.setApprovalHistoryCmd(ApprovalHistoryAction.CMD_APPROVAL_HISTORY_HOLIDAY_WORK_SELECT_SHOW);
		} else if (workflowDto.getFunctionCode().equals(TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY)) {
			vo.setApprovalHistoryCmd(ApprovalHistoryAction.CMD_APPROVAL_HISTORY_LIEU_SELECT_SHOW);
		} else if (workflowDto.getFunctionCode().equals(TimeConst.CODE_FUNCTION_DIFFERENCE)) {
			vo.setApprovalHistoryCmd(ApprovalHistoryAction.CMD_DIFFERENCE_WORK_APPROVAL_HISTORY_SELECT_SHOW);
		}
		vo.setRecordId(String.valueOf(dto.getTmdAttendanceId()));
		// 未仕様
		vo.setLblAttendanceRemark("");
		vo.setLblAttendanceCorrection(getCorrectionInfo(getTransferredCode(), getEmployeeCode(correctionDto
			.getPersonalId())));
	}
	
	/**
	 * <br>
	 * @throws MospException 例外発生時
	 */
	public void attendanceApprovalHistory() throws MospException {
		// VO準備
		AttendanceHistoryVo vo = (AttendanceHistoryVo)mospParams.getVo();
		Date targetDate = getTransferredLocalDate();
		String personalId = getTransferredLocalCode();
		// 一覧表示項目設定
		AttendanceCorrectionReferenceBeanInterface correction = timeReference().attendanceCorrection();
		List<AttendanceCorrectionDtoInterface> list = correction.getAttendanceCorrectionHistory(personalId, targetDate,
				1);
		// 該当社員欄設定
		String[] aryLblCorrectionNumber = new String[list.size()];
		String[] aryLblCorrectionDate = new String[list.size()];
		String[] aryLblCorrectionEmployee = new String[list.size()];
		String[] aryLblCorrectionType = new String[list.size()];
		String[] aryLblCorrectionBefore = new String[list.size()];
		String[] aryLblCorrectionAfter = new String[list.size()];
		String[] aryLblCorrectionComment = new String[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			AttendanceCorrectionDtoInterface dto = list.get(i);
			// 配列に情報を設定
			aryLblCorrectionNumber[i] = String.valueOf(i + 1);
			// 修正日時
			aryLblCorrectionDate[i] = DateUtility.getStringDateAndDay(dto.getCorrectionDate()) + " "
					+ DateUtility.getStringTime(dto.getCorrectionDate());
			// 修正者
			aryLblCorrectionEmployee[i] = getCheckEmployees(getEmployeeCode(dto.getCorrectionPersonalId()), targetDate);
			// 修正箇所
			aryLblCorrectionType[i] = correction.getHistoryAttendanceMoreName(dto.getCorrectionType());
			// 修正前
			aryLblCorrectionBefore[i] = correction.getCorrectionValue(dto.getCorrectionType(), dto
				.getCorrectionBefore(), targetDate);
			// 修正後
			aryLblCorrectionAfter[i] = correction.getCorrectionValue(dto.getCorrectionType(), dto.getCorrectionAfter(),
					targetDate);
			// コメント
			aryLblCorrectionComment[i] = dto.getCorrectionReason();
		}
		vo.setAryLblCorrectionNumber(aryLblCorrectionNumber);
		vo.setAryLblCorrectionDate(aryLblCorrectionDate);
		vo.setAryLblCorrectionEmployee(aryLblCorrectionEmployee);
		vo.setAryLblCorrectionType(aryLblCorrectionType);
		vo.setAryLblCorrectionBefore(aryLblCorrectionBefore);
		vo.setAryLblCorrectionAfter(aryLblCorrectionAfter);
		vo.setAryLblCorrectionComment(aryLblCorrectionComment);
	}
	
	/**
	 * @throws MospException 例外発生時
	 */
	public void attendanceAggregateApprovalHistory() throws MospException {
		// VO準備
		AttendanceHistoryVo vo = (AttendanceHistoryVo)mospParams.getVo();
		// 該当社員欄設定
		setEmployeeParams();
		// 一覧表示項目設定
		Date targetDate = getTransferredLocalDate();
		String personalId = getTransferredLocalCode();
		List<TotalTimeCorrectionDtoInterface> list = timeReference().totalTimeCorrection()
			.getTotalTimeCorrectionHistory(personalId, DateUtility.getYear(targetDate),
					DateUtility.getMonth(targetDate));
		// 該当社員欄設定
		String[] aryLblCorrectionNumber = new String[list.size()];
		String[] aryLblCorrectionDate = new String[list.size()];
		String[] aryLblCorrectionEmployee = new String[list.size()];
		String[] aryLblCorrectionType = new String[list.size()];
		String[] aryLblCorrectionBefore = new String[list.size()];
		String[] aryLblCorrectionAfter = new String[list.size()];
		String[] aryLblCorrectionComment = new String[list.size()];
		// データ作成
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			TotalTimeCorrectionDtoInterface dto = list.get(i);
			// 配列に情報を設定
			aryLblCorrectionNumber[i] = String.valueOf(i + 1);
			aryLblCorrectionDate[i] = DateUtility.getStringDateAndDay(dto.getCorrectionDate()) + " "
					+ DateUtility.getStringTime(dto.getCorrectionDate());
			aryLblCorrectionEmployee[i] = getCheckEmployees(getEmployeeCode(dto.getCorrectionPersonalId()), targetDate);
			aryLblCorrectionType[i] = dto.getCorrectionType();
			aryLblCorrectionBefore[i] = dto.getCorrectionBefore();
			aryLblCorrectionAfter[i] = dto.getCorrectionAfter();
			aryLblCorrectionComment[i] = dto.getCorrectionReason();
		}
		vo.setAryLblCorrectionNumber(aryLblCorrectionNumber);
		vo.setAryLblCorrectionDate(aryLblCorrectionDate);
		vo.setAryLblCorrectionEmployee(aryLblCorrectionEmployee);
		vo.setAryLblCorrectionType(aryLblCorrectionType);
		vo.setAryLblCorrectionBefore(aryLblCorrectionBefore);
		vo.setAryLblCorrectionAfter(aryLblCorrectionAfter);
		vo.setAryLblCorrectionComment(aryLblCorrectionComment);
	}
	
	/**
	 * @param code 社員コード
	 * @param date 対象年月日
	 * @return 修正者情報
	 * @throws MospException 例外発生時
	 */
	public String getCheckEmployees(String code, Date date) throws MospException {
		AttendanceHistoryVo vo = (AttendanceHistoryVo)mospParams.getVo();
		if (code.equals(vo.getLblEmployeeCode())) {
			return mospParams.getName("FrontWithCornerParentheses") + mospParams.getName("Myself")
					+ mospParams.getName("BackWithCornerParentheses");
		} else {
			HumanDtoInterface humanDto = reference().human().getHumanInfoForEmployeeCode(code, date);
			if (humanDto == null) {
				return mospParams.getName("Hyphen");
			}
			return humanDto.getLastName() + " " + humanDto.getFirstName();
		}
	}
	
	/**
	 * 有効日を取得する。譲渡有効日が存在しない場合、VOの有効日を取得する。
	 * @return 有効日
	 */
	public Date getTransferredLocalDate() {
		AttendanceHistoryVo vo = (AttendanceHistoryVo)mospParams.getVo();
		String targetDateString = getTransferredActivateDate();
		if (null == targetDateString) {
			return DateUtility.getDate(vo.getAttendanceDate());
		} else {
			return DateUtility.getDate(targetDateString);
		}
	}
	
	/**
	 * 個人IDを取得する。譲渡IDが存在しない場合、VOのコードを取得する。
	 * @return 個人ID
	 * @throws MospException 例外発生時
	 */
	public String getTransferredLocalCode() throws MospException {
		AttendanceHistoryVo vo = (AttendanceHistoryVo)mospParams.getVo();
		String targetCode = getTransferredCode();
		if (null == targetCode) {
			return vo.getPersonalId();
		} else {
			return getPersonalId(targetCode);
		}
	}
}
