/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowCommentDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceCalcBeanInterface;
import jp.mosp.time.bean.AttendanceCorrectionReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceCorrectionRegistBeanInterface;
import jp.mosp.time.bean.AttendanceRegistBeanInterface;
import jp.mosp.time.bean.GoOutRegistBeanInterface;
import jp.mosp.time.bean.RestRegistBeanInterface;
import jp.mosp.time.bean.ScheduleDateReferenceBeanInterface;
import jp.mosp.time.bean.ScheduleReferenceBeanInterface;
import jp.mosp.time.bean.SubHolidayRegistBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dto.settings.AllowanceDtoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceCorrectionDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.GoOutDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.RestDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkTypeDtoInterface;
import jp.mosp.time.dto.settings.WorkTypeItemDtoInterface;
import jp.mosp.time.input.vo.AttendanceCardVo;

/**
 * 日々勤怠の詳細情報確認とその編集、更新を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_DRAFT}
 * </li><li>
 * {@link #CMD_APPLI}
 * </li><li>
 */

public class AttendanceCardAction extends TimeAction {
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 現在ログインしているユーザの日々勤怠の詳細情報表示画面へ遷移する。<br>
	 */
	public static final String	CMD_SELECT_SHOW	= "TM1201";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 操作者の指定にしたがって表示する対象年月を変更する。
	 * 前月・翌月ボタンといったものは現在表示している年月を基に画面遷移を行う。<br>
	 */
	public static final String	CMD_SEARCH		= "TM1202";
	
	/**
	 * 
	 */
	public static final String	CMD_FIXED		= "TM1203";
	
	/**
	 * 下書きコマンド。<br>
	 * <br>
	 * 編集チェックボックスにチェックを入れた上で編集した入力欄の内容を勤怠情報テーブルに登録し、
	 * 下書情報として保存する。<br>
	 */
	public static final String	CMD_DRAFT		= "TM1204";
	
	/**
	 * 申請コマンド。<br>
	 * <br>
	 * 編集チェックボックスにチェックを入れた上で編集した入力欄の内容を勤怠情報テーブルに登録し、
	 * 下書情報として保存する。	 * <br>
	 */
	public static final String	CMD_APPLI		= "TM1205";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public AttendanceCardAction() {
		super();
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new AttendanceCardVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 選択表示
			prepareVo();
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_FIXED)) {
			// 修正 
			prepareVo(true, false);
			changeMode();
		} else if (mospParams.getCommand().equals(CMD_DRAFT)) {
			// 下書き
			prepareVo();
			draft();
		} else if (mospParams.getCommand().equals(CMD_APPLI)) {
			// 申請
			prepareVo();
			appli();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 社員名/コード等の出力設定
		setEmployeeParams();
		// 初期値設定
		setDefaultValues();
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// プルダウンの設定
		setPulldown();
		// 休日出勤情報の設定
		chkWorkOnHolidayInfo();
	}
	
	/**
	 * 日付の更新後、該当日付の勤怠詳細を取得する。<br>
	 * @throws MospException 例外処理発生時
	 * 
	 */
	protected void search() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 現在表示日付取得
		Date date = DateUtility.getDate(vo.getLblYear(), vo.getLblMonth(), vo.getLblDay());
		if (getTransferredGenericCode() == null) {
			// NULLの場合は申請/下書処理から遷移してきたので日付の更新は行わない
		} else if (getTransferredGenericCode().equals(TimeConst.CODE_DATE_DECREMENT)) {
			date = DateUtility.addDay(date, -1);
		} else if (getTransferredGenericCode().equals(TimeConst.CODE_DATE_INCREMENT)) {
			date = DateUtility.addDay(date, 1);
		} else {
			date = DateUtility.getSystemDate();
		}
		vo.setLblYear(DateUtility.getStringYear(date));
		vo.setLblMonth(DateUtility.getStringMonth(date));
		vo.setLblDay(DateUtility.getStringDay(date));
		vo.setLblDate(DateUtility.getDate(vo.getLblYear(), vo.getLblMonth(), vo.getLblDay()));
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		// 初期値設定
		setDefaultValues();
		setPulldown();
	}
	
	/**
	* 下書処理を行う。<br>
	* @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	*/
	protected void draft() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTOの準備
		// 勤怠データ設定
		AttendanceDtoInterface dto = time().attendanceRegist().getInitDto();
		setAttendanceDtoFields(dto);
		// 休憩データ設定
		List<RestDtoInterface> restList = new ArrayList<RestDtoInterface>();
		setRestFields(restList);
		// 外出データ設定
		// 公用
		List<GoOutDtoInterface> goOutPublicList = new ArrayList<GoOutDtoInterface>();
		// 私用 
		List<GoOutDtoInterface> goOutPrivateList = new ArrayList<GoOutDtoInterface>();
		setGoOutDtoFields(goOutPublicList, goOutPrivateList);
		// 自動計算処理
		time().attendanceCalc().attendanceCalc(dto, restList, goOutPublicList, goOutPrivateList);
		// 妥当性チェック
		time().attendanceRegist().validate(dto);
		// 登録クラスの取得
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto == null) {
			workflowDto = workflowRegist.getInitDto();
			workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_WORK_MANGE);
		}
		workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
		// 登録後ワークフローの取得
		workflowDto = workflowRegist.draft(workflowDto, dto.getPersonalId(), dto.getWorkDate(),
				PlatformConst.WORKFLOW_TYPE_TIME);
		if (workflowDto != null) {
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(
					workflowDto,
					mospParams.getUser().getPersonalId(),
					mospParams.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
							new String[]{ mospParams.getName("WorkPaper") }));
			// ワークフロー番号セット
			dto.setWorkflow(workflowDto.getWorkflow());
			// 勤怠データ登録
			time().attendanceRegist().regist(dto);
			// 休憩データ登録
			for (int i = 0; i < restList.size(); i++) {
				time().restRegist().regist(restList.get(i));
			}
			// 外出データ登録
			for (int i = 0; i < goOutPublicList.size(); i++) {
				time().goOutRegist().regist(goOutPublicList.get(i));
			}
			for (int i = 0; i < goOutPrivateList.size(); i++) {
				time().goOutRegist().regist(goOutPrivateList.get(i));
			}
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 下書成功メッセージ設定
		addDraftMessage();
		// 検索
		search();
		// 承認者用プルダウン再設定
		setApprvalPullDown(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate());
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
	}
	
	/**
	 * 申請処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void appli() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		
		Date date = vo.getLblDate();
		// 個人ID
		String personalId = reference().human().getPersonalId(vo.getLblEmployeeCode(), date);
		int timesWork = 1;
		// 変更前データ取得
		// 勤怠データ
		AttendanceDtoInterface oldAttendance = timeReference().attendance().findForKey(personalId, date, timesWork);
		if (oldAttendance == null) {
			oldAttendance = time().attendanceRegist().getInitDto();
		}
		// 休暇データ
		List<RestDtoInterface> oldRestList = timeReference().rest().getRestList(personalId, date, timesWork);
		if (oldRestList == null) {
			oldRestList = new ArrayList<RestDtoInterface>();
		}
		// 外出データ
		List<GoOutDtoInterface> oldGoOutPublicList = timeReference().goOut().getGoOutList(personalId, date, timesWork,
				1);
		if (oldGoOutPublicList == null) {
			oldGoOutPublicList = new ArrayList<GoOutDtoInterface>();
		}
		// 外出データ
		List<GoOutDtoInterface> oldGoOutPrivateList = timeReference().goOut().getGoOutList(personalId, date, timesWork,
				2);
		if (oldGoOutPrivateList == null) {
			oldGoOutPrivateList = new ArrayList<GoOutDtoInterface>();
		}
		// 勤怠データ設定
		AttendanceDtoInterface dto = time().attendanceRegist().getInitDto();
		setAttendanceDtoFields(dto);
		// 休憩データ設定
		List<RestDtoInterface> restList = new ArrayList<RestDtoInterface>();
		setRestFields(restList);
		// 外出データ設定
		// 公用
		List<GoOutDtoInterface> goOutPublicList = new ArrayList<GoOutDtoInterface>();
		// 私用 
		List<GoOutDtoInterface> goOutPrivateList = new ArrayList<GoOutDtoInterface>();
		setGoOutDtoFields(goOutPublicList, goOutPrivateList);
		// 自動計算処理
		time().attendanceCalc().attendanceCalc(dto, restList, goOutPublicList, goOutPrivateList);
		// 申請の相関チェック
		time().attendanceRegist().checkAppli(dto);
		// 登録クラスの取得
		WorkflowRegistBeanInterface workflowRegist = platform().workflowRegist();
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		if (workflowDto == null) {
			workflowDto = workflowRegist.getInitDto();
			workflowDto.setFunctionCode(TimeConst.CODE_FUNCTION_WORK_MANGE);
		}
		workflowRegist.setDtoApproverIds(workflowDto, getSelectApproverIds());
		// 登録後ワークフローの取得
		workflowDto = workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getWorkDate(),
				PlatformConst.WORKFLOW_TYPE_TIME);
		if (workflowDto != null) {
			// 申請の相関チェック
			time().attendanceRegist().checkApprover(dto, workflowDto);
			// ワークフローコメント登録
			platform().workflowCommentRegist().addComment(workflowDto, mospParams.getUser().getPersonalId(),
					chkApproverSettingCommentName());
			// ワークフロー番号セット
			dto.setWorkflow(workflowDto.getWorkflow());
			// 勤怠データ登録
			time().attendanceRegist().regist(dto);
			// 休憩データ登録
			for (int i = 0; i < restList.size(); i++) {
				time().restRegist().regist(restList.get(i));
			}
			// 外出データ登録
			for (int i = 0; i < goOutPublicList.size(); i++) {
				time().goOutRegist().regist(goOutPublicList.get(i));
			}
			for (int i = 0; i < goOutPrivateList.size(); i++) {
				time().goOutRegist().regist(goOutPrivateList.get(i));
			}
			// 勤怠情報修正履歴の登録
			// 履歴登録クラス
			AttendanceCorrectionRegistBeanInterface correctionRegist = time().attendanceCorrectionRegist();
			AttendanceCorrectionReferenceBeanInterface correctionReference = timeReference().attendanceCorrection();
			// DTOの準備
			AttendanceCorrectionDtoInterface attendanceCorrectionDto = correctionRegist.getInitDto();
			// 勤怠データ修正セット
			// 勤怠データ履歴追加
			setAttendanceCorrectionDtoFields(attendanceCorrectionDto);
			correctionRegist.insertAttendance(attendanceCorrectionDto, oldAttendance, dto);
			// 手当情報
			// 休暇情報
			for (int i = 0; i < 6; i++) {
				if (i < oldRestList.size()) {
					if (correctionReference.chkRest(restList.get(i), oldRestList.get(i))) {
						correctionRegist.insertRest(attendanceCorrectionDto, oldRestList.get(i), restList.get(i));
					}
				} else {
					if (correctionReference.chkRest(restList.get(i), time().restRegist().getInitDto())) {
						correctionRegist.insertRest(attendanceCorrectionDto, time().restRegist().getInitDto(), restList
							.get(i));
					}
				}
			}
			// 外出公用
			for (int i = 0; i < 2; i++) {
				if (i < oldGoOutPublicList.size()) {
					if (correctionReference.chkGoOut(goOutPublicList.get(i), oldGoOutPublicList.get(i))) {
						correctionRegist.insertaGoOut(attendanceCorrectionDto, oldGoOutPublicList.get(i),
								goOutPublicList.get(i));
					}
				} else {
					if (correctionReference.chkGoOut(goOutPublicList.get(i), time().goOutRegist().getInitDto())) {
						correctionRegist.insertaGoOut(attendanceCorrectionDto, time().goOutRegist().getInitDto(),
								goOutPublicList.get(i));
					}
				}
			}
			// 外出私用
			for (int i = 0; i < 2; i++) {
				if (i < oldGoOutPublicList.size()) {
					if (correctionReference.chkGoOut(goOutPrivateList.get(i), oldGoOutPublicList.get(i))) {
						correctionRegist.insertaGoOut(attendanceCorrectionDto, oldGoOutPublicList.get(i),
								goOutPrivateList.get(i));
					}
				} else {
					if (correctionReference.chkGoOut(goOutPrivateList.get(i), time().goOutRegist().getInitDto())) {
						correctionRegist.insertaGoOut(attendanceCorrectionDto, time().goOutRegist().getInitDto(),
								goOutPrivateList.get(i));
					}
				}
			}
			// 代休データの設定
			getSubHolidayTimeInsert(dto);
		}
		// 登録結果確認
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 申請成功メッセージ設定
		addAppliMessage();
		// 検索
		search();
		// 状態
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNAUTHORIZED);
		// 編集モード設定
		vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
	}
	
	/**
	 * 出退勤情報
	 * @throws MospException 例外発生時
	 */
	protected void attendanceInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		Date localTargetDate = vo.getLblDate();
		// DTO準備
		// 勤怠が既に登録されている場合
		AttendanceDtoInterface dto = timeReference().attendance().findForKey(personalId, localTargetDate, 1);
		if (dto != null) {
			WorkTypeDtoInterface workTypeDto = timeReference().workType().getWorkTypeInfo(dto.getWorkTypeCode(),
					localTargetDate);
			// 修正情報
			vo.setTxtCorrectionReason("");
			// 出退勤情報
			vo.setLblCorrectionHistory("");
			// 勤務形態コード
			vo.setPltWorkType("");
			if (workTypeDto != null) {
				vo.setPltWorkType(workTypeDto.getWorkTypeCode());
				if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workTypeDto.getWorkTypeCode())
						|| TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(workTypeDto.getWorkTypeCode())) {
					WorkOnHolidayRequestDtoInterface workOnHolidayDto = timeReference().workOnHolidayRequest()
						.findForWorkflow(dto.getWorkflow());
					if (workOnHolidayDto == null) {
						vo.setJsModeSelectRead("HoliDayOff");
						vo.setJsModeHoliDay("on");
					} else {
						vo.setJsModeSelectRead("HoliDayOn");
						vo.setJsModeHoliDay("off");
					}
					if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workTypeDto.getWorkTypeCode())) {
						vo.setPltWorkType(TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY);
					} else {
						vo.setPltWorkType(TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY);
					}
				} else {
					vo.setJsModeSelectRead("HoliDayOff");
					vo.setJsModeHoliDay("off");
				}
			}
			vo.setTxtStartTimeHour(DateUtility.getStringHour(dto.getStartTime()));
			vo.setTxtStartTimeMinute(DateUtility.getStringMinute(dto.getStartTime()));
			vo.setTxtEndTimeHour(getEndTimeHour(dto.getStartTime(), dto.getEndTime()));
			vo.setTxtEndTimeMinute(DateUtility.getStringMinute(dto.getEndTime()));
			
			vo.setLblWorkTime(getTimeTimeFormat(dto.getWorkTime()));
			vo.setLblGeneralWorkTime(getTimeTimeFormat(dto.getGeneralWorkTime()));
			vo.setCkbDirectStart(String.valueOf(dto.getDirectStart()));
			vo.setCkbDirectEnd(String.valueOf(dto.getDirectEnd()));
			vo.setTxtTimeComment(dto.getTimeComment());
			vo.setTmdAttendanceId(String.valueOf(dto.getTmdAttendanceId()));
			vo.setPltLateReason(dto.getLateReason());
			vo.setPltLateCertificate(String.valueOf(dto.getLateCertificate()));
			vo.setTxtLateComment(dto.getLateComment());
			vo.setPltLeaveEarlyReason(dto.getLeaveEarlyReason());
			vo.setPltLeaveEarlyCertificate(String.valueOf(dto.getLeaveEarlyCertificate()));
			vo.setTxtLeaveEarlyComment(dto.getLeaveEarlyComment());
			
			vo.setLblLateTime(getTimeTimeFormat(dto.getLateTime()));
			vo.setLblLeaveEarlyTime(getTimeTimeFormat(dto.getLeaveEarlyTime()));
			vo.setLblOvertimeIn(getTimeTimeFormat(dto.getOvertime()));
			vo.setLblOvertimeOut(getTimeTimeFormat(dto.getOvertimeOut()));
			vo.setLblLateNightTime(getTimeTimeFormat(dto.getLateNightTime()));
			vo.setLblSpecificWorkTimeIn(getTimeTimeFormat(dto.getSpecificWorkTime()));
			vo.setLblLegalWorkTime(getTimeTimeFormat(dto.getLegalWorkTime()));
			vo.setLblDecreaseTime(getTimeTimeFormat(dto.getDecreaseTime()));
			// 後から必要になるかもしれないのでコメント化
			// dto.getNightRestTime();
			// dto.getOverRestTime();
			// dto.getOvertimeIn();
			// dto.getPrivateTime();
			// dto.getPublicTime();
			// dto.getRestTime();
			// 承認状況取得
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto != null) {
				if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
					// 下書状態
					vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_DRAFT);
				} else {
					// 下書状態以外
					vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNAUTHORIZED);
				}
			}
		} else {
			// 新規に登録する場合
			// 予定の時間を画面表示する
			ApplicationDtoInterface applicationDto = timeReference().application().findForPerson(personalId,
					localTargetDate);
			timeReference().application().chkExistApplication(applicationDto, localTargetDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			ScheduleDateDtoInterface scheduleDateDto = timeReference().scheduleDate().getScheduleDateInfo(
					applicationDto.getScheduleCode(), localTargetDate, localTargetDate);
			if (scheduleDateDto == null) {
				return;
			}
			WorkTypeDtoInterface workTypeDto = timeReference().workType().getWorkTypeInfo(
					scheduleDateDto.getWorkTypeCode(), localTargetDate);
			if (workTypeDto != null) {
				vo.setPltWorkType(workTypeDto.getWorkTypeCode());
				if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workTypeDto.getWorkTypeCode())
						|| TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(workTypeDto.getWorkTypeCode())) {
					vo.setJsModeSelectRead("HoliDayOff");
					vo.setJsModeHoliDay("on");
				} else {
					vo.setJsModeSelectRead("HoliDayOff");
					vo.setJsModeHoliDay("off");
				}
			} else {
				// 勤務形態が存在しない場合
				vo.setPltWorkType("");
			}
		}
		DifferenceRequestDtoInterface differenceDto = timeReference().differenceRequest().findForKeyOnWorkflow(
				personalId, localTargetDate);
		if (differenceDto == null) {
			return;
		}
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(differenceDto.getWorkflow());
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
			vo.setPltWorkType(differenceDto.getDifferenceType());
		}
	}
	
	/**
	 * 出退勤修正データ
	 * @throws MospException 例外発生時
	 */
	protected void attendanceCorrectInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceCorrectionDtoInterface dto = timeReference().attendanceCorrection()
			.getLatestAttendanceCorrectionInfo(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1);
		if (dto != null) {
			// 修正理由
			vo.setTxtCorrectionReason(dto.getCorrectionReason());
		}
	}
	
	/**
	 * 休憩情報
	 * @throws MospException 例外発生時
	 */
	protected void breakInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceDtoInterface attendanceDto = timeReference().attendance().findForKey(
				getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1);
		// 休憩情報
		if (attendanceDto == null) {
			Date targetDate = vo.getLblDate();
			// 休日出勤
			WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = timeReference().workOnHolidayRequest()
				.findForKeyOnWorkflow(vo.getPersonalId(), vo.getLblDate());
			if (workOnHolidayRequestDto != null) {
				long workflow = workOnHolidayRequestDto.getWorkflow();
				WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
				if (workflowDto != null && PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					int substitute = workOnHolidayRequestDto.getSubstitute();
					if (substitute == 1) {
						// 振替申請する場合
						List<SubstituteDtoInterface> substituteList = timeReference().substitute().getSubstituteList(
								workflow);
						for (SubstituteDtoInterface substituteDto : substituteList) {
							targetDate = substituteDto.getSubstituteDate();
							break;
						}
					} else if (substitute == 2) {
						// 振替申請しない場合
						return;
					}
				}
			}
			// 時差出勤
			DifferenceRequestDtoInterface differenceRequestDto = timeReference().differenceRequest()
				.findForKeyOnWorkflow(vo.getPersonalId(), vo.getLblDate());
			if (differenceRequestDto != null) {
				WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(
						differenceRequestDto.getWorkflow());
				if (workflowDto != null && PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					if (TimeConst.CODE_DIFFERENCE_TYPE_A.equals(differenceRequestDto.getDifferenceType())
							|| TimeConst.CODE_DIFFERENCE_TYPE_B.equals(differenceRequestDto.getDifferenceType())
							|| TimeConst.CODE_DIFFERENCE_TYPE_C.equals(differenceRequestDto.getDifferenceType())
							|| TimeConst.CODE_DIFFERENCE_TYPE_D.equals(differenceRequestDto.getDifferenceType())) {
						// 時差出勤区分A・B・C・D
						Date startTime = DateUtility.getTime(12, 0);
						Date endTime = DateUtility.getTime(13, 0);
						vo.setTxtRestStartHour1(DateUtility.getStringHour(startTime));
						vo.setTxtRestStartMinute1(DateUtility.getStringMinute(startTime));
						vo.setTxtRestEndHour1(DateUtility.getStringHour(endTime));
						vo.setTxtRestEndMinute1(DateUtility.getStringMinute(endTime));
						return;
					} else if (TimeConst.CODE_DIFFERENCE_TYPE_S.equals(differenceRequestDto.getDifferenceType())) {
						// 時差出勤区分S
						Date startTime = DateUtility.addHour(differenceRequestDto.getRequestStart(), 3);
						Date endTime = DateUtility.addHour(differenceRequestDto.getRequestStart(), 4);
						vo.setTxtRestStartHour1(DateUtility.getStringHour(startTime));
						vo.setTxtRestStartMinute1(DateUtility.getStringMinute(startTime));
						vo.setTxtRestEndHour1(DateUtility.getStringHour(endTime));
						vo.setTxtRestEndMinute1(DateUtility.getStringMinute(endTime));
						return;
					}
				}
			}
			ApplicationReferenceBeanInterface applicationReference = timeReference().application();
			ApplicationDtoInterface applicationDto = applicationReference.findForPerson(vo.getPersonalId(), targetDate);
			applicationReference.chkExistApplication(applicationDto, targetDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			ScheduleReferenceBeanInterface scheduleReference = timeReference().schedule();
			ScheduleDtoInterface scheduleDto = scheduleReference.getScheduleInfo(applicationDto.getScheduleCode(),
					targetDate);
			scheduleReference.chkExistSchedule(scheduleDto, targetDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			ScheduleDateReferenceBeanInterface scheduleDateReference = timeReference().scheduleDate();
			ScheduleDateDtoInterface scheduleDateDto = scheduleDateReference.getScheduleDateInfo(scheduleDto
				.getScheduleCode(), scheduleDto.getActivateDate(), targetDate);
			scheduleDateReference.chkExistScheduleDate(scheduleDateDto, targetDate);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			WorkTypeDtoInterface workTypeDto = timeReference().workType().getWorkTypeInfo(
					scheduleDateDto.getWorkTypeCode(), targetDate);
			// 休憩1
			WorkTypeItemDtoInterface restStart1Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTSTART1);
			WorkTypeItemDtoInterface restEnd1Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTEND1);
			if (restStart1Dto != null && restEnd1Dto != null) {
				vo.setTxtRestStartHour1(DateUtility.getStringHour(restStart1Dto.getWorkTypeItemValue()));
				vo.setTxtRestStartMinute1(DateUtility.getStringMinute(restStart1Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndHour1(DateUtility.getStringHour(restEnd1Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndMinute1(DateUtility.getStringMinute(restEnd1Dto.getWorkTypeItemValue()));
			}
			// 休憩2
			WorkTypeItemDtoInterface restStart2Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTSTART2);
			WorkTypeItemDtoInterface restEnd2Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTEND2);
			if (restStart2Dto != null && restEnd2Dto != null) {
				vo.setTxtRestStartHour2(DateUtility.getStringHour(restStart2Dto.getWorkTypeItemValue()));
				vo.setTxtRestStartMinute2(DateUtility.getStringMinute(restStart2Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndHour2(DateUtility.getStringHour(restEnd2Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndMinute2(DateUtility.getStringMinute(restEnd2Dto.getWorkTypeItemValue()));
			}
			// 休憩3
			WorkTypeItemDtoInterface restStart3Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTSTART3);
			WorkTypeItemDtoInterface restEnd3Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTEND3);
			if (restStart3Dto != null && restEnd3Dto != null) {
				vo.setTxtRestStartHour3(DateUtility.getStringHour(restStart3Dto.getWorkTypeItemValue()));
				vo.setTxtRestStartMinute3(DateUtility.getStringMinute(restStart3Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndHour3(DateUtility.getStringHour(restEnd3Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndMinute3(DateUtility.getStringMinute(restEnd3Dto.getWorkTypeItemValue()));
			}
			// 休憩4
			WorkTypeItemDtoInterface restStart4Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTSTART4);
			WorkTypeItemDtoInterface restEnd4Dto = timeReference().workTypeItem().getWorkTypeItemInfo(
					workTypeDto.getWorkTypeCode(), workTypeDto.getActivateDate(), TimeConst.CODE_RESTEND4);
			if (restStart4Dto != null && restEnd4Dto != null) {
				vo.setTxtRestStartHour4(DateUtility.getStringHour(restStart4Dto.getWorkTypeItemValue()));
				vo.setTxtRestStartMinute4(DateUtility.getStringMinute(restStart4Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndHour4(DateUtility.getStringHour(restEnd4Dto.getWorkTypeItemValue()));
				vo.setTxtRestEndMinute4(DateUtility.getStringMinute(restEnd4Dto.getWorkTypeItemValue()));
			}
		} else {
			vo.setLblRestTime(getTimeTimeFormat(attendanceDto.getRestTime()));
			vo.setLblOverRestTime(getTimeTimeFormat(attendanceDto.getOverRestTime()));
			vo.setLblNightRestTime(getTimeTimeFormat(attendanceDto.getNightRestTime()));
			List<RestDtoInterface> restDtoList = timeReference().rest().getRestList(
					getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1);
			for (RestDtoInterface restDto : restDtoList) {
				if (restDto.getRest() == 1) {
					vo.setTxtRestStartHour1(DateUtility.getStringHour(restDto.getRestStart()));
					vo.setTxtRestStartMinute1(DateUtility.getStringMinute(restDto.getRestStart()));
					vo.setTxtRestEndHour1(DateUtility.getStringHour(restDto.getRestEnd()));
					vo.setTxtRestEndMinute1(DateUtility.getStringMinute(restDto.getRestEnd()));
				}
				if (restDto.getRest() == 2) {
					vo.setTxtRestStartHour2(DateUtility.getStringHour(restDto.getRestStart()));
					vo.setTxtRestStartMinute2(DateUtility.getStringMinute(restDto.getRestStart()));
					vo.setTxtRestEndHour2(DateUtility.getStringHour(restDto.getRestEnd()));
					vo.setTxtRestEndMinute2(DateUtility.getStringMinute(restDto.getRestEnd()));
				}
				if (restDto.getRest() == 3) {
					vo.setTxtRestStartHour3(DateUtility.getStringHour(restDto.getRestStart()));
					vo.setTxtRestStartMinute3(DateUtility.getStringMinute(restDto.getRestStart()));
					vo.setTxtRestEndHour3(DateUtility.getStringHour(restDto.getRestEnd()));
					vo.setTxtRestEndMinute3(DateUtility.getStringMinute(restDto.getRestEnd()));
				}
				if (restDto.getRest() == 4) {
					vo.setTxtRestStartHour4(DateUtility.getStringHour(restDto.getRestStart()));
					vo.setTxtRestStartMinute4(DateUtility.getStringMinute(restDto.getRestStart()));
					vo.setTxtRestEndHour4(DateUtility.getStringHour(restDto.getRestEnd()));
					vo.setTxtRestEndMinute4(DateUtility.getStringMinute(restDto.getRestEnd()));
				}
				if (restDto.getRest() == 5) {
					vo.setTxtRestStartHour5(DateUtility.getStringHour(restDto.getRestStart()));
					vo.setTxtRestStartMinute5(DateUtility.getStringMinute(restDto.getRestStart()));
					vo.setTxtRestEndHour5(DateUtility.getStringHour(restDto.getRestEnd()));
					vo.setTxtRestEndMinute5(DateUtility.getStringMinute(restDto.getRestEnd()));
				}
				if (restDto.getRest() == 6) {
					vo.setTxtRestStartHour6(DateUtility.getStringHour(restDto.getRestStart()));
					vo.setTxtRestStartMinute6(DateUtility.getStringMinute(restDto.getRestStart()));
					vo.setTxtRestEndHour6(DateUtility.getStringHour(restDto.getRestEnd()));
					vo.setTxtRestEndMinute6(DateUtility.getStringMinute(restDto.getRestEnd()));
				}
			}
		}
	}
	
	/**
	 * 外出情報を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setGoOut() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceDtoInterface attendanceDto = timeReference().attendance().findForKey(
				getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1);
		if (attendanceDto != null) {
			vo.setLblPublicTime(getTimeTimeFormat(attendanceDto.getPublicTime()));
			vo.setLblPrivateTime(getTimeTimeFormat(attendanceDto.getPrivateTime()));
		}
		for (int i = 0; i < 2; i++) {
			for (int j = 0; j < 2; j++) {
				GoOutDtoInterface goOutDto = timeReference().goOut().findForKey(vo.getPersonalId(), vo.getLblDate(), 1,
						i + 1, j + 1);
				// データが存在しない場合
				if (goOutDto == null) {
					continue;
				}
				if (goOutDto.getGoOutType() == 1) {
					if (goOutDto.getTimesGoOut() == 1) {
						// 外出開始時刻。
						vo.setTxtPublicStartHour1(DateUtility.getStringHour(goOutDto.getGoOutStart()));
						vo.setTxtPublicStartMinute1(DateUtility.getStringMinute(goOutDto.getGoOutStart()));
						// 外出終了時刻。
						vo.setTxtPublicEndHour1(DateUtility.getStringHour(goOutDto.getGoOutEnd()));
						vo.setTxtPublicEndMinute1(DateUtility.getStringMinute(goOutDto.getGoOutEnd()));
					}
					if (goOutDto.getTimesGoOut() == 2) {
						// 外出開始時刻。
						vo.setTxtPublicStartHour2(DateUtility.getStringHour(goOutDto.getGoOutStart()));
						vo.setTxtPublicStartMinute2(DateUtility.getStringMinute(goOutDto.getGoOutStart()));
						// 外出終了時刻。
						vo.setTxtPublicEndHour2(DateUtility.getStringHour(goOutDto.getGoOutEnd()));
						vo.setTxtPublicEndMinute2(DateUtility.getStringMinute(goOutDto.getGoOutEnd()));
					}
				}
				// 私用
				if (goOutDto.getGoOutType() == 2) {
					if (goOutDto.getTimesGoOut() == 1) {
						// 外出開始時刻。
						vo.setTxtPrivateStartHour1(DateUtility.getStringHour(goOutDto.getGoOutStart()));
						vo.setTxtPrivateStartMinute1(DateUtility.getStringMinute(goOutDto.getGoOutStart()));
						// 外出終了時刻。
						vo.setTxtPrivateEndHour1(DateUtility.getStringHour(goOutDto.getGoOutEnd()));
						vo.setTxtPrivateEndMinute1(DateUtility.getStringMinute(goOutDto.getGoOutEnd()));
					}
					if (goOutDto.getTimesGoOut() == 2) {
						// 外出開始時刻。
						vo.setTxtPrivateStartHour2(DateUtility.getStringHour(goOutDto.getGoOutStart()));
						vo.setTxtPrivateStartMinute2(DateUtility.getStringMinute(goOutDto.getGoOutStart()));
						// 外出終了時刻。
						vo.setTxtPrivateEndHour2(DateUtility.getStringHour(goOutDto.getGoOutEnd()));
						vo.setTxtPrivateEndMinute2(DateUtility.getStringMinute(goOutDto.getGoOutEnd()));
					}
				}
			}
		}
	}
	
	/**
	 * 遅刻早退情報
	 * @throws MospException 例外発生時
	 */
	protected void tardinessLeaveearlyInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceDtoInterface dto = timeReference().attendance().findForKey(vo.getLblEmployeeCode(), vo.getLblDate(),
				1);
		// 遅刻早退情報
		if (dto != null) {
			vo.setLblLateTime(getTimeTimeFormat(dto.getLateTime()));
			vo.setPltLateReason(dto.getLateReason());
			vo.setPltLateCertificate(String.valueOf(dto.getLateCertificate()));
			vo.setTxtLateComment(dto.getLateComment());
			vo.setLblLeaveEarlyTime(getTimeTimeFormat(dto.getLeaveEarlyTime()));
			vo.setPltLeaveEarlyReason(dto.getLeaveEarlyReason());
			vo.setPltLeaveEarlyCertificate(String.valueOf(dto.getLeaveEarlyCertificate()));
			vo.setTxtLeaveEarlyComment(dto.getLeaveEarlyComment());
		}
	}
	
	/**
	 * 割増情報
	 * @throws MospException 例外発生時
	 */
	protected void premiumInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceDtoInterface dto = timeReference().attendance().findForKey(vo.getLblEmployeeCode(), vo.getLblDate(),
				1);
		// 割増情報
		if (dto != null) {
			vo.setLblOvertimeIn(getTimeTimeFormat(dto.getOvertime()));
			vo.setLblOvertimeOut(getTimeTimeFormat(dto.getOvertimeOut()));
			vo.setLblLateNightTime(getTimeTimeFormat(dto.getLateNightTime()));
			vo.setLblSpecificWorkTimeIn(getTimeTimeFormat(dto.getSpecificWorkTime()));
			vo.setLblSpecificWorkTimeOver("");
			vo.setLblLegalWorkTime(getTimeTimeFormat(dto.getLegalWorkTime()));
			vo.setLblDecreaseTime(getTimeTimeFormat(dto.getDecreaseTime()));
		}
	}
	
	/**
	 * 手当情報
	 * @throws MospException 例外発生時
	 */
	protected void allowanceInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 手当情報
		// 勤務回数は固定で「1」
		AllowanceDtoInterface dto = null;
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO1);
		if (dto != null) {
			vo.setPltAllowance1(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO2);
		if (dto != null) {
			vo.setPltAllowance2(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO3);
		if (dto != null) {
			vo.setPltAllowance3(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO4);
		if (dto != null) {
			vo.setPltAllowance4(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO5);
		if (dto != null) {
			vo.setPltAllowance5(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO6);
		if (dto != null) {
			vo.setPltAllowance6(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO7);
		if (dto != null) {
			vo.setPltAllowance7(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO8);
		if (dto != null) {
			vo.setPltAllowance8(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INFO9);
		if (dto != null) {
			vo.setPltAllowance9(String.valueOf(dto.getAllowance()));
		}
		dto = timeReference().allowance().findForKey(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), 1,
				TimeConst.CODE_ALLOWANCE_CODE_INF10);
		if (dto != null) {
			vo.setPltAllowance10(String.valueOf(dto.getAllowance()));
		}
	}
	
	/**
	 * 勤怠情報承認状況
	 * @throws MospException 例外発生時
	 */
	protected void attendanceInfoApprovalStatus() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceDtoInterface dto = timeReference().attendance().findForKey(getPersonalId(vo.getLblEmployeeCode()),
				vo.getLblDate(), 1);
		if (dto != null) {
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			WorkflowCommentDtoInterface commentDto = reference().workflowComment().getLatestWorkflowCommentInfo(
					dto.getWorkflow());
			// 勤怠情報承認状況
			if (workflowDto != null) {
				vo.setLblAttendanceState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
					.getWorkflowStage()));
				if (commentDto != null) {
					vo.setLblAttendanceComment(getWorkflowCommentDtoComment(workflowDto, commentDto));
					vo.setLblAttendanceApprover(getWorkflowCommentDtoLastFirstName(workflowDto, commentDto));
				}
				// 差戻フラグの設定
				if (workflowDto.getWorkflowStage() == 0
						&& workflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_REVERT)) {
					// 1次差戻であればフラグを立てる 
					vo.setJsModeReverting("1");
				}
			}
		}
	}
	
	/**
	 * 残業申請状況
	 * @throws MospException 例外発生時
	 */
	protected void overtimeApplicationStatus() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		AttendanceDtoInterface attendanceDto = timeReference().attendance().findForKey(vo.getPersonalId(),
				vo.getLblDate(), 1);
		List<OvertimeRequestDtoInterface> list = timeReference().overtimeRequest().getOvertimeRequestList(
				getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate(), vo.getLblDate());
		for (OvertimeRequestDtoInterface dto : list) {
			long workflow = dto.getWorkflow();
			// ワークフロー情報の取得
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			WorkflowCommentDtoInterface commentDto = reference().workflowComment().getLatestWorkflowCommentInfo(
					workflow);
			int overtimeType = dto.getOvertimeType();
			int requestTime = dto.getRequestTime();
			int workflowStage = workflowDto.getWorkflowStage();
			if (overtimeType == TimeConst.CODE_OVERTIME_WORK_BEFORE) {
				// 勤務前残業
				vo.setLblOvertimeType1(getOvertimeTypeName(TimeConst.CODE_OVERTIME_WORK_BEFORE));
				vo.setLblOvertimeSchedule1(getTimeTimeFormat(requestTime));
				vo.setLblOvertimeResult1(mospParams.getName("Hyphen"));
				if (attendanceDto != null) {
					vo.setLblOvertimeResult1(getTimeTimeFormat(attendanceDto.getOvertimeBefore()));
				}
				vo.setLblOvertimeReason1(dto.getRequestReason());
				vo.setLblOvertimeState1(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowStage));
				vo.setLblOvertimeComment1(getWorkflowCommentDtoComment(workflowDto, commentDto));
				vo.setLblOvertimeApprover1(getWorkflowCommentDtoLastFirstName(workflowDto, commentDto));
			} else if (overtimeType == TimeConst.CODE_OVERTIME_WORK_AFTER) {
				// 勤務後残業
				vo.setLblOvertimeType2(getOvertimeTypeName(TimeConst.CODE_OVERTIME_WORK_AFTER));
				vo.setLblOvertimeSchedule2(getTimeTimeFormat(requestTime));
				vo.setLblOvertimeResult2(mospParams.getName("Hyphen"));
				if (attendanceDto != null) {
					vo.setLblOvertimeResult2(getTimeTimeFormat(attendanceDto.getOvertimeAfter()));
				}
				vo.setLblOvertimeReason2(dto.getRequestReason());
				vo.setLblOvertimeState2(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowStage));
				vo.setLblOvertimeComment2(getWorkflowCommentDtoComment(workflowDto, commentDto));
				vo.setLblOvertimeApprover2(getWorkflowCommentDtoLastFirstName(workflowDto, commentDto));
			}
		}
	}
	
	/**
	 * 休暇申請状況
	 * @throws MospException 例外発生時
	 */
	protected void vacationApplicationStatus() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		List<HolidayRequestDtoInterface> list = timeReference().holidayRequest().getHolidayRequestList(
				vo.getPersonalId(), vo.getLblDate());
		List<HolidayRequestDtoInterface> holidayRequestList = new ArrayList<HolidayRequestDtoInterface>();
		for (HolidayRequestDtoInterface dto : list) {
			// ワークフロー情報の取得
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			holidayRequestList.add(dto);
		}
		int size = holidayRequestList.size();
		String[] aryLblHolidayType = new String[size];
		String[] aryLblHolidayLength = new String[size];
		String[] aryLblHolidayTime = new String[size];
		String[] aryLblHolidayReason = new String[size];
		String[] aryLblHolidayState = new String[size];
		String[] aryLblHolidayApprover = new String[size];
		String[] aryLblHolidayComment = new String[size];
		int i = 0;
		for (HolidayRequestDtoInterface dto : holidayRequestList) {
			long workflow = dto.getWorkflow();
			// ワークフロー情報の取得
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
			WorkflowCommentDtoInterface commentDto = reference().workflowComment().getLatestWorkflowCommentInfo(
					workflow);
			String holidayName = getHolidayTypeValueViewName(dto.getHolidayType1(), dto.getHolidayType2());
			String holidayAbbr = timeReference().holiday().getHolidayAbbr(dto.getHolidayType2(), vo.getLblDate(),
					dto.getHolidayType1());
			aryLblHolidayType[i] = holidayName + " " + holidayAbbr;
			aryLblHolidayLength[i] = getHolidayRange(dto.getHolidayRange());
			aryLblHolidayTime[i] = getTimeWaveFormat(dto.getStartTime(), dto.getEndTime());
			aryLblHolidayReason[i] = dto.getRequestReason();
			aryLblHolidayState[i] = getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
				.getWorkflowStage());
			aryLblHolidayApprover[i] = getWorkflowCommentDtoLastFirstName(workflowDto, commentDto);
			aryLblHolidayComment[i] = getWorkflowCommentDtoComment(workflowDto, commentDto);
			i++;
		}
		// VOに項目を設定
		vo.setLblHolidayType(aryLblHolidayType);
		vo.setLblHolidayLength(aryLblHolidayLength);
		vo.setLblHolidayTime(aryLblHolidayTime);
		vo.setLblHolidayReason(aryLblHolidayReason);
		vo.setLblHolidayState(aryLblHolidayState);
		vo.setLblHolidayComment(aryLblHolidayComment);
		vo.setLblHolidayApprover(aryLblHolidayApprover);
	}
	
	/**
	 * 休日出勤申請状況
	 * @throws MospException 例外発生時
	 */
	protected void holidayWorkApplicationStatus() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForKeyOnWorkflow(
				vo.getPersonalId(), vo.getLblDate());
		if (dto == null) {
			return;
		}
		long workflow = dto.getWorkflow();
		// ワークフロー情報の取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 取下の場合
			return;
		}
		WorkflowCommentDtoInterface commentDto = reference().workflowComment().getLatestWorkflowCommentInfo(
				dto.getWorkflow());
		// 振替休日情報の取得
		List<SubstituteDtoInterface> list = timeReference().substitute().getSubstituteList(workflow);
		// 休日出勤申請状況
		vo.setLblWorkOnHolidayDate(DateUtility.getStringDateAndDay(dto.getRequestDate()));
		vo.setLblWorkOnHolidayTime(getTimeWaveFormat(dto.getStartTime(), dto.getEndTime()));
		vo.setLblSubStituteDate("");
		for (SubstituteDtoInterface substituteDto : list) {
			vo.setLblSubStituteDate(DateUtility.getStringDateAndDay(substituteDto.getSubstituteDate()));
			break;
		}
		vo.setLblWorkOnHolidayReason(dto.getRequestReason());
		// 状態
		vo.setLblWorkOnHolidayState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		vo.setLblWorkOnHolidayComment(getWorkflowCommentDtoComment(workflowDto, commentDto));
		vo.setLblWorkOnHolidayApprover(getWorkflowCommentDtoLastFirstName(workflowDto, commentDto));
		// 承認状況が、承認済/承認解除/差戻の場合は勤務形態を変更
//		if (workflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_CANCEL)
//				|| workflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_COMPLETE)
//				|| workflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_REVERT)) {
//			SubHolidayRequestDtoInterface subHolidayDto = timeReference().subHolidayRequest().findForWorkflow(
//					workflowDto.getWorkflow());
//			if (subHolidayDto != null) {
//				if (subHolidayDto.getWorkDateSubHolidayType() == TimeConst.CODE_PRESCRIBED_SUBHOLIDAY_CODE) {
//					String[][] aryPltWorkType = { { "", "所定代休" } };
//					vo.setAryPltWorkType(aryPltWorkType);
//				} else if (subHolidayDto.getWorkDateSubHolidayType() == TimeConst.CODE_LEGAL_SUBHOLIDAY_CODE) {
//					String[][] aryPltWorkType = { { "", "法定代休" } };
//					vo.setAryPltWorkType(aryPltWorkType);
//				} else if (subHolidayDto.getWorkDateSubHolidayType() == TimeConst.CODE_MIDNIGHT_SUBHOLIDAY_CODE) {
//					String[][] aryPltWorkType = { { "", "深夜代休" } };
//					vo.setAryPltWorkType(aryPltWorkType);
//				}
//			}
//			vo.setPltWorkType("work_on_legal");
//		}
	}
	
	/**
	 * 代休申請状況
	 * @throws MospException 例外発生時
	 */
	protected void compensatoryleaveApplicationStatus() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		// 年月日は締め日
		List<SubHolidayRequestDtoInterface> list = timeReference().subHolidayRequest().getSubHolidayRequestList(
				vo.getPersonalId(), vo.getLblDate());
		List<SubHolidayRequestDtoInterface> subHolidayRequestList = new ArrayList<SubHolidayRequestDtoInterface>();
		for (SubHolidayRequestDtoInterface dto : list) {
			// ワークフロー情報の取得
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			subHolidayRequestList.add(dto);
		}
		int size = subHolidayRequestList.size();
		String[] aryLblSubHolidayDate = new String[size];
		String[] aryLblSubHolidayWorkDate = new String[size];
		String[] aryLblSubHolidayState = new String[size];
		String[] aryLblSubHolidayComment = new String[size];
		String[] aryLblSubHolidayApprover = new String[size];
		int i = 0;
		for (SubHolidayRequestDtoInterface dto : subHolidayRequestList) {
			long workflow = dto.getWorkflow();
			// ワークフロー情報の取得
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
			WorkflowCommentDtoInterface commentDto = reference().workflowComment().getLatestWorkflowCommentInfo(
					workflow);
			aryLblSubHolidayDate[i] = DateUtility.getStringDateAndDay(dto.getRequestDate());
			aryLblSubHolidayWorkDate[i] = DateUtility.getStringDateAndDay(dto.getWorkDate());
			aryLblSubHolidayState[i] = getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
				.getWorkflowStage());
			aryLblSubHolidayComment[i] = getWorkflowCommentDtoComment(workflowDto, commentDto);
			aryLblSubHolidayApprover[i] = getWorkflowCommentDtoLastFirstName(workflowDto, commentDto);
			i++;
		}
		// VOに項目を設定
		vo.setLblSubHolidayDate(aryLblSubHolidayDate);
		vo.setLblSubHolidayWorkDate(aryLblSubHolidayWorkDate);
		vo.setLblSubHolidayState(aryLblSubHolidayState);
		vo.setLblSubHolidayComment(aryLblSubHolidayComment);
		vo.setLblSubHolidayApprover(aryLblSubHolidayApprover);
	}
	
	/**
	 * 時差出勤申請状況
	 * @throws MospException 例外発生時
	 */
	protected void timedifferenceWorkApplicationStatus() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// DTO準備
		// 年月日は締め日
		DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKeyOnWorkflow(
				vo.getPersonalId(), vo.getLblDate());
		if (dto == null) {
			return;
		}
		long workflow = dto.getWorkflow();
		// ワークフロー情報の取得
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 取下の場合
			return;
		}
		WorkflowCommentDtoInterface commentDto = reference().workflowComment().getLatestWorkflowCommentInfo(workflow);
		// 時差出勤申請状況
		vo.setLblDifferenceDate(DateUtility.getStringDateAndDay(dto.getRequestDate()));
		vo.setLblDifferenceWorkType(getDifferenceTypeName(dto.getDifferenceType()));
		vo.setLblDifferenceWorkTime(getTimeWaveFormat(dto.getRequestStart(), dto.getRequestEnd()));
		vo.setLblDifferenceReason(dto.getRequestReason());
		// 状態
		vo.setLblDifferenceState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		vo.setLblDifferenceComment(getWorkflowCommentDtoComment(workflowDto, commentDto));
		vo.setLblDifferenceApprover(getWorkflowCommentDtoLastFirstName(workflowDto, commentDto));
	}
	
	/**
	 * 初期値を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	public void setDefaultValues() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 日付取得
		Date date;
		if (getTransferredActivateDate() != null) {
			date = DateUtility.getDate(getTransferredActivateDate());
			vo.setLblYear(DateUtility.getStringYear(date));
			vo.setLblMonth(DateUtility.getStringMonth(date));
			vo.setLblDay(DateUtility.getStringDay(date));
		} else if (vo.getLblDate() == null) {
			date = DateUtility.getSystemDate();
			vo.setLblYear(DateUtility.getStringYear(date));
			vo.setLblMonth(DateUtility.getStringMonth(date));
			vo.setLblDay(DateUtility.getStringDay(date));
		}
		// 
		vo.setLblDate(DateUtility.getDate(vo.getLblYear(), vo.getLblMonth(), vo.getLblDay()));
		// 各領域の初期化
		vo.setTmdAttendanceId("0");
		vo.setTxtCorrectionReason("");
		vo.setLblCorrectionHistory("");
		vo.setTxtStartTimeHour("");
		vo.setTxtStartTimeMinute("");
		vo.setTxtEndTimeHour("");
		vo.setTxtEndTimeMinute("");
		vo.setLblWorkTime("");
		vo.setLblRestTime("");
		vo.setLblOverRestTime("");
		vo.setLblNightRestTime("");
		vo.setLblPublicTime("");
		vo.setLblPrivateTime("");
		vo.setTxtRestStartHour1("");
		vo.setTxtRestStartMinute1("");
		vo.setTxtRestEndHour1("");
		vo.setTxtRestEndMinute1("");
		vo.setTxtRestStartHour2("");
		vo.setTxtRestStartMinute2("");
		vo.setTxtRestEndHour2("");
		vo.setTxtRestEndMinute2("");
		vo.setTxtRestStartHour3("");
		vo.setTxtRestStartMinute3("");
		vo.setTxtRestEndHour3("");
		vo.setTxtRestEndMinute3("");
		vo.setTxtRestStartHour4("");
		vo.setTxtRestStartMinute4("");
		vo.setTxtRestEndHour4("");
		vo.setTxtRestEndMinute4("");
		vo.setTxtRestStartHour5("");
		vo.setTxtRestStartMinute5("");
		vo.setTxtRestEndHour5("");
		vo.setTxtRestEndMinute5("");
		vo.setTxtRestStartHour6("");
		vo.setTxtRestStartMinute6("");
		vo.setTxtRestEndHour6("");
		vo.setTxtRestEndMinute6("");
		vo.setTxtPublicStartHour1("");
		vo.setTxtPublicStartMinute1("");
		vo.setTxtPublicEndHour1("");
		vo.setTxtPublicEndMinute1("");
		vo.setTxtPublicStartHour2("");
		vo.setTxtPublicStartMinute2("");
		vo.setTxtPublicEndHour2("");
		vo.setTxtPublicEndMinute2("");
		vo.setTxtPrivateStartHour1("");
		vo.setTxtPrivateStartMinute1("");
		vo.setTxtPrivateEndHour1("");
		vo.setTxtPrivateEndMinute1("");
		vo.setTxtPrivateStartHour2("");
		vo.setTxtPrivateStartMinute2("");
		vo.setTxtPrivateEndHour2("");
		vo.setTxtPrivateEndMinute2("");
		vo.setLblLateTime("");
		vo.setPltLateReason("");
		vo.setPltLateCertificate("");
		vo.setTxtLateComment("");
		vo.setLblLeaveEarlyTime("");
		vo.setPltLeaveEarlyReason("");
		vo.setPltLeaveEarlyCertificate("");
		vo.setTxtLeaveEarlyComment("");
		vo.setLblOvertimeIn("");
		vo.setLblOvertimeOut("");
		vo.setLblLateNightTime("");
		vo.setLblSpecificWorkTimeIn("");
		vo.setLblSpecificWorkTimeOver("");
		vo.setLblLegalWorkTime("");
		vo.setLblDecreaseTime("");
		vo.setLblAttendanceComment("");
		vo.setLblAttendanceState("");
		vo.setLblAttendanceApprover("");
		vo.setLblOvertimeType1("");
		vo.setLblOvertimeSchedule1("");
		vo.setLblOvertimeResult1("");
		vo.setLblOvertimeReason1("");
		vo.setLblOvertimeState1("");
		vo.setLblOvertimeComment1("");
		vo.setLblOvertimeApprover1("");
		vo.setLblOvertimeType2("");
		vo.setLblOvertimeSchedule2("");
		vo.setLblOvertimeResult2("");
		vo.setLblOvertimeReason2("");
		vo.setLblOvertimeState2("");
		vo.setLblOvertimeComment2("");
		vo.setLblOvertimeApprover2("");
		vo.setLblHolidayType(new String[0]);
		vo.setLblHolidayLength(new String[0]);
		vo.setLblHolidayTime(new String[0]);
		vo.setLblHolidayReason(new String[0]);
		vo.setLblHolidayState(new String[0]);
		vo.setLblHolidayComment(new String[0]);
		vo.setLblHolidayApprover(new String[0]);
		vo.setLblWorkOnHolidayDate("");
		vo.setLblWorkOnHolidayTime("");
		vo.setLblSubStituteDate("");
		vo.setLblWorkOnHolidayReason("");
		vo.setLblWorkOnHolidayState("");
		vo.setLblWorkOnHolidayComment("");
		vo.setLblWorkOnHolidayApprover("");
		vo.setLblSubHolidayDate(new String[0]);
		vo.setLblSubHolidayWorkDate(new String[0]);
		vo.setLblSubHolidayState(new String[0]);
		vo.setLblSubHolidayComment(new String[0]);
		vo.setLblSubHolidayApprover(new String[0]);
		vo.setLblDifferenceDate("");
		vo.setLblDifferenceWorkType("");
		vo.setLblDifferenceWorkTime("");
		vo.setLblDifferenceReason("");
		vo.setLblDifferenceState("");
		vo.setLblDifferenceComment("");
		vo.setLblDifferenceApprover("");
		vo.setLblGeneralWorkTime("");
		vo.setTxtTimeComment("");
		vo.setPltAllowance1("0");
		vo.setPltAllowance2("0");
		vo.setPltAllowance3("0");
		vo.setPltAllowance4("0");
		vo.setPltAllowance5("0");
		vo.setPltAllowance6("0");
		vo.setPltAllowance7("0");
		vo.setPltAllowance8("0");
		vo.setPltAllowance9("0");
		vo.setPltAllowance10("0");
		vo.setPltWorkType("");
		vo.setJsModeSelectRead("HoliDayOff");
		vo.setJsModeHoliDay("off");
		// 差戻フラグの初期化
		vo.setJsModeReverting("0");
		// 申請状況
		vo.setJsStatusApplication(TimeConst.CODE_APPLICATION_STATUS_UNCLAIMED);
		// 申請状況を設定する
		setApplicationStatus();
		boolean amFlag = false;
		boolean pmFlag = false;
		// 休暇申請
		List<HolidayRequestDtoInterface> holidayList = timeReference().holidayRequest().getHolidayRequestList(
				vo.getPersonalId(), vo.getLblDate());
		for (HolidayRequestDtoInterface requestDto : holidayList) {
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				continue;
			}
			if (requestDto.getHolidayRange() == 1) {
				// 全休
				vo.setJsModeSelectRead("HoliDayOn");
				return;
			} else if (requestDto.getHolidayRange() == 2) {
				// 午前休
				amFlag = true;
				continue;
			} else if (requestDto.getHolidayRange() == 3) {
				// 午後休
				pmFlag = true;
			}
		}
		// 代休申請
		List<SubHolidayRequestDtoInterface> subHolidayRequestDto = timeReference().subHolidayRequest()
			.getSubHolidayRequestList(vo.getPersonalId(), vo.getLblDate());
		for (SubHolidayRequestDtoInterface requestDto : subHolidayRequestDto) {
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				continue;
			}
			if (requestDto.getHolidayRange() == 1) {
				// 全休
				vo.setJsModeSelectRead("HoliDayOn");
				return;
			} else if (requestDto.getHolidayRange() == 2) {
				// 午前休
				amFlag = true;
				continue;
			} else if (requestDto.getHolidayRange() == 3) {
				// 午後休
				pmFlag = true;
			}
		}
		// 振替休日
		List<SubstituteDtoInterface> substituteList = timeReference().substitute().getSubstituteList(
				vo.getPersonalId(), vo.getLblDate());
		for (SubstituteDtoInterface substituteDto : substituteList) {
			WorkflowDtoInterface workflowDto = reference().workflow()
				.getLatestWorkflowInfo(substituteDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				continue;
			}
			if (substituteDto.getSubstituteRange() == 1) {
				// 全休
				vo.setJsModeSelectRead("HoliDayOn");
				return;
			} else if (substituteDto.getSubstituteRange() == 2) {
				// 午前休
				amFlag = true;
				continue;
			} else if (substituteDto.getSubstituteRange() == 3) {
				// 午後休
				pmFlag = true;
			}
		}
		if (amFlag && pmFlag) {
			vo.setJsModeSelectRead("HoliDayOn");
			return;
		}
		// 出退勤情報
		attendanceInfo();
		// 出退勤修正データ
		attendanceCorrectInfo();
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 休憩情報
		breakInfo();
		// 外出情報を設定する。
		setGoOut();
		// 遅刻早退情報
		tardinessLeaveearlyInfo();
		// 割増情報
		premiumInfo();
		// 手当情報
		allowanceInfo();
		// 勤怠情報承認状況
		attendanceInfoApprovalStatus();
		// 承認者欄の初期化
		String[] aryPltLblApproverSetting = new String[0];
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		vo.setPltApproverSetting1("");
		vo.setPltApproverSetting2("");
		vo.setPltApproverSetting3("");
		vo.setPltApproverSetting4("");
		vo.setPltApproverSetting5("");
		vo.setPltApproverSetting6("");
		vo.setPltApproverSetting7("");
		vo.setPltApproverSetting8("");
		vo.setPltApproverSetting9("");
		vo.setPltApproverSetting10("");
		// 所定/法定休日判定
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		int year = Integer.parseInt(vo.getLblYear());
		int month = Integer.parseInt(vo.getLblMonth());
		int lastday = getLastDay(year, month);
		Date startDate = DateUtility.getDate(year, month, 1);
		Date endDate = DateUtility.getDate(year, month, lastday);
		ApplicationDtoInterface appDto = timeReference().application().findForPerson(personalId, vo.getLblDate());
		if (appDto == null) {
			// 該当する設定適用が存在しない
			String errorMes = mospParams.getName("Set") + mospParams.getName("Apply");
			mospParams.addErrorMessage(TimeMessageConst.MSG_APPLICATION_UNSETTING,
					DateUtility.getStringDate(startDate), errorMes);
			vo.setJsModeSelectRead("HoliDayOn");
			return;
		}
		String[][] legalHoliday = timeReference().scheduleDate().getLegalHoliday(appDto.getScheduleCode(), startDate,
				endDate, personalId);
		if (legalHoliday.length != 0 && legalHoliday[0][0] != null) {
			for (String[] element : legalHoliday) {
				if (Integer.parseInt(element[0]) == Integer.parseInt(vo.getLblDay())) {
					WorkOnHolidayRequestDtoInterface workOnHolidayDto = timeReference().workOnHolidayRequest()
						.findForKeyOnWorkflow(personalId,
								DateUtility.getDate(vo.getLblYear(), vo.getLblMonth(), vo.getLblDay()));
					if (workOnHolidayDto != null) {
						vo.setJsModeSelectRead("HoliDayOff");
					} else {
						vo.setJsModeSelectRead("HoliDayOn");
					}
				}
			}
		}
		// リクエストされた値を設定する
		setTransferredValues();
		if (!"HoliDayOn".equals(vo.getJsModeSelectRead())) {
			AttendanceRegistBeanInterface regist = time().attendanceRegist();
			AttendanceDtoInterface dto = regist.getInitDto();
			dto.setPersonalId(vo.getPersonalId());
			dto.setWorkDate(vo.getLblDate());
			regist.checkTemporaryClosingFinal(dto);
			if (mospParams.hasErrorMessage()) {
				vo.setJsModeSelectRead("Cutoff");
			}
		}
	}
	
	/**
	 * 申請状況を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	private void setApplicationStatus() throws MospException {
		// 残業申請状況
		overtimeApplicationStatus();
		// 休暇申請状況
		vacationApplicationStatus();
		// 休日出勤申請状況
		holidayWorkApplicationStatus();
		// 代休申請状況
		compensatoryleaveApplicationStatus();
		// 時差出勤申請状況
		timedifferenceWorkApplicationStatus();
	}
	
	/**
	 * リクエストされた値を設定する。<br>
	 */
	private void setTransferredValues() {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		if (getTransferredStartHour() != null) {
			// 始業時刻
			vo.setTxtStartTimeHour(getTransferredStartHour());
			vo.setTxtStartTimeMinute(getTransferredStartMinute());
		}
		if (getTransferredEndHour() != null) {
			// 終業時刻
			vo.setTxtEndTimeHour(getTransferredEndHour());
			vo.setTxtEndTimeMinute(getTransferredEndMinute());
		}
		if (getTransferredDirectStart() != null) {
			// 直行
			vo.setCkbDirectStart(getTransferredDirectStart());
		}
		if (getTransferredDirectEnd() != null) {
			// 直帰
			vo.setCkbDirectEnd(getTransferredDirectEnd());
		}
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 編集項目設定
		vo.setAryPltAllowance(mospParams.getProperties().getCodeArray(TimeConst.CODE_ALLOWANCE, false));
		vo.setAryPltLateReason(mospParams.getProperties().getCodeArray(TimeConst.CODE_REASON_OF_LATE, true));
		vo.setAryPltLeaveEarlyReason(mospParams.getProperties()
			.getCodeArray(TimeConst.CODE_REASON_OF_LEAVE_EARLY, true));
		vo.setAryPltLateCertificate(mospParams.getProperties().getCodeArray(TimeConst.CODE_ALLOWANCE, true));
		// 時差出勤
		DifferenceRequestDtoInterface differenceDto = timeReference().differenceRequest().findForKeyOnWorkflow(
				getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate());
		WorkflowDtoInterface workflowDto = null;
		if (differenceDto != null) {
			workflowDto = reference().workflow().getLatestWorkflowInfo(differenceDto.getWorkflow());
		}
		if (vo.getJsModeSelectRead() != null
				&& (vo.getJsModeSelectRead().equals("HoliDayOff") && vo.getJsModeHoliDay().equals("on"))) {
			vo.setAryPltWorkType(mospParams.getProperties().getCodeArray(TimeConst.CODE_DIFFERENCE_TYPE, false));
		} else if (workflowDto != null && PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
			vo.setAryPltWorkType(timeReference().differenceRequest().getDifferenceSelectArray(
					differenceDto.getDifferenceType()));
		} else {
			vo.setAryPltWorkType(timeReference().workType().getStartTimeEndTimeSelectArray(vo.getLblDate()));
		}
		// 休日出勤
		chkWorkOnHolidayInfo();
	}
	
	/**
	 * 新規登録モードに設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void changeMode() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		setDefaultValues();
		// 編集モード設定
		if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_INSERT)) {
			vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_EDIT);
			setApprvalPullDown(getPersonalId(vo.getLblEmployeeCode()), vo.getLblDate());
			setPulldown();
		} else if (vo.getModeCardEdit().equals(PlatformConst.MODE_CARD_EDIT_EDIT)) {
			vo.setModeCardEdit(PlatformConst.MODE_CARD_EDIT_INSERT);
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setAttendanceDtoFields(AttendanceDtoInterface dto) throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 修正理由
		dto.setTmdAttendanceId(Long.valueOf(vo.getTmdAttendanceId()));
		// 出退勤情報
		dto.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		// 勤務形態
		dto.setWorkTypeCode(vo.getPltWorkType());
		// 勤務日
		dto.setWorkDate(vo.getLblDate());
		// 勤務回数
		dto.setTimesWork(1);
		// 出勤時刻
		dto.setStartTime(getAttendanceTime(vo.getTxtStartTimeHour(), vo.getTxtStartTimeMinute()));
		// 退勤時刻
		dto.setEndTime(getAttendanceTime(vo.getTxtEndTimeHour(), vo.getTxtEndTimeMinute()));
		// 直行チェックボックス
		if (vo.getCkbDirectStart().equals("on")) {
			dto.setDirectStart(1);
		} else {
			dto.setDirectStart(0);
		}
		// 直帰チェックボックス
		if (vo.getCkbDirectEnd().equals("on")) {
			dto.setDirectEnd(1);
		} else {
			dto.setDirectEnd(0);
		}
		// 勤怠コメント
		dto.setTimeComment(vo.getTxtTimeComment());
		// 遅刻理由
		dto.setLateReason(vo.getPltLateReason());
		// 遅刻証明書
		dto.setLateCertificate(vo.getPltLateCertificate());
		// 遅刻コメント
		dto.setLateComment(vo.getTxtLateComment());
		// 早退理由
		dto.setLeaveEarlyReason(vo.getPltLeaveEarlyReason());
		// 早退証明書
		dto.setLeaveEarlyCertificate(vo.getPltLeaveEarlyCertificate());
		// 早退コメント
		dto.setLeaveEarlyComment(vo.getTxtLeaveEarlyComment());
		// 勤務回数
		dto.setTimesWork(1);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @param num 手当てコード
	 * @throws MospException 例外発生時
	 */
	protected void setAllowanceDtoFields(AllowanceDtoInterface dto, int num) throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// ■手当情報入力欄
		// 個人ID
		dto.setPersonalId(getPersonalId(vo.getLblEmployeeCode()));
		// 勤務日。
		dto.setWorkDate(vo.getLblDate());
		// 勤務回数
		dto.setWorks(1);
		// 手当コード
		dto.setAllowanceCode(String.valueOf(num));
		// 手当
		if (num == 1) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance1()));
		}
		if (num == 2) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance2()));
		}
		if (num == 3) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance3()));
		}
		if (num == 4) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance4()));
		}
		if (num == 5) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance5()));
		}
		if (num == 6) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance6()));
		}
		if (num == 7) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance7()));
		}
		if (num == 8) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance8()));
		}
		if (num == 9) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance9()));
		}
		if (num == 10) {
			dto.setAllowance(Integer.parseInt(vo.getPltAllowance10()));
		}
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 */
	protected void setAttendanceCorrectionDtoFields(AttendanceCorrectionDtoInterface dto) {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 勤怠修正
		// 個人ID
		dto.setPersonalId(vo.getPersonalId());
		// 勤務日。
		dto.setWorkDate(vo.getLblDate());
		// 勤務回数
		dto.setWorks(1);
		// 修正理由
		dto.setCorrectionReason(vo.getTxtCorrectionReason());
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param restList 休憩DTOのLIST
	 * @throws MospException 例外発生時
	 */
	protected void setRestFields(List<RestDtoInterface> restList) throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		ApplicationReferenceBeanInterface applicationReference = timeReference().application();
		TimeSettingReferenceBeanInterface timeSettingReference = timeReference().timeSetting();
		RestRegistBeanInterface regist = time().restRegist();
		AttendanceCalcBeanInterface calc = time().attendanceCalc();
		ApplicationDtoInterface applicationDto = applicationReference
			.findForPerson(vo.getPersonalId(), vo.getLblDate());
		applicationReference.chkExistApplication(applicationDto, vo.getLblDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), vo.getLblDate());
		timeSettingReference.chkExistTimeSetting(timeSettingDto, vo.getLblDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 休暇
		Date date = vo.getLblDate();
		Date noDate = DateUtility.getDateTime(DateUtility.getYear(date), DateUtility.getMonth(date), DateUtility
			.getDay(date), 0, 0);
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		int timesWork = 1;
		List<RestDtoInterface> list = timeReference().rest().getRestList(personalId, date, timesWork);
		
		// 休憩1の設定
		RestDtoInterface rest1Dto = regist.getInitDto();
		// 個人ID
		rest1Dto.setPersonalId(personalId);
		// 勤務日。
		rest1Dto.setWorkDate(date);
		// 勤務回数。
		rest1Dto.setTimesWork(timesWork);
		// 休憩回数。
		rest1Dto.setRest(1);
		// 開始時刻。
		if (vo.getTxtRestStartHour1() != null && !vo.getTxtRestStartHour1().isEmpty()) {
			Date restStart = getAttendanceTime(vo.getTxtRestStartHour1(), vo.getTxtRestStartMinute1());
			Date restEnd = getAttendanceTime(vo.getTxtRestEndHour1(), vo.getTxtRestEndMinute1());
			// 休憩開始時刻。
			rest1Dto.setRestStart(restStart);
			// 休憩終了時刻。
			rest1Dto.setRestEnd(restEnd);
			Date roundRestStart = calc.getRoundMinute(restStart, timeSettingDto.getRoundDailyRestStart(),
					timeSettingDto.getRoundDailyRestStartUnit());
			Date roundRestEnd = calc.getRoundMinute(restEnd, timeSettingDto.getRoundDailyRestEnd(), timeSettingDto
				.getRoundDailyRestEndUnit());
			// 休憩時間。
			rest1Dto.setRestTime(calc.getRoundMinute(calc.getDefferenceMinutes(roundRestStart, roundRestEnd),
					timeSettingDto.getRoundDailyRestTime(), timeSettingDto.getRoundDailyRestTimeUnit()));
		} else {
			// 休憩開始時刻。
			rest1Dto.setRestStart(noDate);
			// 休憩終了時刻。
			rest1Dto.setRestEnd(noDate);
			// 休憩時間。
			rest1Dto.setRestTime(0);
		}
		// データが存在する場合、IDの追加
		if (list != null && 1 <= list.size()) {
			rest1Dto.setTmdRestId(list.get(0).getTmdRestId());
		}
		restList.add(rest1Dto);
		
		// 休憩2の設定
		RestDtoInterface rest2Dto = regist.getInitDto();
		// 個人ID
		rest2Dto.setPersonalId(personalId);
		// 勤務日。
		rest2Dto.setWorkDate(date);
		// 勤務回数。
		rest2Dto.setTimesWork(timesWork);
		// 休憩回数。
		rest2Dto.setRest(2);
		// 開始時刻。
		if (vo.getTxtRestStartHour2() != null && !vo.getTxtRestStartHour2().isEmpty()) {
			Date restStart = getAttendanceTime(vo.getTxtRestStartHour2(), vo.getTxtRestStartMinute2());
			Date restEnd = getAttendanceTime(vo.getTxtRestEndHour2(), vo.getTxtRestEndMinute2());
			// 休憩開始時刻。
			rest2Dto.setRestStart(restStart);
			// 休憩終了時刻。
			rest2Dto.setRestEnd(restEnd);
			Date roundRestStart = calc.getRoundMinute(restStart, timeSettingDto.getRoundDailyRestStart(),
					timeSettingDto.getRoundDailyRestStartUnit());
			Date roundRestEnd = calc.getRoundMinute(restEnd, timeSettingDto.getRoundDailyRestEnd(), timeSettingDto
				.getRoundDailyRestEndUnit());
			// 休憩時間。
			rest2Dto.setRestTime(calc.getRoundMinute(calc.getDefferenceMinutes(roundRestStart, roundRestEnd),
					timeSettingDto.getRoundDailyRestTime(), timeSettingDto.getRoundDailyRestTimeUnit()));
		} else {
			// 休憩開始時刻。
			rest2Dto.setRestStart(noDate);
			// 休憩終了時刻。
			rest2Dto.setRestEnd(noDate);
			// 休憩時間。
			rest2Dto.setRestTime(0);
		}
		// データが存在する場合、IDの追加
		if (list != null && 2 <= list.size()) {
			rest2Dto.setTmdRestId(list.get(1).getTmdRestId());
		}
		restList.add(rest2Dto);
		
		// 休憩3の設定
		RestDtoInterface rest3Dto = regist.getInitDto();
		// 個人ID
		rest3Dto.setPersonalId(personalId);
		// 勤務日。
		rest3Dto.setWorkDate(date);
		// 勤務回数。
		rest3Dto.setTimesWork(timesWork);
		// 休憩回数。
		rest3Dto.setRest(3);
		// 開始時刻
		if (vo.getTxtRestStartHour3() != null && !vo.getTxtRestStartHour3().isEmpty()) {
			Date restStart = getAttendanceTime(vo.getTxtRestStartHour3(), vo.getTxtRestStartMinute3());
			Date restEnd = getAttendanceTime(vo.getTxtRestEndHour3(), vo.getTxtRestEndMinute3());
			// 休憩開始時刻。
			rest3Dto.setRestStart(restStart);
			// 休憩終了時刻。
			rest3Dto.setRestEnd(restEnd);
			Date roundRestStart = calc.getRoundMinute(restStart, timeSettingDto.getRoundDailyRestStart(),
					timeSettingDto.getRoundDailyRestStartUnit());
			Date roundRestEnd = calc.getRoundMinute(restEnd, timeSettingDto.getRoundDailyRestEnd(), timeSettingDto
				.getRoundDailyRestEndUnit());
			// 休憩時間。
			rest3Dto.setRestTime(calc.getRoundMinute(calc.getDefferenceMinutes(roundRestStart, roundRestEnd),
					timeSettingDto.getRoundDailyRestTime(), timeSettingDto.getRoundDailyRestTimeUnit()));
		} else {
			// 休憩開始時刻。
			rest3Dto.setRestStart(noDate);
			// 休憩終了時刻。
			rest3Dto.setRestEnd(noDate);
			// 休憩時間。
			rest3Dto.setRestTime(0);
		}
		// データが存在する場合、IDの追加
		if (list != null && 3 <= list.size()) {
			rest3Dto.setTmdRestId(list.get(2).getTmdRestId());
		}
		restList.add(rest3Dto);
		
		// 休憩4の設定
		RestDtoInterface rest4Dto = regist.getInitDto();
		// 個人ID
		rest4Dto.setPersonalId(personalId);
		// 勤務日。
		rest4Dto.setWorkDate(date);
		// 勤務回数。
		rest4Dto.setTimesWork(timesWork);
		// 休憩回数。
		rest4Dto.setRest(4);
		// 開始時刻
		if (vo.getTxtRestStartHour4() != null && !vo.getTxtRestStartHour4().isEmpty()) {
			Date restStart = getAttendanceTime(vo.getTxtRestStartHour4(), vo.getTxtRestStartMinute4());
			Date restEnd = getAttendanceTime(vo.getTxtRestEndHour4(), vo.getTxtRestEndMinute4());
			// 休憩開始時刻。
			rest4Dto.setRestStart(restStart);
			// 休憩終了時刻。
			rest4Dto.setRestEnd(restEnd);
			Date roundRestStart = calc.getRoundMinute(restStart, timeSettingDto.getRoundDailyRestStart(),
					timeSettingDto.getRoundDailyRestStartUnit());
			Date roundRestEnd = calc.getRoundMinute(restEnd, timeSettingDto.getRoundDailyRestEnd(), timeSettingDto
				.getRoundDailyRestEndUnit());
			// 休憩時間。
			rest4Dto.setRestTime(calc.getRoundMinute(calc.getDefferenceMinutes(roundRestStart, roundRestEnd),
					timeSettingDto.getRoundDailyRestTime(), timeSettingDto.getRoundDailyRestTimeUnit()));
		} else {
			// 休憩開始時刻。
			rest4Dto.setRestStart(noDate);
			// 休憩終了時刻。
			rest4Dto.setRestEnd(noDate);
			// 休憩時間。
			rest4Dto.setRestTime(0);
		}
		// データが存在する場合、IDの追加
		if (list != null && 4 <= list.size()) {
			rest4Dto.setTmdRestId(list.get(3).getTmdRestId());
		}
		restList.add(rest4Dto);
		
		// 休憩5の設定
		RestDtoInterface rest5Dto = regist.getInitDto();
		// 個人ID
		rest5Dto.setPersonalId(personalId);
		// 勤務日。
		rest5Dto.setWorkDate(date);
		// 勤務回数。
		rest5Dto.setTimesWork(timesWork);
		// 休憩回数。
		rest5Dto.setRest(5);
		// 開始時刻
		if (vo.getTxtRestStartHour5() != null && !vo.getTxtRestStartHour5().isEmpty()) {
			Date restStart = getAttendanceTime(vo.getTxtRestStartHour5(), vo.getTxtRestStartMinute5());
			Date restEnd = getAttendanceTime(vo.getTxtRestEndHour5(), vo.getTxtRestEndMinute5());
			// 休憩開始時刻。
			rest5Dto.setRestStart(restStart);
			// 休憩終了時刻。
			rest5Dto.setRestEnd(restEnd);
			Date roundRestStart = calc.getRoundMinute(restStart, timeSettingDto.getRoundDailyRestStart(),
					timeSettingDto.getRoundDailyRestStartUnit());
			Date roundRestEnd = calc.getRoundMinute(restEnd, timeSettingDto.getRoundDailyRestEnd(), timeSettingDto
				.getRoundDailyRestEndUnit());
			// 休憩時間。
			rest5Dto.setRestTime(calc.getRoundMinute(calc.getDefferenceMinutes(roundRestStart, roundRestEnd),
					timeSettingDto.getRoundDailyRestTime(), timeSettingDto.getRoundDailyRestTimeUnit()));
		} else {
			// 休憩開始時刻。
			rest5Dto.setRestStart(noDate);
			// 休憩終了時刻。
			rest5Dto.setRestEnd(noDate);
			// 休憩時間。
			rest5Dto.setRestTime(0);
		}
		// データが存在する場合、IDの追加
		if (list != null && 5 <= list.size()) {
			rest5Dto.setTmdRestId(list.get(4).getTmdRestId());
		}
		restList.add(rest5Dto);
		
		// 休憩6の設定
		RestDtoInterface rest6Dto = regist.getInitDto();
		// 個人ID
		rest6Dto.setPersonalId(personalId);
		// 勤務日。
		rest6Dto.setWorkDate(date);
		// 勤務回数。
		rest6Dto.setTimesWork(timesWork);
		// 休憩回数。
		rest6Dto.setRest(6);
		// 開始時刻。
		if (vo.getTxtRestStartHour6() != null && !vo.getTxtRestStartHour6().isEmpty()) {
			Date restStart = getAttendanceTime(vo.getTxtRestStartHour6(), vo.getTxtRestStartMinute6());
			Date restEnd = getAttendanceTime(vo.getTxtRestEndHour6(), vo.getTxtRestEndMinute6());
			// 休憩開始時刻。
			rest6Dto.setRestStart(restStart);
			// 休憩終了時刻。
			rest6Dto.setRestEnd(restEnd);
			Date roundRestStart = calc.getRoundMinute(restStart, timeSettingDto.getRoundDailyRestStart(),
					timeSettingDto.getRoundDailyRestStartUnit());
			Date roundRestEnd = calc.getRoundMinute(restEnd, timeSettingDto.getRoundDailyRestEnd(), timeSettingDto
				.getRoundDailyRestEndUnit());
			// 休憩時間。
			rest6Dto.setRestTime(calc.getRoundMinute(calc.getDefferenceMinutes(roundRestStart, roundRestEnd),
					timeSettingDto.getRoundDailyRestTime(), timeSettingDto.getRoundDailyRestTimeUnit()));
		} else {
			// 休憩開始時刻。
			rest6Dto.setRestStart(noDate);
			// 休憩終了時刻。
			rest6Dto.setRestEnd(noDate);
			// 休憩時間。
			rest6Dto.setRestTime(0);
		}
		// データが存在する場合、IDの追加
		if (list != null && 6 <= list.size()) {
			rest6Dto.setTmdRestId(list.get(5).getTmdRestId());
		}
		restList.add(rest6Dto);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param publicList 公用DTOのLIST
	 * @param privateList 外用DTOのLIST
	 * @throws MospException 例外発生時
	 */
	protected void setGoOutDtoFields(List<GoOutDtoInterface> publicList, List<GoOutDtoInterface> privateList)
			throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		ApplicationReferenceBeanInterface applicationReference = timeReference().application();
		TimeSettingReferenceBeanInterface timeSettingReference = timeReference().timeSetting();
		GoOutRegistBeanInterface regist = time().goOutRegist();
		AttendanceCalcBeanInterface calc = time().attendanceCalc();
		ApplicationDtoInterface applicationDto = applicationReference
			.findForPerson(vo.getPersonalId(), vo.getLblDate());
		applicationReference.chkExistApplication(applicationDto, vo.getLblDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), vo.getLblDate());
		timeSettingReference.chkExistTimeSetting(timeSettingDto, vo.getLblDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		Date date = vo.getLblDate();
		Date noDate = DateUtility.getDateTime(DateUtility.getYear(date), DateUtility.getMonth(date), DateUtility
			.getDay(date), 0, 0);
		String personalId = getPersonalId(vo.getLblEmployeeCode());
		int timesWork = 1;
		//
		List<GoOutDtoInterface> goOutPublicList = timeReference().goOut().getGoOutList(personalId, date, timesWork, 1);
		List<GoOutDtoInterface> goOutPrivateList = timeReference().goOut().getGoOutList(personalId, date, timesWork, 2);
		
		// 公用1の設定
		GoOutDtoInterface goOutPublic1Dto = regist.getInitDto();
		// 個人ID
		goOutPublic1Dto.setPersonalId(personalId);
		// 勤務日。
		goOutPublic1Dto.setWorkDate(date);
		// 勤務回数。
		goOutPublic1Dto.setTimesWork(timesWork);
		// 外出区分。
		goOutPublic1Dto.setGoOutType(1);
		// 外出回数。
		goOutPublic1Dto.setTimesGoOut(1);
		// 時刻設定
		if (vo.getTxtPublicStartHour1() != null && !vo.getTxtPublicStartHour1().isEmpty()) {
			Date goOutStart = getAttendanceTime(vo.getTxtPublicStartHour1(), vo.getTxtPublicStartMinute1());
			Date goOutEnd = getAttendanceTime(vo.getTxtPublicEndHour1(), vo.getTxtPublicEndMinute1());
			// 公用開始時刻。
			goOutPublic1Dto.setGoOutStart(goOutStart);
			// 公用終了時刻。
			goOutPublic1Dto.setGoOutEnd(goOutEnd);
			Date roundGoOutStart = calc.getRoundMinute(goOutStart, timeSettingDto.getRoundDailyPublicStart(),
					timeSettingDto.getRoundDailyPublicStartUnit());
			Date roundGoOutEnd = calc.getRoundMinute(goOutEnd, timeSettingDto.getRoundDailyPublicEnd(), timeSettingDto
				.getRoundDailyPublicEndUnit());
			// 外出時間。
			goOutPublic1Dto.setGoOutTime(calc.getDefferenceMinutes(roundGoOutStart, roundGoOutEnd));
		} else {
			// 公用開始時刻。
			goOutPublic1Dto.setGoOutStart(noDate);
			// 公用終了時刻。
			goOutPublic1Dto.setGoOutEnd(noDate);
			// 外出時間。
			goOutPublic1Dto.setGoOutTime(0);
		}
		// データが存在する場合、IDの追加
		if (goOutPublicList != null && 1 <= goOutPublicList.size()) {
			goOutPublic1Dto.setTmdGoOutId(goOutPublicList.get(0).getTmdGoOutId());
		}
		publicList.add(goOutPublic1Dto);
		
		// 公用2の設定
		GoOutDtoInterface goOutPublic2Dto = regist.getInitDto();
		// 個人ID
		goOutPublic2Dto.setPersonalId(personalId);
		// 勤務日。
		goOutPublic2Dto.setWorkDate(date);
		// 勤務回数。
		goOutPublic2Dto.setTimesWork(timesWork);
		// 外出区分。
		goOutPublic2Dto.setGoOutType(1);
		// 外出回数。
		goOutPublic2Dto.setTimesGoOut(2);
		// 時刻設定
		if (vo.getTxtPublicStartHour2() != null && !vo.getTxtPublicStartHour2().isEmpty()) {
			Date goOutStart = getAttendanceTime(vo.getTxtPublicStartHour2(), vo.getTxtPublicStartMinute2());
			Date goOutEnd = getAttendanceTime(vo.getTxtPublicEndHour2(), vo.getTxtPublicEndMinute2());
			// 公用開始時刻。
			goOutPublic2Dto.setGoOutStart(goOutStart);
			// 公用終了時刻。
			goOutPublic2Dto.setGoOutEnd(goOutEnd);
			Date roundGoOutStart = calc.getRoundMinute(goOutStart, timeSettingDto.getRoundDailyPublicStart(),
					timeSettingDto.getRoundDailyPublicStartUnit());
			Date roundGoOutEnd = calc.getRoundMinute(goOutEnd, timeSettingDto.getRoundDailyPublicEnd(), timeSettingDto
				.getRoundDailyPublicEndUnit());
			// 外出時間。
			goOutPublic2Dto.setGoOutTime(calc.getDefferenceMinutes(roundGoOutStart, roundGoOutEnd));
		} else {
			// 公用開始時刻。
			goOutPublic2Dto.setGoOutStart(noDate);
			// 公用終了時刻。
			goOutPublic2Dto.setGoOutEnd(noDate);
			// 外出時間。
			goOutPublic2Dto.setGoOutTime(0);
		}
		// データが存在する場合、IDの追加
		if (goOutPublicList != null && 2 <= goOutPublicList.size()) {
			goOutPublic2Dto.setTmdGoOutId(goOutPublicList.get(1).getTmdGoOutId());
		}
		publicList.add(goOutPublic2Dto);
		// 私用1の設定
		GoOutDtoInterface goOutPrivate1Dto = regist.getInitDto();
		// 個人ID
		goOutPrivate1Dto.setPersonalId(personalId);
		// 勤務日。
		goOutPrivate1Dto.setWorkDate(date);
		// 勤務回数。
		goOutPrivate1Dto.setTimesWork(timesWork);
		// 外出区分。
		goOutPrivate1Dto.setGoOutType(2);
		// 外出回数。
		goOutPrivate1Dto.setTimesGoOut(1);
		// 時刻設定
		if (vo.getTxtPrivateStartHour1() != null && !vo.getTxtPrivateStartHour1().isEmpty()) {
			Date goOutStart = getAttendanceTime(vo.getTxtPrivateStartHour1(), vo.getTxtPrivateStartMinute1());
			Date goOutEnd = getAttendanceTime(vo.getTxtPrivateEndHour1(), vo.getTxtPrivateEndMinute1());
			// 私用開始時刻。
			goOutPrivate1Dto.setGoOutStart(goOutStart);
			// 私用終了時刻。
			goOutPrivate1Dto.setGoOutEnd(goOutEnd);
			Date roundGoOutStart = calc.getRoundMinute(goOutStart, timeSettingDto.getRoundDailyPrivateStart(),
					timeSettingDto.getRoundDailyPrivateStartUnit());
			Date roundGoOutEnd = calc.getRoundMinute(goOutEnd, timeSettingDto.getRoundDailyPrivateEnd(), timeSettingDto
				.getRoundDailyPrivateEndUnit());
			// 外出時間。
			goOutPrivate1Dto.setGoOutTime(calc.getDefferenceMinutes(roundGoOutStart, roundGoOutEnd));
		} else {
			// 私用開始時刻。
			goOutPrivate1Dto.setGoOutStart(noDate);
			// 私用終了時刻。
			goOutPrivate1Dto.setGoOutEnd(noDate);
			// 外出時間。
			goOutPrivate1Dto.setGoOutTime(0);
		}
		// データが存在する場合、IDの追加
		if (goOutPrivateList != null && 1 <= goOutPrivateList.size()) {
			goOutPrivate1Dto.setTmdGoOutId(goOutPrivateList.get(0).getTmdGoOutId());
		}
		privateList.add(goOutPrivate1Dto);
		
		// 私用2の設定
		GoOutDtoInterface goOutPrivate2Dto = regist.getInitDto();
		// 個人ID
		goOutPrivate2Dto.setPersonalId(personalId);
		// 勤務日。
		goOutPrivate2Dto.setWorkDate(date);
		// 勤務回数。
		goOutPrivate2Dto.setTimesWork(timesWork);
		// 外出区分。
		goOutPrivate2Dto.setGoOutType(2);
		// 外出回数。
		goOutPrivate2Dto.setTimesGoOut(2);
		// 時刻設定
		if (vo.getTxtPrivateStartHour2() != null && !vo.getTxtPrivateStartHour2().isEmpty()) {
			Date goOutStart = getAttendanceTime(vo.getTxtPrivateStartHour2(), vo.getTxtPrivateStartMinute2());
			Date goOutEnd = getAttendanceTime(vo.getTxtPrivateEndHour2(), vo.getTxtPrivateEndMinute2());
			// 私用開始時刻。
			goOutPrivate2Dto.setGoOutStart(goOutStart);
			// 私用終了時刻。
			goOutPrivate2Dto.setGoOutEnd(goOutEnd);
			Date roundGoOutStart = calc.getRoundMinute(goOutStart, timeSettingDto.getRoundDailyPrivateStart(),
					timeSettingDto.getRoundDailyPrivateStartUnit());
			Date roundGoOutEnd = calc.getRoundMinute(goOutEnd, timeSettingDto.getRoundDailyPrivateEnd(), timeSettingDto
				.getRoundDailyPrivateEndUnit());
			// 外出時間。
			goOutPrivate2Dto.setGoOutTime(calc.getDefferenceMinutes(roundGoOutStart, roundGoOutEnd));
		} else {
			// 私用開始時刻。
			goOutPrivate2Dto.setGoOutStart(noDate);
			// 私用終了時刻。
			goOutPrivate2Dto.setGoOutEnd(noDate);
			// 外出時間。
			goOutPrivate2Dto.setGoOutTime(0);
		}
		// データが存在する場合、IDの追加
		if (goOutPrivateList != null && 2 <= goOutPrivateList.size()) {
			goOutPrivate2Dto.setTmdGoOutId(goOutPrivateList.get(1).getTmdGoOutId());
		}
		privateList.add(goOutPrivate2Dto);
	}
	
	/**
	 * 休憩/公用外出/私用外出重複チェック。<br>
	 * @throws MospException 例外発生時
	 */
	protected void restCheck() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 始業時刻
		int startTimeHour = 0;
		if (vo.getTxtStartTimeHour().length() != 0) {
			startTimeHour = Integer.parseInt(vo.getTxtStartTimeHour());
		}
		int startTimeMinute = 0;
		if (vo.getTxtStartTimeMinute().length() != 0) {
			startTimeMinute = Integer.parseInt(vo.getTxtStartTimeMinute());
		}
		// 終業時刻
		int endTimeHour = 0;
		if (vo.getTxtEndTimeHour().length() != 0) {
			endTimeHour = Integer.parseInt(vo.getTxtEndTimeHour());
		}
		int endTimeMinute = 0;
		if (vo.getTxtEndTimeMinute().length() != 0) {
			endTimeMinute = Integer.parseInt(vo.getTxtEndTimeMinute());
		}
		// 休憩1
		int restStartTime1Hour = 0;
		if (vo.getTxtRestStartHour1().length() != 0) {
			restStartTime1Hour = Integer.parseInt(vo.getTxtRestStartHour1());
		}
		int restStartTime1Minute = 0;
		if (vo.getTxtRestStartMinute1().length() != 0) {
			restStartTime1Minute = Integer.parseInt(vo.getTxtRestStartMinute1());
		}
		int restEndTime1Hour = 0;
		if (vo.getTxtRestEndHour1().length() != 0) {
			restEndTime1Hour = Integer.parseInt(vo.getTxtRestEndHour1());
		}
		int restEndTime1Minute = 0;
		if (vo.getTxtRestEndMinute1().length() != 0) {
			restEndTime1Minute = Integer.parseInt(vo.getTxtRestEndMinute1());
		}
		// 休憩2
		int restStartTime2Hour = 0;
		if (vo.getTxtRestStartHour2().length() != 0) {
			restStartTime2Hour = Integer.parseInt(vo.getTxtRestStartHour2());
		}
		int restStartTime2Minute = 0;
		if (vo.getTxtRestStartMinute2().length() != 0) {
			restStartTime2Minute = Integer.parseInt(vo.getTxtRestStartMinute2());
		}
		int restEndTime2Hour = 0;
		if (vo.getTxtRestEndHour2().length() != 0) {
			restEndTime2Hour = Integer.parseInt(vo.getTxtRestEndHour2());
		}
		int restEndTime2Minute = 0;
		if (vo.getTxtRestEndMinute2().length() != 0) {
			restEndTime2Minute = Integer.parseInt(vo.getTxtRestEndMinute2());
		}
		// 休憩3
		int restStartTime3Hour = 0;
		if (vo.getTxtRestStartHour3().length() != 0) {
			restStartTime3Hour = Integer.parseInt(vo.getTxtRestStartHour3());
		}
		int restStartTime3Minute = 0;
		if (vo.getTxtRestStartMinute3().length() != 0) {
			restStartTime3Minute = Integer.parseInt(vo.getTxtRestStartMinute3());
		}
		int restEndTime3Hour = 0;
		if (vo.getTxtRestEndHour3().length() != 0) {
			restEndTime3Hour = Integer.parseInt(vo.getTxtRestEndHour3());
		}
		int restEndTime3Minute = 0;
		if (vo.getTxtRestEndMinute3().length() != 0) {
			restEndTime3Minute = Integer.parseInt(vo.getTxtRestEndMinute3());
		}
		// 休憩4
		int restStartTime4Hour = 0;
		if (vo.getTxtRestStartHour4().length() != 0) {
			restStartTime4Hour = Integer.parseInt(vo.getTxtRestStartHour4());
		}
		int restStartTime4Minute = 0;
		if (vo.getTxtRestStartMinute4().length() != 0) {
			restStartTime4Minute = Integer.parseInt(vo.getTxtRestStartMinute4());
		}
		int restEndTime4Hour = 0;
		if (vo.getTxtRestEndHour4().length() != 0) {
			restEndTime4Hour = Integer.parseInt(vo.getTxtRestEndHour4());
		}
		int restEndTime4Minute = 0;
		if (vo.getTxtRestEndMinute4().length() != 0) {
			restEndTime4Minute = Integer.parseInt(vo.getTxtRestEndMinute4());
		}
		// 休憩5
		int restStartTime5Hour = 0;
		if (vo.getTxtRestStartHour5().length() != 0) {
			restStartTime5Hour = Integer.parseInt(vo.getTxtRestStartHour5());
		}
		int restStartTime5Minute = 0;
		if (vo.getTxtRestStartMinute5().length() != 0) {
			restStartTime5Minute = Integer.parseInt(vo.getTxtRestStartMinute5());
		}
		int restEndTime5Hour = 0;
		if (vo.getTxtRestEndHour5().length() != 0) {
			restEndTime5Hour = Integer.parseInt(vo.getTxtRestEndHour5());
		}
		int restEndTime5Minute = 0;
		if (vo.getTxtRestEndMinute5().length() != 0) {
			restEndTime5Minute = Integer.parseInt(vo.getTxtRestEndMinute5());
		}
		// 休憩6
		int restStartTime6Hour = 0;
		if (vo.getTxtRestStartHour6().length() != 0) {
			restStartTime6Hour = Integer.parseInt(vo.getTxtRestStartHour6());
		}
		int restStartTime6Minute = 0;
		if (vo.getTxtRestStartMinute6().length() != 0) {
			restStartTime6Minute = Integer.parseInt(vo.getTxtRestStartMinute6());
		}
		int restEndTime6Hour = 0;
		if (vo.getTxtRestEndHour6().length() != 0) {
			restEndTime6Hour = Integer.parseInt(vo.getTxtRestEndHour6());
		}
		int restEndTime6Minute = 0;
		if (vo.getTxtRestEndMinute6().length() != 0) {
			restEndTime6Minute = Integer.parseInt(vo.getTxtRestEndMinute6());
		}
		// 公用外出1
		int publicStartTime1Hour = 0;
		if (vo.getTxtPublicStartHour1().length() != 0) {
			publicStartTime1Hour = Integer.parseInt(vo.getTxtPublicStartHour1());
		}
		int publicStartTime1Minute = 0;
		if (vo.getTxtPublicStartMinute1().length() != 0) {
			publicStartTime1Minute = Integer.parseInt(vo.getTxtPublicStartMinute1());
		}
		int publicEndTime1Hour = 0;
		if (vo.getTxtPublicEndHour1().length() != 0) {
			publicEndTime1Hour = Integer.parseInt(vo.getTxtPublicEndHour1());
		}
		int publicEndTime1Minute = 0;
		if (vo.getTxtPublicEndMinute1().length() != 0) {
			publicEndTime1Minute = Integer.parseInt(vo.getTxtPublicEndMinute1());
		}
		// 公用外出2
		int publicStartTime2Hour = 0;
		if (vo.getTxtPublicStartHour2().length() != 0) {
			publicStartTime2Hour = Integer.parseInt(vo.getTxtPublicStartHour2());
		}
		int publicStartTime2Minute = 0;
		if (vo.getTxtPublicStartMinute2().length() != 0) {
			publicStartTime2Minute = Integer.parseInt(vo.getTxtPublicStartMinute2());
		}
		int publicEndTime2Hour = 0;
		if (vo.getTxtPublicEndHour2().length() != 0) {
			publicEndTime2Hour = Integer.parseInt(vo.getTxtPublicEndHour2());
		}
		int publicEndTime2Minute = 0;
		if (vo.getTxtPublicEndMinute2().length() != 0) {
			publicEndTime2Minute = Integer.parseInt(vo.getTxtPublicEndMinute2());
		}
		// 私用外出1
		int privateStartTime1Hour = 0;
		if (vo.getTxtPrivateStartHour1().length() != 0) {
			privateStartTime1Hour = Integer.parseInt(vo.getTxtPrivateStartHour1());
		}
		int privateStartTime1Minute = 0;
		if (vo.getTxtPrivateStartMinute1().length() != 0) {
			privateStartTime1Minute = Integer.parseInt(vo.getTxtPrivateStartMinute1());
		}
		int privateEndTime1Hour = 0;
		if (vo.getTxtPrivateEndHour1().length() != 0) {
			privateEndTime1Hour = Integer.parseInt(vo.getTxtPrivateEndHour1());
		}
		int privateEndTime1Minute = 0;
		if (vo.getTxtPrivateEndMinute1().length() != 0) {
			privateEndTime1Minute = Integer.parseInt(vo.getTxtPrivateEndMinute1());
		}
		// 私用外出2
		int privateStartTime2Hour = 0;
		if (vo.getTxtPrivateStartHour2().length() != 0) {
			privateStartTime2Hour = Integer.parseInt(vo.getTxtPrivateStartHour2());
		}
		int privateStartTime2Minute = 0;
		if (vo.getTxtPrivateStartMinute2().length() != 0) {
			privateStartTime2Minute = Integer.parseInt(vo.getTxtPrivateStartMinute2());
		}
		int privateEndTime2Hour = 0;
		if (vo.getTxtPrivateEndHour2().length() != 0) {
			privateEndTime2Hour = Integer.parseInt(vo.getTxtPrivateEndHour2());
		}
		int privateEndTime2Minute = 0;
		if (vo.getTxtPrivateEndMinute2().length() != 0) {
			privateEndTime2Minute = Integer.parseInt(vo.getTxtPrivateEndMinute2());
		}
		int[] aryStartTimeHour = { restStartTime1Hour, restStartTime2Hour, restStartTime3Hour, restStartTime4Hour,
			restStartTime5Hour, restStartTime6Hour, publicStartTime1Hour, publicStartTime2Hour, privateStartTime1Hour,
			privateStartTime2Hour };
		int[] aryStartTimeMinute = { restStartTime1Minute, restStartTime2Minute, restStartTime3Minute,
			restStartTime4Minute, restStartTime5Minute, restStartTime6Minute, publicStartTime1Minute,
			publicStartTime2Minute, privateStartTime1Minute, privateStartTime2Minute };
		int[] aryEndTimeHour = { restEndTime1Hour, restEndTime2Hour, restEndTime3Hour, restEndTime4Hour,
			restEndTime5Hour, restEndTime6Hour, publicEndTime1Hour, publicEndTime2Hour, privateEndTime1Hour,
			privateEndTime2Hour };
		int[] aryEndTimeMinute = { restEndTime1Minute, restEndTime2Minute, restEndTime3Minute, restEndTime4Minute,
			restEndTime5Minute, restEndTime6Minute, publicEndTime1Minute, publicEndTime2Minute, privateEndTime1Minute,
			privateEndTime2Minute };
		// エラーメッセージの設定
		String errMes = mospParams.getName("RestTime") + mospParams.getName("Slash") + mospParams.getName("GoingOut")
				+ mospParams.getName("Time");
		
		// 始業/終業時間チェック
		if (startTimeHour > endTimeHour) {
			mospParams.addErrorMessage(TimeMessageConst.MSG_START_END_TIME_CHECK, null, null);
			return;
		} else if (startTimeHour == endTimeHour) {
			if (startTimeMinute >= endTimeMinute) {
				mospParams.addErrorMessage(TimeMessageConst.MSG_START_END_TIME_CHECK, null, null);
				return;
			}
		}
		for (int i = 0; i < aryStartTimeHour.length; i++) {
			// 時分がすべて0(未入力)なのでチェックしない
			if (aryStartTimeHour[i] == 0 && aryStartTimeMinute[i] == 0 && aryEndTimeHour[i] == 0
					&& aryEndTimeMinute[i] == 0) {
				continue;
			}
			if (startTimeHour > aryStartTimeHour[i]) {
				mospParams.addErrorMessage(TimeMessageConst.MSG_WORK_TIME_OUT_CHECK, errMes, null);
				return;
			} else if (startTimeHour == aryStartTimeHour[i]) {
				if (startTimeMinute > aryStartTimeMinute[i]) {
					mospParams.addErrorMessage(TimeMessageConst.MSG_WORK_TIME_OUT_CHECK, errMes, null);
					return;
				}
			}
			if (endTimeHour < aryEndTimeHour[i]) {
				mospParams.addErrorMessage(TimeMessageConst.MSG_WORK_TIME_OUT_CHECK, errMes, null);
				return;
			} else if (endTimeHour == aryEndTimeHour[i]) {
				if (endTimeMinute < aryEndTimeMinute[i]) {
					mospParams.addErrorMessage(TimeMessageConst.MSG_WORK_TIME_OUT_CHECK, errMes, null);
					return;
				}
			}
		}
		// 時間重複チェック
		for (int i = 0; i < aryStartTimeHour.length; i++) {
			// 時分がすべて0(未入力)なのでチェックしない
			if (aryStartTimeHour[i] == 0 && aryStartTimeMinute[i] == 0 && aryEndTimeHour[i] == 0
					&& aryEndTimeMinute[i] == 0) {
				continue;
			}
			for (int j = i; j < aryStartTimeHour.length; j++) {
				if (i == j) {
					continue;
				}
				// 時分がすべて0(未入力)なのでチェックしない
				if (aryStartTimeHour[j] == 0 && aryStartTimeMinute[j] == 0 && aryEndTimeHour[j] == 0
						&& aryEndTimeMinute[j] == 0) {
					continue;
				}
				if (aryEndTimeHour[i] > aryStartTimeHour[j]) {
					mospParams.addErrorMessage(TimeMessageConst.MSG_REST_GOING_OUT_CHECK, errMes, null);
					return;
				} else if (aryEndTimeHour[i] == aryStartTimeHour[j]) {
					if (aryEndTimeMinute[i] > aryStartTimeMinute[j]) {
						mospParams.addErrorMessage(TimeMessageConst.MSG_REST_GOING_OUT_CHECK, errMes, null);
						return;
					}
				}
			}
		}
		
		return;
	}
	
	/**
	 * 休日出勤申請情報を確認し、勤怠詳細に表示する。
	 * @throws MospException 例外発生時
	 */
	protected void chkWorkOnHolidayInfo() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 対象日
		Date targetDate = DateUtility.getDate(vo.getLblYear(), vo.getLblMonth(), vo.getLblDay());
		// 対象日の休日出勤データを取得
		WorkOnHolidayRequestDtoInterface workOnHolidayDto = timeReference().workOnHolidayRequest()
			.findForKeyOnWorkflow(vo.getPersonalId(), targetDate);
		if (workOnHolidayDto != null) {
			// 承認状況の確認
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(
					workOnHolidayDto.getWorkflow());
			if (workflowDto != null) {
				if (PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					// 
					if (TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_ON == workOnHolidayDto.getSubstitute()) {
						// 振替申請する場合
						// 振替申請日の取得
						List<SubstituteDtoInterface> substituteList = timeReference().substitute().getSubstituteList(
								workOnHolidayDto.getWorkflow());
						if (substituteList != null && !substituteList.isEmpty()) {
							// 設定適用の取得
							ApplicationDtoInterface applicationDto = timeReference().application().findForPerson(
									getPersonalId(vo.getLblEmployeeCode()), targetDate);
							timeReference().application().chkExistApplication(applicationDto, targetDate);
							if (mospParams.hasErrorMessage()) {
								return;
							}
							// 範囲チェック
							if (chkSubstituteAllHoliday(substituteList)) {
								// 振替範囲が全休のみ
								Date substituteDate = substituteList.get(0).getSubstituteDate();
								// カレンダ日の取得
								ScheduleDateDtoInterface scheduleDateDto = timeReference().scheduleDate().findForKey(
										applicationDto.getScheduleCode(), getSearchDate(), substituteDate);
								timeReference().scheduleDate().chkExistScheduleDate(scheduleDateDto, substituteDate);
								if (mospParams.hasErrorMessage()) {
									return;
								}
								// 勤務形態の設定
								vo.setPltWorkType(scheduleDateDto.getWorkTypeCode());
								vo.setAryPltWorkType(timeReference().workType().getStartTimeEndTimeSelectArray(
										substituteDate));
							} else {
								// 振替範囲が全休以外
								// 始業時間
								Date localOpeningDate = null;
								// 終業時間
								Date localClosingDate = null;
								// 勤務形態
								String localWorkForm = "";
								for (int i = 0; i < substituteList.size(); i++) {
									// 振替日の取得
									Date substituteDate = substituteList.get(i).getSubstituteDate();
									// カレンダ日の取得
									ScheduleDateDtoInterface scheduleDateDto = timeReference().scheduleDate()
										.findForKey(applicationDto.getScheduleCode(), getSearchDate(), substituteDate);
									timeReference().scheduleDate()
										.chkExistScheduleDate(scheduleDateDto, substituteDate);
									if (mospParams.hasErrorMessage()) {
										return;
									}
									WorkTypeItemDtoInterface workTypeItemDto = null;
									if (substituteList.get(i).getSubstituteRange() == 1) {
										// 振替範囲が午前半の場合
										// 午前
										workTypeItemDto = timeReference().workTypeItem().findForKey(
												scheduleDateDto.getWorkTypeCode(), substituteDate,
												TimeConst.CODE_WORKSTART);
										// 始業時間の取得
										if (null == localOpeningDate) {
											localOpeningDate = workTypeItemDto.getWorkTypeItemValue();
										} else {
											if (workTypeItemDto.getWorkTypeItemValue().compareTo(localOpeningDate) == -1) {
												localOpeningDate = workTypeItemDto.getWorkTypeItemValue();
											}
										}
										workTypeItemDto = timeReference().workTypeItem().findForKey(
												scheduleDateDto.getWorkTypeCode(), substituteDate,
												TimeConst.CODE_BACKSTART);
										// 終業時間の取得
										if (null == localClosingDate) {
											localClosingDate = workTypeItemDto.getWorkTypeItemValue();
										} else {
											if (workTypeItemDto.getWorkTypeItemValue().compareTo(localOpeningDate) == 1) {
												localClosingDate = workTypeItemDto.getWorkTypeItemValue();
											}
										}
										// 勤務形態の取得
										localWorkForm = scheduleDateDto.getWorkTypeCode();
									} else if (substituteList.get(i).getSubstituteRange() == 2) {
										// 振替範囲が午後休の場合
										// 午前
										workTypeItemDto = timeReference().workTypeItem().findForKey(
												scheduleDateDto.getWorkTypeCode(), substituteDate,
												TimeConst.CODE_WORKSTART);
										// 始業時間の取得
										if (null == localOpeningDate) {
											localOpeningDate = workTypeItemDto.getWorkTypeItemValue();
										} else {
											if (workTypeItemDto.getWorkTypeItemValue().compareTo(localOpeningDate) == 1) {
												localOpeningDate = workTypeItemDto.getWorkTypeItemValue();
											}
										}
										workTypeItemDto = timeReference().workTypeItem().findForKey(
												scheduleDateDto.getWorkTypeCode(), substituteDate,
												TimeConst.CODE_BACKSTART);
										// 終業時間の取得
										if (null == localClosingDate) {
											localClosingDate = workTypeItemDto.getWorkTypeItemValue();
										} else {
											if (workTypeItemDto.getWorkTypeItemValue().compareTo(localOpeningDate) == -1) {
												localClosingDate = workTypeItemDto.getWorkTypeItemValue();
											}
										}
										// 勤務形態の取得
										if ("".equals(localWorkForm)) {
											// 午前休の勤務形態が存在する場合、午前休の勤務形態を優先する
											localWorkForm = scheduleDateDto.getWorkTypeCode();
										}
									} else {
										
									}
									// 始業時間の取得
									vo.setTxtStartTimeHour(DateUtility.getStringHour(localOpeningDate));
									vo.setTxtStartTimeMinute(DateUtility.getStringMinute(localOpeningDate));
									// 終業時間の取得
									vo.setTxtEndTimeHour(DateUtility.getStringHour(localClosingDate));
									vo.setTxtEndTimeMinute(DateUtility.getStringMinute(localClosingDate));
									// 勤務形態の取得
									vo.setPltWorkType(localWorkForm);
									vo.setAryPltWorkType(timeReference().workType().getStartTimeEndTimeSelectArray(
											substituteDate));
								}
							}
						} else {
							// 例外
						}
						// カレンダーの
						
					} else if (TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF == workOnHolidayDto.getSubstitute()) {
						// 振替申請しない場合
						if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(workOnHolidayDto.getWorkOnHolidayType())) {
							// 法定
							// 勤務形態の設定
							vo.setPltWorkType(TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY);
							String[][] aryPltWorkType = { { TimeConst.CODE_WORK_ON_LEGAL_HOLIDAY,
								mospParams.getName("Legal") + mospParams.getName("WorkingHoliday") } };
							vo.setAryPltWorkType(aryPltWorkType);
						} else if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workOnHolidayDto
							.getWorkOnHolidayType())) {
							// 所定
							// 勤務形態の設定
							vo.setPltWorkType(TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY);
							String[][] aryPltWorkType = { { TimeConst.CODE_WORK_ON_PRESCRIBED_HOLIDAY,
								mospParams.getName("Prescribed") + mospParams.getName("WorkingHoliday") } };
							vo.setAryPltWorkType(aryPltWorkType);
						} else {
							// 例外
						}
					} else {
						// 例外
					}
				}
			}
		}
	}
	
	/**
	 * @param lsit 振替休日データLISTDTO
	 * @return 全休有り無しのフラグ
	 * @throws MospException 例外発生時
	 */
	protected boolean chkSubstituteAllHoliday(List<SubstituteDtoInterface> lsit) throws MospException {
		if (lsit == null) {
			return false;
		}
		for (int i = 0; i < lsit.size(); i++) {
			if (1 == lsit.get(i).getSubstituteRange()) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * @return 検索用年月日
	 * @throws MospException 例外発生時
	 */
	protected Date getSearchDate() throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		// 検索用年月日の作成
		String targetYear = vo.getLblYear();
		if (Integer.parseInt(vo.getLblMonth()) <= 3) {
			targetYear = String.valueOf(Integer.parseInt(vo.getLblYear()) - 1);
		} else {
			targetYear = vo.getLblYear();
		}
		return DateUtility.getDate(targetYear, "4", "1");
	}
	
	/**
	 * @param hour 時
	 * @param minute 分
	 * @return 日付
	 * @throws MospException 例外発生時
	 */
	protected Date getAttendanceTime(String hour, String minute) throws MospException {
		// VO準備
		AttendanceCardVo vo = (AttendanceCardVo)mospParams.getVo();
		if (hour.isEmpty() || minute.isEmpty()) {
			return null;
		}
		return time().attendanceCalc().getAttendanceTime(vo.getLblDate(), hour, minute);
	}
	
	/**
	 * 代休データを登録する。
	 * @param dto 勤怠データDTOインターフェース
	 * @throws MospException 例外発生時
	 */
	protected void getSubHolidayTimeInsert(AttendanceDtoInterface dto) throws MospException {
		SubHolidayRegistBeanInterface subHolidayRegist = time().subHolidayRegist();
		subHolidayRegist.delete(dto.getPersonalId(), dto.getWorkDate());
		// 代休データ登録
		// 初期化
		String personalId = dto.getPersonalId();
		Date workDate = dto.getWorkDate();
		int timesWork = dto.getTimesWork();
		SubHolidayDtoInterface subHolidayDto = subHolidayRegist.getInitDto();
		// 共通部分の設定
		subHolidayDto.setPersonalId(personalId);
		subHolidayDto.setWorkDate(workDate);
		subHolidayDto.setTimesWork(timesWork);
		// 代休時間の取得
		// 今日が休日出勤の日の場合、勤務時間が勤怠設定の代休取得時間以上の場合、代休データの作成
		// 対象日の休日出勤の状態を確認
		WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = timeReference().workOnHolidayRequest()
			.findForKeyOnWorkflow(personalId, workDate);
		// 休日出勤申請が存在する場合
		if (workOnHolidayRequestDto != null
				&& workOnHolidayRequestDto.getSubstitute() == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF) {
			// 承認状況の確認
			WorkflowDtoInterface workOnHolidayWorkflowDto = reference().workflow().getLatestWorkflowInfo(
					workOnHolidayRequestDto.getWorkflow());
			if (workOnHolidayWorkflowDto != null) {
				// 承認が完了状態の場合
				if (PlatformConst.CODE_STATUS_COMPLETE.equals(workOnHolidayWorkflowDto.getWorkflowStatus())) {
					// 設定適用DTOの取得
					ApplicationDtoInterface applicationDto = timeReference().application().findForPerson(personalId,
							workDate);
					if (applicationDto != null) {
						// 代休種別の設定
						// カレンダDTOの取得
						ScheduleDateDtoInterface scheduleDateDto = timeReference().scheduleDate().findForKey(
								applicationDto.getScheduleCode(), getSearchDate(), workDate);
						if (scheduleDateDto != null) {
							if (TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
								// 所定代休を設定
								subHolidayDto.setSubHolidayType(TimeConst.CODE_PRESCRIBED_SUBHOLIDAY_CODE);
							} else if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
								// 法定代休を設定
								subHolidayDto.setSubHolidayType(TimeConst.CODE_LEGAL_SUBHOLIDAY_CODE);
							}
						}
						// 代休日数の設定。
						// 勤怠設定DTOの取得
						TimeSettingDtoInterface timeSettingDto = timeReference().timeSetting().getTimeSettingInfo(
								applicationDto.getWorkSettingCode(), workDate);
						if (timeSettingDto != null) {
							if (dto.getWorkTime() >= setHourMinute(timeSettingDto.getSubHolidayAllNorm())) {
								// 全休を設定
								subHolidayDto.setSubHolidayDays(1.0);
							} else if (dto.getWorkTime() >= setHourMinute(timeSettingDto.getSubHolidayHalfNorm())) {
								// 半休を設定
								subHolidayDto.setSubHolidayDays(0.5);
							}
						}
					}
				}
			}
			// 代休種別、代休日数が設定されている場合、コミットを行う
			if (subHolidayDto.getSubHolidayType() != 0 && subHolidayDto.getSubHolidayDays() != 0) {
				// コミット
				subHolidayRegist.regist(subHolidayDto);
			}
		}
		// 深夜時間の取得
		// 初期化
		subHolidayDto = subHolidayRegist.getInitDto();
		// 共通部分の設定
		subHolidayDto.setPersonalId(personalId);
		subHolidayDto.setWorkDate(workDate);
		subHolidayDto.setTimesWork(1);
		subHolidayDto.setSubHolidayType(TimeConst.CODE_MIDNIGHT_SUBHOLIDAY_CODE);
		// 深夜勤務を除いた勤務時間が7時間以上
		if ((7 * 60) <= (dto.getWorkTime() - dto.getLateNightTime())) {
			// 深夜時間
			if ((6 * 60) <= dto.getLateNightTime()) {
				// 6時間以上の場合、全休を設定
				subHolidayDto.setSubHolidayDays(1.0);
			} else if ((3 * 60) <= dto.getLateNightTime()) {
				// 3時間以上の場合、半休を設定
				subHolidayDto.setSubHolidayDays(0.5);
			}
			// 代休種別、代休日数が設定されている場合、コミットを行う
			if (subHolidayDto.getSubHolidayType() != 0 && subHolidayDto.getSubHolidayDays() != 0) {
				// コミット
				subHolidayRegist.regist(subHolidayDto);
			}
		}
	}
}
