/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.input.action;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dto.workflow.WorkflowCommentDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.AttendanceCorrectionDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkTypeDtoInterface;
import jp.mosp.time.input.vo.ApprovalHistoryVo;

/**
 * 各申請区分毎に承認履歴情報を判別して表示する。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_TIME_APPROVAL_HISTORY_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_OVERTIME_APPROVAL_HISTORY_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_APPROVAL_HISTORY_HOLIDAY_WORK_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_APPROVAL_HISTORY_LIEU_SELECT_SHOW}
 * </li><li>
 */
public class ApprovalHistoryAction extends TimeAction {
	
	/**
	 * 勤怠承認履歴選択表示コマンド。<br>
	 * <br>
	 * 勤怠一覧画面から遷移する際に実行される。画面上部には勤怠一覧画面遷移ボタンを表示させる。<br>
	 * 勤怠情報テーブルより取得した申請承認情報とその履歴を勤怠情報表示欄と承認履歴欄に表示する。<br>
	 */
	public static final String	CMD_TIME_APPROVAL_HISTORY_SELECT_SHOW				= "TM1811";
	
	/**
	 * 残業承認履歴選択表示コマンド。<br>
	 * <br>
	 * 残業申請画面から遷移する際に実行される。画面上部には残業申請画面遷移ボタンを表示させる。<br>
	 * 残業情報テーブルより取得した申請承認情報とその履歴を残業情報表示欄と承認履歴欄に表示する。<br>
	 */
	public static final String	CMD_OVERTIME_APPROVAL_HISTORY_SELECT_SHOW			= "TM1821";
	
	/**
	 * 休暇承認履歴選択表示コマンド。<br>
	 * <br>
	 * 休暇申請画面から遷移する際に実行される。画面上部には休暇申請画面遷移ボタンを表示させる。<br>
	 * 休暇情報テーブルより取得した申請承認情報とその履歴を休暇情報表示欄と承認履歴欄に表示する。<br>
	 */
	public static final String	CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW				= "TM1831";
	
	/**
	 * 休日出勤承認履歴選択表示コマンド。<br>
	 * <br>
	 * 休日出勤申請画面から遷移する際に実行される。画面上部には休日出勤申請画面遷移ボタンを表示させる。<br>
	 * 休出情報テーブルより取得した申請承認情報とその履歴を休出情報表示欄と承認履歴欄に表示する。<br>
	 */
	public static final String	CMD_APPROVAL_HISTORY_HOLIDAY_WORK_SELECT_SHOW		= "TM1841";
	
	/**
	 * 代休承認履歴選択表示コマンド。<br>
	 * <br>
	 * 代休申請画面から遷移する際に実行される。画面上部には代休申請画面遷移ボタンを表示させる。<br>
	 * 代休情報テーブルより取得した申請承認情報とその履歴を代休情報表示欄と承認履歴欄に表示する。<br>
	 */
	public static final String	CMD_APPROVAL_HISTORY_LIEU_SELECT_SHOW				= "TM1851";
	
	/**
	 * 時差出勤承認履歴選択表示コマンド。<br>
	 * <br>
	 * 時差出勤申請画面から遷移する際に実行される。画面上部には時差出勤申請画面遷移ボタンを表示させる。<br>
	 * 時差出勤情報テーブルより取得した申請承認情報とその履歴を時差出勤情報表示欄と承認履歴欄に表示する。<br>
	 */
	public static final String	CMD_DIFFERENCE_WORK_APPROVAL_HISTORY_SELECT_SHOW	= "TM1861";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public ApprovalHistoryAction() {
		super();
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new ApprovalHistoryVo();
	}
	
	@Override
	public void action() throws MospException {
		// VO準備
		prepareVo();
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_TIME_APPROVAL_HISTORY_SELECT_SHOW)) {
			// 勤怠承認履歴選択表示
			timeApprovalHistorySelectShow();
		} else if (mospParams.getCommand().equals(CMD_OVERTIME_APPROVAL_HISTORY_SELECT_SHOW)) {
			// 残業承認履歴選択表示
			overtimeApprovalHistorySelectShow();
		} else if (mospParams.getCommand().equals(CMD_LEAVE_APPROVAL_HISTORY_SELECT_SHOW)) {
			// 休暇承認履歴選択表示
			leaveApprovalHistorySelectShow();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_HISTORY_HOLIDAY_WORK_SELECT_SHOW)) {
			// 休日出勤承認履歴選択表示
			approvalHistoryHolidayWorkSelectShow();
		} else if (mospParams.getCommand().equals(CMD_APPROVAL_HISTORY_LIEU_SELECT_SHOW)) {
			// 代休承認履歴選択表示
			approvalHistoryLieuSelectShow();
		} else if (mospParams.getCommand().equals(CMD_DIFFERENCE_WORK_APPROVAL_HISTORY_SELECT_SHOW)) {
			// 時差出勤承認履歴選択表示
			differenceWorkApprovalHistorySelectShow();
		}
		// 承認履歴設定
		setApprovalHistory();
	}
	
	/**
	 * 勤怠承認履歴選択表示設定<br>
	 * @throws MospException 例外処理発生時
	 */
	public void timeApprovalHistorySelectShow() throws MospException {
		// VO準備
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		String personalId = "";
		if (getTransferredCode() != null) {
			personalId = getPersonalId(getTransferredCode());
			vo.setEmployeeCode(getTransferredCode());
		} else {
			personalId = vo.getEmployeeCode();
		}
		if (getTransferredRecordId() != null) {
			vo.setRecordId(getTransferredRecordId());
		}
		Date date;
		if (getTransferredActivateDate() != null) {
			vo.setActivateDate(getTransferredActivateDate());
			date = DateUtility.getDate(vo.getActivateDate());
		} else {
			date = DateUtility.getSystemDate();
		}
		// 該当社員欄設定
		setEmployeeParams();
		// DTO準備
		// 2010/10/21 現在仕様では勤務回数は固定で「1」
		AttendanceDtoInterface dto = timeReference().attendance().findForId(getLong(vo.getRecordId()));
		// 存在確認
		checkSelectedDataExist(dto);
		AttendanceCorrectionDtoInterface correctionDto = timeReference().attendanceCorrection()
			.getLatestAttendanceCorrectionInfo(personalId, date, 1);
		if (correctionDto != null) {
			personalId = correctionDto.getPersonalId();
		} else {
			personalId = dto.getPersonalId();
		}
		// 勤怠承認履歴
		vo.setLblAttendanceDate(DateUtility.getStringDateAndDay(dto.getWorkDate()));
		WorkTypeDtoInterface workDto = timeReference().workType().getWorkTypeInfo(dto.getWorkTypeCode(), date);
		if (workDto != null) {
			vo.setLblAttendanceWorkType(workDto.getWorkTypeAbbr());
		} else {
			vo.setLblAttendanceWorkType("");
		}
		vo.setLblAttendanceStartDate(DateUtility.getStringTime(dto.getStartTime()));
		vo.setLblAttendanceEndDate(DateUtility.getStringTime(dto.getEndTime()));
		vo.setLblAttendanceWorkTime(getValueHyphenView(dto.getWorkTime()));
		vo.setLblAttendanceRestTime(getValueHyphenView(dto.getRestTime()));
		vo.setLblAttendanceLateTime(getValueHyphenView(dto.getLateTime()));
		vo.setLblAttendanceLeaveEarly(getValueHyphenView(dto.getLeaveEarlyTime()));
		vo.setLblAttendanceOverTimeIn(getValueHyphenView(dto.getOvertime()));
		vo.setLblAttendanceOverTimeOut(getValueHyphenView(dto.getOvertimeOut()));
		vo.setLblAttendanceWorkOnHoliday(getValueHyphenView(dto.getSpecificWorkTime()));
		vo.setLblAttendanceLateNight(getValueHyphenView(dto.getLateNightTime()));
		// 未使用
		vo.setLblAttendanceRemark("");
		// ワークフロー
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setLblAttendanceState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		// 日付が対象日
		vo.setLblAttendanceCorrection("");
		if (correctionDto != null) {
			vo.setLblAttendanceCorrection(getCorrectionInfo(vo.getEmployeeCode(), reference().human().getEmployeeCode(
					personalId, date)));
		}
	}
	
	/**
	 * 残業承認履歴選択表示設定<br>
	 * @throws MospException 例外処理発生時
	 */
	public void overtimeApprovalHistorySelectShow() throws MospException {
		// VO準備
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		// 該当社員欄設定
		setEmployeeParams();
		if (getTransferredRecordId() != null) {
			vo.setRecordId(getTransferredRecordId());
		}
		// DTO準備
		OvertimeRequestDtoInterface dto = timeReference().overtimeRequest().findForId(getLong(vo.getRecordId()));
		// 存在確認
		checkSelectedDataExist(dto);
		// 2010/10/21 現在仕様では勤務回数は固定で「1」
		// 勤怠データ
		AttendanceDtoInterface attendanceDto = timeReference().attendance().findForKey(dto.getPersonalId(),
				dto.getRequestDate(), 1);
		WorkflowDtoInterface attendanceWorkflowDto = null;
		if (attendanceDto != null) {
			attendanceWorkflowDto = reference().workflow().getLatestWorkflowInfo(attendanceDto.getWorkflow());
		}
		// 残業承認履歴
		vo.setLblOverTimeDate(DateUtility.getStringDateAndDay(dto.getRequestDate()));
		vo.setLblOverTimeType(getOvertimeTypeName(dto.getOvertimeType()));
		vo.setLblOverTimeSchedule(getTimeColonFormat(dto.getRequestTime()));
		if (attendanceWorkflowDto != null
				&& attendanceWorkflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_COMPLETE)) {
			if (dto.getOvertimeType() == TimeConst.CODE_OVERTIME_WORK_BEFORE) {
				vo.setLblOverTimeResultTime(getTimeColonFormat(attendanceDto.getOvertimeBefore()));
			} else {
				vo.setLblOverTimeResultTime(getTimeColonFormat(attendanceDto.getOvertimeAfter()));
			}
		} else {
			vo.setLblOverTimeResultTime(mospParams.getName("Hyphen"));
		}
		vo.setLblOverTimeRequestReason(dto.getRequestReason());
		// ワークフロー
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo
			.setLblOverTimeState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
				.getWorkflowStage()));
		vo.setLblOverTimeApprover(getWorkflowCommentDtoLastFirstName(workflowDto, reference().workflowComment()
			.getLatestWorkflowCommentInfo(dto.getWorkflow())));
	}
	
	/**
	 * 休暇承認履歴選択表示
	 * @throws MospException 例外処理発生時
	 */
	public void leaveApprovalHistorySelectShow() throws MospException {
		// VO準備		
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		// 初期化
		vo.setRecordId("");
		vo.setLblHolidayDate("");
		vo.setLblHolidayType1("");
		vo.setLblHolidayType2("");
		vo.setLblHolidayLength("");
		vo.setLblHolidayTime("");
		vo.setLblHolidayRequestReason("");
		vo.setLblHolidayState("");
		vo.setLblHolidayApprover("");
		// 該当社員設定
		setEmployeeParams();
		if (getTransferredRecordId() == null) {
			return;
		}
		// 個人IDの設定
		vo.setRecordId(getTransferredRecordId());
		// 休暇申請DTO取得
		HolidayRequestDtoInterface dto = timeReference().holidayRequest().findForId(getLong(vo.getRecordId()));
		if (dto == null) {
			return;
		}
		// 休暇種別管理DTO取得
		HolidayDtoInterface holidayDto = timeReference().holiday().getHolidayInfo(dto.getHolidayType2(),
				dto.getRequestStartDate(), dto.getHolidayType1());
		if (holidayDto == null) {
			return;
		}
		// 休暇日の設定
		vo.setLblHolidayDate(DateUtility.getStringDateAndDay(dto.getRequestStartDate()) + mospParams.getName("Wave")
				+ DateUtility.getStringDateAndDay(dto.getRequestEndDate()));
		// 区分の設定
		if (dto.getHolidayType1() == 1) {
			vo.setLblHolidayType1(mospParams.getName("PaidHoliday"));
		} else if (dto.getHolidayType1() == 2) {
			vo.setLblHolidayType1(mospParams.getName("SpecialLeave"));
		} else if (dto.getHolidayType1() == 3) {
			vo.setLblHolidayType1(mospParams.getName("Other") + mospParams.getName("Rest"));
		} else if (dto.getHolidayType1() == 4) {
			vo.setLblHolidayType1(mospParams.getName("Absence"));
		}
		// 種別の設定
		if (dto.getHolidayType1() == 1) {
			if (dto.getHolidayType2().equals("1")) {
				vo.setLblHolidayType2(mospParams.getName("Usually"));
			} else if (dto.getHolidayType2().equals("2")) {
				vo.setLblHolidayType2(mospParams.getName("Stock"));
			}
		} else {
			vo.setLblHolidayType2(holidayDto.getHolidayAbbr());
		}
		// 範囲の設定
		if (dto.getHolidayRange() == 1) {
			vo.setLblHolidayLength(mospParams.getName("Whole"));
		} else if (dto.getHolidayRange() == 2) {
			vo.setLblHolidayLength(mospParams.getName("AmRest"));
		} else if (dto.getHolidayRange() == 3) {
			vo.setLblHolidayLength(mospParams.getName("PmRest"));
		}
		// 申請時間の設定
		vo.setLblHolidayTime(getTimeWaveFormat(dto.getStartTime(), dto.getEndTime()));
		// 理由の設定
		vo.setLblHolidayRequestReason(dto.getRequestReason());
		// ワークフローの設定
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setLblHolidayState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto.getWorkflowStage()));
		vo.setLblHolidayApprover(getWorkflowCommentDtoLastFirstName(workflowDto, reference().workflowComment()
			.getLatestWorkflowCommentInfo(dto.getWorkflow())));
	}
	
	/**
	 * 休日出勤承認履歴選択表示
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalHistoryHolidayWorkSelectShow() throws MospException {
		// VO準備
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		// 該当社員欄設定
		setEmployeeParams();
		if (getTransferredRecordId() != null) {
			vo.setRecordId(getTransferredRecordId());
		}
		// DTO準備
		WorkOnHolidayRequestDtoInterface dto = timeReference().workOnHolidayRequest().findForId(
				getLong(vo.getRecordId()));
		// 存在確認
		checkSelectedDataExist(dto);
		List<SubstituteDtoInterface> substituteList = timeReference().substitute().getSubstituteList(dto.getWorkflow());
		// 休日出勤承認履歴
		vo.setLblWorkOnHolidayWorkDate(DateUtility.getStringDateAndDay(dto.getRequestDate()));
		vo.setLblWorkOnHolidayTime(getTimeWaveFormat(dto.getStartTime(), dto.getEndTime()));
		vo.setLblWorkOnHolidayReason(dto.getRequestReason());
		vo.setLblWorkOnHolidayDate1("");
		vo.setLblWorkOnHolidayDate2("");
		int i = 0;
		for (SubstituteDtoInterface substituteDto : substituteList) {
			Date substituteDate = substituteDto.getSubstituteDate();
			int substituteRange = substituteDto.getSubstituteRange();
			if (i == 0) {
				vo.setLblWorkOnHolidayDate1(getHolidayDate(substituteDate, substituteRange));
			} else if (i == 1) {
				vo.setLblWorkOnHolidayDate2(getHolidayDate(substituteDate, substituteRange));
			}
			i++;
		}
		// ワークフロー
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setLblWorkOnHolidayState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		vo.setLblWorkOnHolidayApprover(getWorkflowCommentDtoLastFirstName(workflowDto, reference().workflowComment()
			.getLatestWorkflowCommentInfo(dto.getWorkflow())));
	}
	
	/**
	 * 代休承認履歴選択表示
	 * @throws MospException 例外処理発生時
	 */
	protected void approvalHistoryLieuSelectShow() throws MospException {
		// VO準備
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		// 該当社員欄設定
		setEmployeeParams();
		if (getTransferredRecordId() != null) {
			vo.setRecordId(getTransferredRecordId());
		}
		// DTO準備
		SubHolidayRequestDtoInterface dto = timeReference().subHolidayRequest().findForId(getLong(vo.getRecordId()));
		// 存在確認
		checkSelectedDataExist(dto);
		// 代休承認履歴
		vo.setLblSubHolidayDate(DateUtility.getStringDateAndDay(dto.getRequestDate()) + " "
				+ getSubHolidayTypeName(dto.getHolidayRange()));
		vo.setLblSubHolidayWorkDate(dto.getWorkDateSubHolidayType() + " "
				+ DateUtility.getStringDateAndDay(dto.getWorkDate()) + " "
				+ getSubHolidayTypeName(dto.getHolidayRange()));
		// ワークフロー
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setLblSubHolidayState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		vo.setLblSubHolidayApprover(getWorkflowCommentDtoLastFirstName(workflowDto, reference().workflowComment()
			.getLatestWorkflowCommentInfo(dto.getWorkflow())));
	}
	
	/**
	 * 時差出勤承認履歴選択表示
	 * @throws MospException 例外処理発生時
	 */
	protected void differenceWorkApprovalHistorySelectShow() throws MospException {
		// VO準備
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		// 該当社員欄設定
		setEmployeeParams();
		if (getTransferredRecordId() != null) {
			vo.setRecordId(getTransferredRecordId());
		}
		// DTO準備
		DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForId(getLong(vo.getRecordId()));
		// 存在確認
		checkSelectedDataExist(dto);
		// 時差出勤承認履歴
		vo.setLblDifferenceDate(DateUtility.getStringDateAndDay(dto.getRequestDate()));
		// 時差出勤区分
		vo.setLblDifferenceType(getDifferenceTypeName(dto.getDifferenceType()));
		
		vo.setLblDifferenceTime(DateUtility.getStringTime(dto.getRequestStart()) + mospParams.getName("Wave")
				+ getEndTime(dto.getRequestStart(), dto.getRequestEnd()));
		vo.setLblDifferenceReason(dto.getRequestReason());
		// ワークフロー
		WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(dto.getWorkflow());
		vo.setLblDifferenceState(getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto
			.getWorkflowStage()));
		vo.setLblDifferenceApprover(getWorkflowCommentDtoLastFirstName(workflowDto, reference().workflowComment()
			.getLatestWorkflowCommentInfo(dto.getWorkflow())));
	}
	
	/**
	 * 承認履歴設定
	 * @throws MospException 例外処理発生時
	 */
	protected void setApprovalHistory() throws MospException {
		// VO準備
		ApprovalHistoryVo vo = (ApprovalHistoryVo)mospParams.getVo();
		if (getTransferredGenericCode() != null) {
			vo.setFunctionCode(getTransferredGenericCode());
		}
		if (getTransferredRecordId() != null) {
			vo.setRecordId(getTransferredRecordId());
		}
		// 該当社員欄設定
		setEmployeeParams();
		WorkflowDtoInterface dto = timeReference().approvalInfo().getWorkflowForFunctionRecordId(vo.getFunctionCode(),
				getLong(vo.getRecordId()));
		// 存在確認
		checkSelectedDataExist(dto);
		// DTO準備
		List<WorkflowCommentDtoInterface> list = reference().workflowComment().getWorkflowCommentHistory(
				dto.getWorkflow());
		// データ配列初期化
		String[] aryLblApprovalState = new String[list.size()];
		String[] aryLblApprovalResult = new String[list.size()];
		String[] aryLblApprovalApprover = new String[list.size()];
		String[] aryLblApprovalComment = new String[list.size()];
		String[] aryLblApproverDate = new String[list.size()];
		// データ作成
		int i = 0;
		for (WorkflowCommentDtoInterface comment : list) {
			// 配列に情報を設定
			aryLblApprovalState[i] = getStatusStageValueView(comment.getWorkflowStatus(), comment.getWorkflowStage());
			aryLblApprovalResult[i] = reference().workflowIntegrate().getWorkflowOperation(comment.getWorkflowStatus());
			aryLblApprovalApprover[i] = getEmployeeName(comment.getPersonalId());
			aryLblApprovalComment[i] = comment.getWorkflowComment();
			aryLblApproverDate[i] = DateUtility.getStringDateAndTime(comment.getWorkflowDate());
			i++;
		}
		// データをVOに設定
		vo.setLblApprovalState(aryLblApprovalState);
		vo.setLblApprovalResult(aryLblApprovalResult);
		vo.setLblApprovalApprover(aryLblApprovalApprover);
		vo.setLblApprovalComment(aryLblApprovalComment);
		vo.setLblApprovalDate(aryLblApproverDate);
		
	}
	
	/**
	 * @param date 振替日
	 * @param range 振替範囲
	 * @return 一覧に表示する振替日の文字列
	 */
	protected String getHolidayDate(Date date, int range) {
		// 半日の振替休日が実装された場合は振替休日の範囲も表示する
		return DateUtility.getStringDateAndDay(date);
	}
	
}
