/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.calculation.action;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.property.MospProperties;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.bean.system.SectionReferenceBeanInterface;
import jp.mosp.platform.comparator.base.EmployeeCodeComparator;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.time.base.TimeAction;
import jp.mosp.time.bean.TotalTimeCalcBeanInterface;
import jp.mosp.time.bean.TotalTimeSearchBeanInterface;
import jp.mosp.time.bean.TotalTimeTransactionRegistBeanInterface;
import jp.mosp.time.calculation.vo.TotalTimeListVo;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.CutoffDtoInterface;
import jp.mosp.time.dto.settings.CutoffErrorListDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.SubordinateListDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.TotalAbsenceDtoInterface;
import jp.mosp.time.dto.settings.TotalLeaveDtoInterface;
import jp.mosp.time.dto.settings.TotalOtherVacationDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDataDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeDtoInterface;
import jp.mosp.time.dto.settings.TotalTimeEmployeeDtoInterface;
import jp.mosp.time.dto.settings.impl.SubordinateListDto;
import jp.mosp.time.dto.settings.impl.TmdTotalTimeDataDto;

/**
 * 勤怠集計実行後の結果の確認を行う。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SELECT_SHOW}
 * </li><li>
 * {@link #CMD_SEARCH}
 * </li><li>
 * {@link #CMD_SORT}
 * </li><li>
 * {@link #CMD_PAGE}
 * </li><li>
 * {@link #CMD_OUTPUT_FORMS}
 * </li><li>
 */
public class TotalTimeListAction extends TimeAction {
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 勤怠集計管理画面で選択した締日レコードの勤怠集計結果一覧を表示する。<br>
	 */
	public static final String	CMD_SELECT_SHOW				= "TM3121";
	
	/**
	 * 検索コマンド。<br>
	 * <br>
	 * 検索欄に入力された各種情報項目を基に検索を行い、条件に沿った社員情報の一覧表示を行う。<br>
	 * 一覧表示の際には有効日で社員コードを行う。<br>
	 */
	public static final String	CMD_SEARCH					= "TM3122";
	
	/**
	 * 再表示コマンド。<br>
	 * <br>
	 * この画面よりも奥の階層の画面から改めて遷移した場合、承認状況の更新を反映し、
	 * 直前の検索条件（申請項目）で検索を行って画面表示をする。<br>
	 */
	public static final String	CMD_RE_SHOW					= "TM3123";
	
	/**
	 * 個別仮締コマンド。<br>
	 * <br>
	 * 一覧の選択チェックボックスでチェックが入っているレコードを対象に個別の仮締処理を繰り返し行う。<br>
	 * 締日全体でまだ仮締が行われていなくてもここで個別仮締を行うことが可能で、その場合はこれ以降解除処理を行わない限り勤怠修正画面以外では情報の編集ができなくなる。<br>
	 * 選択チェックボックスにひとつもチェックが入っていない状態で実行しようとした場合にはエラーメッセージを表示し、処理を中止する。<br>
	 */
	public static final String	CMD_TEMPORARY_TIGHTENING	= "TM3124";
	
	/**
	 * 勤怠集計コマンド。<br>
	 * <br>
	 * 実行時点で社員一覧に表示されている社員の内、選択チェックボックスにチェックの
	 * 入っている社員の勤怠情報をシステム日付時点の締期間を基準に集計を行う。<br>
	 * 集計結果を社員一覧部分に表示するが、データベースには反映されない。<br>
	 */
	public static final String	CMD_CALC					= "TM3126";
	
	/**
	 * 個別解除コマンド。<br>
	 * <br>
	 * 一覧の選択チェックボックスでチェックが入っているレコードを対象に個別の解除処理を繰り返し行う。<br>
	 * 仮締がまだ一度も行われていないレコードが含まれている場合でも実行することは可能だが、そのレコードに対して何も処理は実行されない。<br>
	 * 対象レコードの締状態を一段階ずつ下げるので確定状態であれば仮締状態に、仮締状態であれば未締状態へそれぞれ移行する。<br>
	 * 選択チェックボックスにひとつもチェックが入っていない状態で実行しようとした場合にはエラーメッセージを表示し、処理を中止する。<br>
	 */
	public static final String	CMD_RELEASE					= "TM3127";
	
	/**
	 * ソートコマンド。<br>
	 * <br>
	 * それぞれのレコードの値を比較して一覧表示欄の各情報毎に並び替えを行う。
	 * これが実行される度に並び替えが昇順・降順と交互に切り替わる。<br>
	 */
	public static final String	CMD_SORT					= "TM3128";
	
	/**
	 * ページ繰りコマンド。<br>
	 * <br>
	 * 検索処理を行った際に検索結果が100件を超えた場合に分割されるページ間の遷移を行う。<br>
	 */
	public static final String	CMD_PAGE					= "TM3129";
	
	/**
	 * 帳票出力コマンド。<br>
	 * <br>
	 * 現在表示している勤怠集計結果を表計算ファイルに出勤簿として出力する。<br>
	 */
	public static final String	CMD_OUTPUT_FORMS			= "TM3176";
	

	/**
	 * {@link TimeAction#TimeAction()}を実行する。<br>
	 */
	public TotalTimeListAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SHOW;
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new TotalTimeListVo();
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SELECT_SHOW)) {
			// 表示
			prepareVo();
			show();
		} else if (mospParams.getCommand().equals(CMD_SEARCH)) {
			// 検索処理
			prepareVo();
			search();
		} else if (mospParams.getCommand().equals(CMD_RE_SHOW)) {
			// 再表示
			prepareVo(true, false);
			search();
		} else if (mospParams.getCommand().equals(CMD_TEMPORARY_TIGHTENING)) {
			// 個別仮締処理
			prepareVo();
			batchTightening();
		} else if (mospParams.getCommand().equals(CMD_CALC)) {
			// 勤怠集計
			prepareVo();
			calc();
		} else if (mospParams.getCommand().equals(CMD_RELEASE)) {
			// 個別解除処理
			prepareVo();
			batchRelease();
		} else if (mospParams.getCommand().equals(CMD_SORT)) {
			// ソート
			prepareVo();
			sort();
		} else if (mospParams.getCommand().equals(CMD_PAGE)) {
			// ページ繰り
			prepareVo();
			page();
		} else if (mospParams.getCommand().equals(CMD_OUTPUT_FORMS)) {
			// 帳票出力
			prepareVo();
			outputAttendanceBooks();
		}
	}
	
	/**
	 * 初期表示処理を行う。<br>
	 * @throws MospException 例外発生時
	 */
	protected void show() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 初期値設定
		setDefaultValues();
		// ページ繰り設定
		setPageInfo(CMD_PAGE, getListLength());
		// プルダウン設定
		setPulldown();
		// ソートキー設定
		vo.setComparatorName(EmployeeCodeComparator.class.getName());
		// 検索日付の設定
		if (getTransferredYear() != null) {
			vo.setSearchDate(DateUtility.getFirstDateOfMonth(Integer.parseInt(getTransferredYear()), Integer
				.parseInt(getTransferredMonth())));
		} else {
			vo.setSearchDate(getSystemDate());
		}
		// 締日の取得
		vo.setCutoffCode(getTransferredCode());
		vo.setJsCutoffState("");
		setCutoffDate();
	}
	
	/**
	 * @throws MospException 例外処理が発生した場合 
	 */
	protected void search() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 検索クラス取得
		TotalTimeSearchBeanInterface search = timeReference().totalTimeSearch();
		// VOの値を検索クラスへ設定
		search.setTargetDate(vo.getSearchDate());
		search.setEmployeeCode(vo.getTxtEditEmployeeCode());
		search.setEmployeeName(vo.getTxtEditEmployeeName());
		search.setWorkPlaceCode(vo.getPltEditWorkPlace());
		search.setEmploymentContractCode(vo.getPltEditEmployment());
		search.setSectionCode(vo.getPltEditSection());
		search.setPositionCode(vo.getPltEditPosition());
		search.setApproval(vo.getPltEditApproval());
		search.setCalc(vo.getPltEditCalc());
		search.setCutoffCode(vo.getCutoffCode());
		// 検索条件をもとに検索クラスからマスタリストを取得
		List<SubordinateListDtoInterface> list = search.getSearchList();
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(EmployeeCodeComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
		// 検索結果確認
		if (list.isEmpty()) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
		}
		// 一覧選択情報初期化
		setCheckOff();
		TotalTimeDtoInterface totalTimeDto = timeReference().totalTimeTransaction().findForKey(
				DateUtility.getYear(vo.getSearchDate()), DateUtility.getMonth(vo.getSearchDate()), vo.getCutoffCode());
		if (totalTimeDto == null) {
			return;
		}
		vo.setJsCutoffState(Integer.toString(totalTimeDto.getCutoffState()));
	}
	
	/**
	 * 個別仮締処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void batchTightening() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 登録クラス取得
		TotalTimeTransactionRegistBeanInterface regist = time().totalTimeTransactionRegist();
		// 集計クラス取得
		TotalTimeCalcBeanInterface calc = time().totalTimeCalc();
		// 選択個人ID配列取得
		String[] aryPersonalId = vo.getCkbSelect();
		// 個別仮締処理
		Date date = vo.getSearchDate();
		int year = DateUtility.getYear(date);
		int month = DateUtility.getMonth(date);
		String cutoffCode = vo.getCutoffCode();
		List<CutoffErrorListDtoInterface> errorList = new ArrayList<CutoffErrorListDtoInterface>();
		List<String> registerdPersonalId = new ArrayList<String>();
		for (String personalId : aryPersonalId) {
			TotalTimeEmployeeDtoInterface totalTimeEmployeeDto = timeReference().totalTimeEmployeeTransaction()
				.findForKey(personalId, year, month);
			if (totalTimeEmployeeDto != null && totalTimeEmployeeDto.getCutoffState() != 0) {
				continue;
			}
			time().totalLeaveRegist().delete(personalId, year, month);
			List<HolidayDtoInterface> specialHolidayList = timeReference().holiday().getHolidayList(date, 2);
			for (HolidayDtoInterface holidayDto : specialHolidayList) {
				TotalLeaveDtoInterface totalLeaveDto = calc.calcTotalLeave(personalId, year, month, cutoffCode,
						holidayDto.getHolidayCode());
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
				time().totalLeaveRegist().insert(totalLeaveDto);
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
			}
			time().totalOtherVacationRegist().delete(personalId, year, month);
			List<HolidayDtoInterface> otherHolidayList = timeReference().holiday().getHolidayList(date, 3);
			for (HolidayDtoInterface holidayDto : otherHolidayList) {
				TotalOtherVacationDtoInterface totalOtherVacationDto = calc.calcTotalOtherVacation(personalId, year,
						month, cutoffCode, holidayDto.getHolidayCode());
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
				time().totalOtherVacationRegist().insert(totalOtherVacationDto);
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
			}
			time().totalAbsenceRegist().delete(personalId, year, month);
			List<HolidayDtoInterface> absenceList = timeReference().holiday().getHolidayList(date, 4);
			for (HolidayDtoInterface holidayDto : absenceList) {
				TotalAbsenceDtoInterface totalAbsenceDto = calc.calcTotalAbsence(personalId, year, month, cutoffCode,
						holidayDto.getHolidayCode());
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
				time().totalAbsenceRegist().insert(totalAbsenceDto);
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
			}
			// 手当を使わない
			TotalTimeDataDtoInterface totalTimeDataDto = timeReference().totalTime()
				.findForKey(personalId, year, month);
			if (totalTimeDataDto == null) {
				totalTimeDataDto = time().totalTimeRegist().getInitDto();
			}
			List<CutoffErrorListDtoInterface> list = calc.calc(totalTimeDataDto, personalId, year, month, cutoffCode);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			if (!list.isEmpty()) {
				errorList.addAll(list);
				continue;
			}
			time().totalTimeRegist().regist(totalTimeDataDto);
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			registerdPersonalId.add(personalId);
		}
		// 締日コードが必要
		time().totalTimeEmployeeTransactionRegist().draft(registerdPersonalId, year, month, cutoffCode);
		// 個別仮締結果確認
		if (mospParams.hasErrorMessage()) {
			// 仮締失敗メッセージ設定
			addUpdateFailedMessage();
			return;
		}
		// エラーリストがあるか取得する
		if (!errorList.isEmpty()) {
			// 仮締処理からlistが帰ってきたら集計時エラー内容参照画面へ遷移する
			mospParams.addGeneralParam(TimeConst.CODE_TOTALTIME_ERROR, errorList);
			mospParams.setNextCommand(CutoffErrorListAction.CMD_SHOW);
			return;
		}
		List<String> workSettingCodeList = timeReference().timeSetting().getWorkSettingCode(cutoffCode, date);
		if (workSettingCodeList.isEmpty()) {
			// 勤怠設定コードが存在しないなら処理終了
			addInsertFailedMessage();
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("menuTimeSettings"), null);
			return;
		}
		List<String> personalIdList = timeReference().application().getPersonalIdList(workSettingCodeList, date);
		if (personalIdList.isEmpty()) {
			// メッセージ設定
			addInsertFailedMessage();
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Employee"), null);
			return;
		}
		boolean tentativeCutoff = true;
		for (String personalId : personalIdList) {
			TotalTimeEmployeeDtoInterface totalTimeEmployeeDto = timeReference().totalTimeEmployeeTransaction()
				.findForKey(personalId, year, month);
			if (totalTimeEmployeeDto != null && totalTimeEmployeeDto.getCutoffState() != 0) {
				continue;
			}
			tentativeCutoff = false;
			break;
		}
		if (tentativeCutoff) {
			TotalTimeDtoInterface dto = timeReference().totalTimeTransaction().findForKey(year, month, cutoffCode);
			if (dto == null) {
				dto = regist.getInitDto();
			}
			// DTOに値を設定
			setDtoFields(dto);
			// 仮締をセット
			dto.setCutoffState(1);
			regist.draft(dto);
		}
		if (mospParams.hasErrorMessage()) {
			// 登録失敗メッセージ設定
			addInsertFailedMessage();
			return;
		}
		// コミット
		commit();
		// 仮締成功メッセージ設定
		addTighteningMessage();
		// チェックボックス初期化
		setCheckOff();
		search();
	}
	
	/**
	 * 勤怠集計処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void calc() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 集計クラス取得
		TotalTimeCalcBeanInterface calc = time().totalTimeCalc();
		// 選択個人ID配列取得
		String[] aryPersonalId = vo.getCkbSelect();
		// 勤怠集計
		int year = DateUtility.getYear(vo.getSearchDate());
		int month = DateUtility.getMonth(vo.getSearchDate());
		Date date = vo.getSearchDate();
		TotalTimeDataDtoInterface totalTimeDateDto;
		List<SubordinateListDtoInterface> list = new ArrayList<SubordinateListDtoInterface>();
		for (String personalId : aryPersonalId) {
			ApplicationDtoInterface appDto = timeReference().application().findForMonth(personalId,
					DateUtility.getYear(date), DateUtility.getMonth(date));
			timeReference().application().chkExistApplication(appDto, date);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			TimeSettingDtoInterface timeDto = timeReference().timeSetting().getTimeSettingInfo(
					appDto.getWorkSettingCode(), date);
			timeReference().timeSetting().chkExistTimeSetting(timeDto, date);
			if (mospParams.hasErrorMessage()) {
				return;
			}
			totalTimeDateDto = new TmdTotalTimeDataDto();
			calc.calc(totalTimeDateDto, personalId, year, month, timeDto.getCutoffCode());
			if (mospParams.hasErrorMessage()) {
				// 登録失敗メッセージ設定
				addInsertFailedMessage();
				return;
			}
			double totalSpecialHoliday = 0;
			List<HolidayDtoInterface> specialHolidayList = timeReference().holiday().getHolidayList(date, 2);
			for (HolidayDtoInterface holidayDto : specialHolidayList) {
				TotalLeaveDtoInterface totalLeaveDto = calc.calcTotalLeave(personalId, year, month, timeDto
					.getCutoffCode(), holidayDto.getHolidayCode());
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
				totalSpecialHoliday += totalLeaveDto.getTimes();
			}
			double totalOtherHoliday = 0;
			List<HolidayDtoInterface> otherHolidayList = timeReference().holiday().getHolidayList(date, 3);
			for (HolidayDtoInterface holidayDto : otherHolidayList) {
				TotalOtherVacationDtoInterface totalOtherVacationDto = calc.calcTotalOtherVacation(personalId, year,
						month, timeDto.getCutoffCode(), holidayDto.getHolidayCode());
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
				totalOtherHoliday += totalOtherVacationDto.getTimes();
			}
			double totalAbsence = 0;
			List<HolidayDtoInterface> absenceList = timeReference().holiday().getHolidayList(date, 4);
			for (HolidayDtoInterface holidayDto : absenceList) {
				TotalAbsenceDtoInterface totalAbsenceDto = calc.calcTotalAbsence(personalId, year, month, timeDto
					.getCutoffCode(), holidayDto.getHolidayCode());
				if (mospParams.hasErrorMessage()) {
					// 登録失敗メッセージ設定
					addInsertFailedMessage();
					return;
				}
				totalAbsence += totalAbsenceDto.getTimes();
			}
			totalTimeDateDto.setTotalSpecialHoliday(totalSpecialHoliday);
			totalTimeDateDto.setTotalOtherHoliday(totalOtherHoliday);
			totalTimeDateDto.setTotalAbsence(totalAbsence);
			list.add(setSubordinateList(totalTimeDateDto));
		}
		if (list.size() <= 0) {
			// 検索結果無しメッセージ設定
			addNoSearchResultMessage();
			return;
		}
		// 検索結果リスト設定
		vo.setList(list);
		// デフォルトソートキー及びソート順設定
		vo.setComparatorName(EmployeeCodeComparator.class.getName());
		vo.setAscending(false);
		// ソート
		sort();
	}
	
	/**
	 * 個別解除処理を行う。<br>
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected void batchRelease() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 選択個人ID配列取得
		String[] aryPersonalId = vo.getCkbSelect();
		// 個別解除処理
		int year = DateUtility.getYear(vo.getSearchDate());
		int month = DateUtility.getMonth(vo.getSearchDate());
		String cutoffCode = vo.getCutoffCode();
		List<String> targetPersonalIdList = new ArrayList<String>();
		for (String personalId : aryPersonalId) {
			TotalTimeEmployeeDtoInterface totalTimeEmployeeDto = timeReference().totalTimeEmployeeTransaction()
				.findForKey(personalId, year, month);
			if (totalTimeEmployeeDto == null) {
				continue;
			}
			if (totalTimeEmployeeDto.getCutoffState() != 1) {
				continue;
			}
			targetPersonalIdList.add(personalId);
		}
		if (targetPersonalIdList.isEmpty()) {
			return;
		}
		time().totalTimeEmployeeTransactionRegist().draftRelease(targetPersonalIdList, year, month, cutoffCode);
		time().totalTimeRegist().delete(targetPersonalIdList, year, month);
		time().totalLeaveRegist().delete(targetPersonalIdList, year, month);
		time().totalOtherVacationRegist().delete(targetPersonalIdList, year, month);
		time().totalAbsenceRegist().delete(targetPersonalIdList, year, month);
		TotalTimeDtoInterface totalTimeDto = timeReference().totalTimeTransaction().findForKey(
				DateUtility.getYear(vo.getSearchDate()), DateUtility.getMonth(vo.getSearchDate()), vo.getCutoffCode());
		if (totalTimeDto != null) {
			int cutoffState = totalTimeDto.getCutoffState();
			if (cutoffState == 1) {
				// 仮締の場合
				totalTimeDto.setCutoffState(0);
				time().totalTimeTransactionRegist().draftRelease(totalTimeDto);
			} else if (cutoffState == 2) {
				// 確定の場合
				return;
			}
		}
		if (mospParams.hasErrorMessage()) {
			// 解除失敗メッセージ設定
			addUpdateFailedMessage();
			return;
		}
		// コミット
		commit();
		// 解除成功メッセージ設定
		addReleaseMessage();
		// チェックボックス初期化
		setCheckOff();
		search();
	}
	
	/**
	 * 一覧のソート処理を行う。<br>
	 * @throws MospException 比較クラスのインスタンス生成に失敗した場合
	 */
	protected void sort() throws MospException {
		setVoList(sortList(getTransferredSortKey()));
	}
	
	/**
	 * 一覧のページ処理を行う。
	 * @throws MospException 例外発生時
	 */
	protected void page() throws MospException {
		setVoList(pageList());
	}
	
	/**
	 * 初期値を設定する。<br>
	 */
	public void setDefaultValues() {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 検索項目設定
		vo.setTxtEditEmployeeCode("");
		vo.setTxtEditEmployeeName("");
	}
	
	/**
	 * プルダウン設定
	 * @throws MospException 例外発生時
	 */
	private void setPulldown() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// プルダウン値設定
		Date date;
		if (getTransferredYear() == null) {
			date = DateUtility.getSystemDate();
		} else {
			date = DateUtility.getDate(getTransferredYear(), getTransferredMonth(), "1");
		}
		String[][] aryWorkPlace = reference().workPlace().getNameSelectArray(date, true, null);
		String[][] aryEmployment = reference().employmentContract().getNameSelectArray(date, true, null);
		String[][] arySection = reference().section().getNameSelectArray(date, true, null);
		String[][] aryPosition = reference().position().getNameSelectArray(date, true, null);
		MospProperties properties = mospParams.getProperties();
		String[][] aryApproval = properties.getCodeArray(TimeConst.CODE_NOT_APPROVED, true);
		String[][] aryCalc = properties.getCodeArray(TimeConst.CODE_WELL_SET_STATE, true);
		// プルダウンの設定
		// 編集欄 勤務地
		vo.setAryPltEditWorkPlace(aryWorkPlace);
		// 編集欄 雇用契約
		vo.setAryPltEditEmployment(aryEmployment);
		// 編集欄 所属
		vo.setAryPltEditSection(arySection);
		// 編集欄 職位
		vo.setAryPltEditPosition(aryPosition);
		// 検索欄 未承認
		vo.setAryPltEditApproval(aryApproval);
		// 検索欄 未集計
		vo.setAryPltEditCalc(aryCalc);
	}
	
	/**
	 * 締日を設定する。<br>
	 * @throws MospException 例外発生時
	 */
	protected void setCutoffDate() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		CutoffDtoInterface dto = timeReference().cutoff().getCutoffInfo(vo.getCutoffCode(), vo.getSearchDate());
		int cutoffDate = dto.getCutoffDate();
		if (cutoffDate == 0) {
			// 締日が末日の場合
			vo.setCutoffDate(DateUtility.getLastDateOfMonth(vo.getSearchDate()));
			return;
		}
		// 締日が末日でない場合
		int year = DateUtility.getYear(vo.getSearchDate());
		int month = DateUtility.getMonth(vo.getSearchDate());
		if (cutoffDate >= 16) {
			// 締日が今月の場合
			vo.setCutoffDate(DateUtility.getDate(year, month, cutoffDate));
			return;
		}
		// 締日が今月でない場合
		Date nextMonth = DateUtility.addMonth(DateUtility.getFirstDateOfMonth(year, month), 1);
		vo.setCutoffDate(DateUtility.getDate(DateUtility.getYear(nextMonth), DateUtility.getMonth(nextMonth),
				cutoffDate));
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param list 対象リスト
	 * @throws MospException 例外発生時
	 */
	protected void setVoList(List<? extends BaseDtoInterface> list) throws MospException {
		// VO取得
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// データ配列初期化
		String[] aryPersonalId = new String[list.size()];
		String[] aryLblEmployeeCode = new String[list.size()];
		String[] aryLblEmployeeName = new String[list.size()];
		String[] aryLblSection = new String[list.size()];
		String[] aryLblWorkDate = new String[list.size()];
		String[] aryLblPaidHoliday = new String[list.size()];
		String[] aryLblAllHoliday = new String[list.size()];
		String[] aryLblAbsence = new String[list.size()];
		String[] aryLblWorkTime = new String[list.size()];
		String[] aryLblRestTime = new String[list.size()];
		String[] aryLblLateTime = new String[list.size()];
		String[] aryLblLeaveEarlyTime = new String[list.size()];
		String[] aryLblOverTimeIn = new String[list.size()];
		String[] aryLblOverTimeOut = new String[list.size()];
		String[] aryLblWorkOnHolidayTime = new String[list.size()];
		String[] aryLblLateNightTime = new String[list.size()];
		String[] aryLblApploval = new String[list.size()];
		String[] aryLblCalc = new String[list.size()];
		String[] aryLblCorrection = new String[list.size()];
		String[] claApploval = new String[list.size()];
		String[] claCalc = new String[list.size()];
		String[] aryDetailOnOff = new String[list.size()];
		
		// データ作成
		SectionReferenceBeanInterface getSection = reference().section();
		for (int i = 0; i < list.size(); i++) {
			// リストから情報を取得
			SubordinateListDtoInterface dto = (SubordinateListDtoInterface)list.get(i);
			// 配列に情報を設定
			aryPersonalId[i] = dto.getPersonalId();
			aryLblEmployeeCode[i] = dto.getEmployeeCode();
			aryLblEmployeeName[i] = getLastFirstName(dto.getLastName(), dto.getFirstName());
			aryLblSection[i] = getSection.getSectionAbbr(dto.getSectionCode(), vo.getSearchDate());
			aryLblWorkDate[i] = getNumberString(dto.getWorkDate(), 1);
			aryLblPaidHoliday[i] = getNumberString(dto.getPaidHoliday(), 1);
			aryLblAllHoliday[i] = getNumberString(dto.getAllHoliday(), 1);
			aryLblAbsence[i] = getNumberString(dto.getAbsence(), 1);
			aryLblWorkTime[i] = toTimeDotFormatString(dto.getWorkTime());
			aryLblRestTime[i] = toTimeDotFormatString(dto.getRestTime());
			aryLblLateTime[i] = toTimeDotFormatString(dto.getLateTime());
			aryLblLeaveEarlyTime[i] = toTimeDotFormatString(dto.getLeaveEarlyTime());
			aryLblOverTimeIn[i] = toTimeDotFormatString(dto.getOverTimeIn());
			aryLblOverTimeOut[i] = toTimeDotFormatString(dto.getOverTimeOut());
			aryLblWorkOnHolidayTime[i] = toTimeDotFormatString(dto.getWorkOnHolidayTime());
			aryLblLateNightTime[i] = toTimeDotFormatString(dto.getLateNightTime());
			if (!dto.getApproval().isEmpty()) {
				aryLblApploval[i] = dto.getApproval();
				claApploval[i] = setStringInactiveColor(dto.getApproval());
			} else {
				aryLblApploval[i] = "";
				claApploval[i] = "";
			}
			aryLblCalc[i] = dto.getCalc();
			if ((mospParams.getName("Provisional") + mospParams.getName("Cutoff")).equals(dto.getCalc())) {
				claCalc[i] = setInactiveColor(0);
			} else {
				claCalc[i] = setInactiveColor(1);
			}
			aryLblCorrection[i] = dto.getCorrection();
			aryDetailOnOff[i] = "off";
			// 締状態取得
			Integer cutoffState = timeReference().totalTimeEmployeeTransaction().getCutoffState(dto.getPersonalId(),
					DateUtility.getYear(vo.getSearchDate()), DateUtility.getMonth(vo.getSearchDate()));
			// 締状態確認
			if (cutoffState != null && cutoffState == 1) {
				aryDetailOnOff[i] = "on";
			}
		}
		// データをVOに設定
		vo.setAryPersonalId(aryPersonalId);
		vo.setAryLblEmployeeCode(aryLblEmployeeCode);
		vo.setAryLblEmployeeName(aryLblEmployeeName);
		vo.setAryLblSection(aryLblSection);
		vo.setAryLblWorkDate(aryLblWorkDate);
		vo.setAryLblWorkTime(aryLblWorkTime);
		vo.setAryLblRestTime(aryLblRestTime);
		vo.setAryLblLateTime(aryLblLateTime);
		vo.setAryLblLeaveEarlyTime(aryLblLeaveEarlyTime);
		vo.setAryLblOverTimeIn(aryLblOverTimeIn);
		vo.setAryLblOverTimeOut(aryLblOverTimeOut);
		vo.setAryLblWorkOnHolidayTime(aryLblWorkOnHolidayTime);
		vo.setAryLblLateNightTime(aryLblLateNightTime);
		vo.setAryLblPaidHoliday(aryLblPaidHoliday);
		vo.setAryLblAllHoliday(aryLblAllHoliday);
		vo.setAryLblAbsence(aryLblAbsence);
		vo.setAryLblApploval(aryLblApploval);
		vo.setAryLblCalc(aryLblCalc);
		vo.setAryLblCorrection(aryLblCorrection);
		vo.setClaApploval(claApploval);
		vo.setClaCalc(claCalc);
		vo.setAryDetailOnOff(aryDetailOnOff);
	}
	
	/**
	 * VO(編集項目)の値をDTOに設定する。<br>
	 * @param dto 対象DTO
	 * @throws MospException 例外発生時
	 */
	protected void setDtoFields(TotalTimeDtoInterface dto) throws MospException {
		// VO取得
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// VOの値をDTOに設定
		dto.setCalculationYear(DateUtility.getYear(vo.getSearchDate()));
		dto.setCalculationMonth(DateUtility.getMonth(vo.getSearchDate()));
		dto.setCutoffCode(vo.getCutoffCode());
		dto.setCalculationDate(vo.getCutoffDate());
		// 未締に設定
		dto.setCutoffState(0);
	}
	
	/**
	 * チェックボックスの初期化を行う。<br>
	 */
	protected void setCheckOff() {
		// VO取得
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 一覧選択情報初期化
		vo.setCkbSelect(new String[0]);
	}
	
	/**
	 * 検索結果リストの内容をVOに設定する。<br>
	 * @param dto 勤怠集計データDTO
	 * @return 一覧表示用のListデータ
	 * @throws MospException 例外発生時
	 */
	protected SubordinateListDtoInterface setSubordinateList(TotalTimeDataDtoInterface dto) throws MospException {
		SubordinateListDtoInterface subordDto = new SubordinateListDto();
		
		subordDto.setAbsence(dto.getTotalAbsence());
		subordDto.setAllHoliday(dto.getTotalSpecialHoliday() + dto.getTotalOtherHoliday() + dto.getTimesCompensation());
		subordDto.setApproval("");
		String approvalStatus = time().totalTimeCalc().getApprovalStatus(dto.getPersonalId(), dto.getCalculationYear(),
				dto.getCalculationMonth());
		if (approvalStatus != null) {
			if ("0".equals(approvalStatus)) {
				// 未承認なし
				subordDto.setApproval(mospParams.getName("Without"));
			} else if ("1".equals(approvalStatus)) {
				// 未承認あり
				subordDto.setApproval(mospParams.getName("Yes"));
			}
		}
		subordDto.setCalc(mospParams.getName("Provisional") + mospParams.getName("Cutoff"));
		TotalTimeDataDtoInterface totalTimeDateDto = timeReference().totalTime().findForKey(dto.getPersonalId(),
				dto.getCalculationYear(), dto.getCalculationMonth());
		if (totalTimeDateDto == null) {
			subordDto.setCalc(mospParams.getName("Ram") + mospParams.getName("Cutoff"));
		}
		subordDto.setCorrection("");
		subordDto.setEmployeeCode(getEmployeeCode(dto.getPersonalId()));
		subordDto.setFirstName(getEmployeeFirstName(dto.getPersonalId()));
		subordDto.setLastName(getEmployeeLastName(dto.getPersonalId()));
		subordDto.setLateNightTime(dto.getLateNight());
		subordDto.setLateTime(dto.getLateTime());
		subordDto.setLeaveEarlyTime(dto.getLeaveEarlyTime());
		subordDto.setOverTimeIn(dto.getOvertimeIn());
		subordDto.setOverTimeOut(dto.getOvertimeOut());
		subordDto.setPaidHoliday(dto.getTimesPaidHoliday());
		subordDto.setPersonalId(dto.getPersonalId());
		subordDto.setRestTime(dto.getRestTime());
		Date date = DateUtility.getSystemDate();
		HumanDtoInterface humanDto = reference().human().getHumanInfoForEmployeeCode(subordDto.getEmployeeCode(), date);
		if (humanDto == null) {
			subordDto.setSectionCode("");
		} else {
			subordDto.setSectionCode(humanDto.getSectionCode());
		}
		subordDto.setWorkDate(dto.getTimesWorkDate());
		subordDto.setWorkOnHolidayTime(dto.getWorkOnSpecificHoliday() + dto.getWorkOnHoliday());
		subordDto.setWorkTime(dto.getWorkTime());
		return subordDto;
	}
	
	/**
	 * 帳票(出勤簿)を作成し、送出ファイルとして設定する。<br>
	 * @throws MospException 帳票の作成に失敗した場合
	 */
	protected void outputAttendanceBooks() throws MospException {
		// VO準備
		TotalTimeListVo vo = (TotalTimeListVo)mospParams.getVo();
		// 個人ID配列取得
		String[] personalIds = vo.getCkbSelect();
		// 対象年月取得(VOから)
		Date tergetDate = vo.getSearchDate();
		int year = DateUtility.getYear(tergetDate);
		int month = DateUtility.getMonth(tergetDate);
		// 年月で出勤簿を作成
		timeReference().attendanceBook().makeAttendanceBooks(personalIds, year, month);
		// エラー確認
		if (mospParams.hasErrorMessage()) {
			// メッセージ設定
			addNoSearchResultMessage();
		}
	}
	
}
