/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.BaseDto;
import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDateDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dao.settings.WorkOnHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;

/**
 * 休日出勤申請参照クラス。
 */
public class WorkOnHolidayRequestReferenceBean extends TimeBean implements WorkOnHolidayRequestReferenceBeanInterface {
	
	/**
	 * 休日出勤申請マスタDAOクラス。<br>
	 */
	WorkOnHolidayRequestDaoInterface			dao;
	
	/**
	 * 代休マスタDAOクラス。<br>
	 */
	SubHolidayRequestDaoInterface				subHolidayDao;
	
	/**
	 * 勤怠データマスタDAOクラス。<br>
	 */
	AttendanceDaoInterface						attendanceDao;
	
	/**
	 * ワークフローDAOクラス。<br>
	 */
	WorkflowDaoInterface						workflowDao;
	
	/**
	 * カレンダ日マスタDTOクラス。<br>
	 */
	protected ScheduleDateDaoInterface			scheduleDateDao;
	
	/**
	 * 勤怠設定参照インターフェース。<br>
	 */
	protected TimeSettingReferenceBeanInterface	timeSettingReference;
	
	/**
	 * 振替休日データ参照インターフェース。<br>
	 */
	protected SubstituteDaoInterface			substituteDao;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public WorkOnHolidayRequestReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public WorkOnHolidayRequestReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (WorkOnHolidayRequestDaoInterface)createDao(WorkOnHolidayRequestDaoInterface.class);
		subHolidayDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		scheduleDateDao = (ScheduleDateDaoInterface)createDao(ScheduleDateDaoInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		timeSettingReference = (TimeSettingReferenceBeanInterface)createBean(TimeSettingReferenceBeanInterface.class);
		substituteDao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
	}
	
	@Override
	public WorkOnHolidayRequestDtoInterface findForKeyOnWorkflow(String personalId, Date requestDate)
			throws MospException {
		return dao.findForKeyOnWorkflow(personalId, requestDate);
	}
	
	@Deprecated
	@Override
	public WorkOnHolidayRequestDtoInterface findForKey(String personalId, Date requestDate) throws MospException {
		WorkOnHolidayRequestDtoInterface dto = dao.findForKeyOnWorkflow(personalId, requestDate);
		if (dto != null) {
			return dto;
		}
		return null;
	}
	
	@Override
	public WorkOnHolidayRequestDtoInterface findForKey(long id) throws MospException {
		BaseDto dto = findForKey(dao, id, false);
		if (dto != null) {
			return (WorkOnHolidayRequestDtoInterface)dto;
		}
		return null;
	}
	
	@Override
	public WorkOnHolidayRequestDtoInterface findForKey(String personalId, Date requestDate, int timesWork)
			throws MospException {
		return dao.findForKey(personalId, requestDate, timesWork);
	}
	
	@Override
	public WorkOnHolidayRequestDtoInterface findForWorkflow(long workflow) throws MospException {
		return dao.findForWorkflow(workflow);
	}
	
	@Override
	public List<WorkOnHolidayRequestDtoInterface> getWorkOnHolidayRequestList(String personalId, Date firstDate,
			Date lastDate) throws MospException {
		return dao.findForTerm(personalId, firstDate, lastDate);
	}
	
	@Override
	public List<WorkOnHolidayRequestDtoInterface> getListForWorkflowStatus(String personalId, int workflowStage,
			String workflowStatus, String routeCode) throws MospException {
		return dao.findForWorkflowStatus(personalId, workflowStage, workflowStatus, routeCode);
	}
	
	@Override
	public List<WorkOnHolidayRequestDtoInterface> getListForNotApproved(String personalId, int approvalStage)
			throws MospException {
		List<WorkOnHolidayRequestDtoInterface> list = new ArrayList<WorkOnHolidayRequestDtoInterface>();
		if (approvalStage > 0) {
			if (approvalStage == 1) {
				// 未承認
				list.addAll(getListForWorkflowStatus(personalId, approvalStage, "1", ""));
			} else {
				// (stage -1)次承認済
				list.addAll(getListForWorkflowStatus(personalId, approvalStage - 1, "2", ""));
			}
			// (stage +1)次差戻
			list.addAll(getListForWorkflowStatus(personalId, approvalStage + 1, "3", ""));
		}
		return list;
	}
	
	@Override
	public WorkOnHolidayRequestDtoInterface findForId(long id) throws MospException {
		BaseDtoInterface baseDto = findForKey(dao, id, false);
		checkExclusive(baseDto);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		return (WorkOnHolidayRequestDtoInterface)baseDto;
	}
	
	@Override
	public boolean chkHalfHoliday(String personalId, Date requestDate) throws MospException {
		WorkOnHolidayRequestDtoInterface workOnHoliday = findForKeyOnWorkflow(personalId, requestDate);
		if (workOnHoliday == null) {
			return false;
		}
		List<SubstituteDtoInterface> substituteList = substituteDao.findForWorkflow(workOnHoliday.getWorkflow());
		for (SubstituteDtoInterface substituteDto : substituteList) {
			if (substituteDto.getSubstituteRange() == TimeConst.CODE_HOLIDAY_RANGE_AM) {
				return true;
			} else if (substituteDto.getSubstituteRange() == TimeConst.CODE_HOLIDAY_RANGE_PM) {
				return true;
			}
		}
		return false;
	}
	
	public String getWorkOnHolidayInfo(WorkOnHolidayRequestDtoInterface dto) {
		// 表示例 勤務予定:10:00～17:00 理由:提案書作成
		StringBuffer sb = new StringBuffer();
		String startTime = DateUtility.getStringTime(dto.getStartTime());
		String endTime = DateUtility.getStringTime(dto.getEndTime());
		if (!"".equals(startTime) && !"".equals(endTime)) {
			// 勤務予定
			sb.append(mospParams.getName("Work") + mospParams.getName("Schedule") + mospParams.getName("Colon")
					+ DateUtility.getStringTime(dto.getStartTime()) + mospParams.getName("Wave")
					+ DateUtility.getStringTime(dto.getEndTime()));
			// 空欄
			sb.append(" ");
		}
		// 理由
		sb.append(mospParams.getName("Reason") + mospParams.getName("Colon") + dto.getRequestReason());
		return sb.toString();
	}
	
	@Override
	public void chkBasicInfo(String personalId, Date targetDate) throws MospException {
		initial(personalId, targetDate);
	}
}
