/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.List;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowCommentRegistBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowRegistBeanInterface;
import jp.mosp.platform.bean.workflow.impl.WorkflowReferenceBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.PaidHolidayReferenceBeanInterface;
import jp.mosp.time.bean.ScheduleDateReferenceBeanInterface;
import jp.mosp.time.bean.ScheduleReferenceBeanInterface;
import jp.mosp.time.bean.SubHolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.SubHolidayRequestRegistBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.bean.TotalTimeEmployeeTransactionReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.DifferenceRequestDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDaoInterface;
import jp.mosp.time.dao.settings.ScheduleDateDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dao.settings.WorkOnHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.PaidHolidayDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdSubHolidayRequestDto;

/**
 * 代休申請登録クラス。
 */
public class SubHolidayRequestRegistBean extends TimeBean implements SubHolidayRequestRegistBeanInterface {
	
	/**
	 * 代休申請マスタDAOクラス。<br>
	 */
	SubHolidayRequestDaoInterface								dao;
	/**
	 * 代休申請参照インターフェース。<br>
	 */
	SubHolidayRequestReferenceBeanInterface						subHolidayReference;
	/**
	 * ワークフローDAOクラス。<br>
	 */
	private WorkflowDaoInterface								workflowDao;
	
	/**
	 * ワークフロー参照クラス。<br>
	 */
	private WorkflowReferenceBeanInterface						workflowReference;
	/**
	 * ワークフロー登録クラス。<br>
	 */
	private WorkflowRegistBeanInterface							workflowRegist;
	
	/**
	 * ワークフローコメント登録クラス。<br>
	 */
	private WorkflowCommentRegistBeanInterface					workflowCommentRegist;
	
	private OvertimeRequestDaoInterface							overtimeRequestDao;
	private HolidayRequestDaoInterface							holidayRequestDao;
	private DifferenceRequestDaoInterface						differenceRequestDao;
	private ApplicationReferenceBeanInterface					applicationReference;
	private ScheduleDaoInterface								scheduleDao;
	private ScheduleReferenceBeanInterface						scheduleReference;
	private ScheduleDateDaoInterface							scheduleDateDao;
	private ScheduleDateReferenceBeanInterface					scheduleDateReference;
	private TimeSettingReferenceBeanInterface					timeSettingReference;
	private PaidHolidayReferenceBeanInterface					paidHolidayReference;
	private WorkOnHolidayRequestDaoInterface					workOnHolidayRequestDao;
	private SubHolidayDaoInterface								subHolidayDao;
	private SubstituteDaoInterface								substituteDao;
	
	private AttendanceDaoInterface								attendanceDao;
	/**
	 * 社員勤怠集計管理参照クラス。
	 */
	private TotalTimeEmployeeTransactionReferenceBeanInterface	totalTimeEmployeeReference;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public SubHolidayRequestRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public SubHolidayRequestRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		subHolidayReference = (SubHolidayRequestReferenceBeanInterface)createBean(SubHolidayRequestReferenceBeanInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
		workflowCommentRegist = (WorkflowCommentRegistBeanInterface)createBean(WorkflowCommentRegistBeanInterface.class);
		overtimeRequestDao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
		holidayRequestDao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		differenceRequestDao = (DifferenceRequestDaoInterface)createDao(DifferenceRequestDaoInterface.class);
		applicationReference = (ApplicationReferenceBeanInterface)createBean(ApplicationReferenceBeanInterface.class);
		scheduleDao = (ScheduleDaoInterface)createDao(ScheduleDaoInterface.class);
		scheduleReference = (ScheduleReferenceBeanInterface)createBean(ScheduleReferenceBeanInterface.class);
		scheduleDateDao = (ScheduleDateDaoInterface)createDao(ScheduleDateDaoInterface.class);
		scheduleDateReference = (ScheduleDateReferenceBeanInterface)createBean(ScheduleDateReferenceBeanInterface.class);
		workflowReference = (WorkflowReferenceBean)createBean(WorkflowReferenceBean.class);
		workOnHolidayRequestDao = (WorkOnHolidayRequestDaoInterface)createDao(WorkOnHolidayRequestDaoInterface.class);
		subHolidayDao = (SubHolidayDaoInterface)createDao(SubHolidayDaoInterface.class);
		substituteDao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		timeSettingReference = (TimeSettingReferenceBeanInterface)createBean(TimeSettingReferenceBeanInterface.class);
		paidHolidayReference = (PaidHolidayReferenceBeanInterface)createBean(PaidHolidayReferenceBeanInterface.class);
		// 社員勤怠集計管理参照クラス
		totalTimeEmployeeReference = (TotalTimeEmployeeTransactionReferenceBeanInterface)createBean(TotalTimeEmployeeTransactionReferenceBeanInterface.class);
	}
	
	@Override
	public SubHolidayRequestDtoInterface getInitDto() {
		return new TmdSubHolidayRequestDto();
	}
	
	@Override
	public void insert(SubHolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdSubHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(long[] idArray) throws MospException {
		if (idArray == null || idArray.length == 0) {
			// 表示例 必ず一件はチェックボックスを選択してください。
			mospParams.addErrorMessage(PlatformMessageConst.MSG_CHECK, null);
			return;
		}
		// Bean初期化
		workflowRegist = (WorkflowRegistBeanInterface)createBean(WorkflowRegistBeanInterface.class);
		// 処理ワークフロー情報リスト準備
		List<WorkflowDtoInterface> workflowList = new ArrayList<WorkflowDtoInterface>();
		for (long id : idArray) {
			// DTOの準備
			BaseDtoInterface baseDto = findForKey(dao, id, true);
			checkExclusive(baseDto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			SubHolidayRequestDtoInterface dto = (SubHolidayRequestDtoInterface)baseDto;
			// 申請時の確認
			checkAppli(dto);
			if (mospParams.hasErrorMessage()) {
				continue;
			}
			// ワークフローDTOの準備
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(dto.getWorkflow());
			// 申請
			workflowRegist.appli(workflowDto, dto.getPersonalId(), dto.getRequestDate(),
					PlatformConst.WORKFLOW_TYPE_TIME);
			// 処理ワークフロー情報リストへ追加
			if (workflowDto != null) {
				workflowList.add(workflowDto);
			}
		}
		// ワークフローコメント登録
		workflowCommentRegist.addCommentList(workflowList, mospParams.getUser().getPersonalId(), mospParams
			.getProperties().getMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED,
					new String[]{ mospParams.getName("Application") }));
	}
	
	@Override
	public void regist(SubHolidayRequestDtoInterface dto) throws MospException {
		if (dao.findForKey(dto.getTmdSubHolidayRequestId(), false) == null) {
			// 新規登録
			insert(dto);
		} else {
			// 履歴追加
			add(dto);
		}
	}
	
	@Override
	public void add(SubHolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdSubHolidayRequestId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdSubHolidayRequestId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void delete(SubHolidayRequestDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 論理削除
		logicalDelete(dao, dto.getTmdSubHolidayRequestId());
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(SubHolidayRequestDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForKeyOnWorkflow(dto.getPersonalId(), dto.getRequestDate(), dto.getHolidayRange()));
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkAdd(SubHolidayRequestDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdSubHolidayRequestId());
	}
	
	@Override
	public void validate(SubHolidayRequestDtoInterface dto) throws MospException {
		// 基本情報のチェック
		subHolidayReference.chkBasicInfo(dto.getPersonalId(), dto.getRequestDate());
	}
	
	@Override
	public void checkDraft(SubHolidayRequestDtoInterface dto) throws MospException {
		checkTemporaryClosingFinal(dto);
	}
	
	@Override
	public void checkAppli(SubHolidayRequestDtoInterface dto) throws MospException {
		// 下書き同様の処理を行う。
		checkDraft(dto);
		// 代休申請の申請期間チェック。
		checkPeriod(dto);
		// 代休申請の重複チェック
		checkSubHolidayOverlap(dto);
		// 他の申請チェック。
		checkRequest(dto);
		// 勤怠の申請チェック
		checkAttendance(dto);
		// 代休日数チェック。
		checkDay(dto);
	}
	
	@Override
	public void checkWithdrawn(SubHolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkApproval(SubHolidayRequestDtoInterface dto) throws MospException {
		// 申請時と同様の処理を行う
		checkAppli(dto);
	}
	
	@Override
	public void checkCancel(SubHolidayRequestDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
	}
	
	@Override
	public void checkPeriod(SubHolidayRequestDtoInterface dto) {
		if (dto.getRequestDate().after(DateUtility.addMonth(DateUtility.getSystemDate(), 1))) {
			addSubHolidayPeriodErrorMessage();
		}
	}
	
	@Override
	public void checkSubHolidayOverlap(SubHolidayRequestDtoInterface dto) throws MospException {
		// 代休申請リスト取得
		List<SubHolidayRequestDtoInterface> list = dao.findForList(dto.getPersonalId(), dto.getRequestDate());
		for (SubHolidayRequestDtoInterface requestDto : list) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 取下の場合
				continue;
			}
			if (dto.getWorkflow() == workflowDto.getWorkflow()) {
				// ワークフロー番号が同じ場合は同じ申請
				continue;
			}
			if (dto.getHolidayRange() == 1 || requestDto.getHolidayRange() == 1) {
				// 全休の場合
				// 休暇内容が重複しています。代休日または休暇範囲を選択し直してください。
				addSubHolidayTargetDateSubHolidayRequestErrorMessage();
				break;
			}
			if (dto.getHolidayRange() == 2 || dto.getHolidayRange() == 3) {
				// 午前休又は午後休の場合
				if (dto.getHolidayRange() == requestDto.getHolidayRange()) {
					// 休暇内容が重複しています。代休日または休暇範囲を選択し直してください。
					addSubHolidayTargetDateSubHolidayRequestErrorMessage();
					break;
				}
			}
		}
	}
	
	@Override
	public void checkRequest(SubHolidayRequestDtoInterface dto) throws MospException {
		String personalId = dto.getPersonalId();
		// 休出申請
		WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = workOnHolidayRequestDao.findForKeyOnWorkflow(
				personalId, dto.getRequestDate());
		if (workOnHolidayRequestDto != null) {
			// 休出申請がある場合
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(workOnHolidayRequestDto
				.getWorkflow());
			if (workflowDto != null) {
				if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 下書でなく且つ取下でない場合
					// （日付）は休日出勤が申請されています。残業年月日を選択し直してください。
					addSubHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				}
			}
		}
		boolean amFlag = false;
		boolean pmFlag = false;
		// 振替休日
		List<SubstituteDtoInterface> substituteList = substituteDao.findForList(personalId, dto.getRequestDate());
		for (SubstituteDtoInterface substituteDto : substituteList) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(substituteDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				if (dto.getHolidayRange() == 1) {
					// 全休の場合
					addSubHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				}
				int range = substituteDto.getSubstituteRange();
				if (range == 1) {
					// 全休の場合
					addSubHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				} else if (range == 2 || range == 3) {
					// 午前休又は午後休の場合
					if (range == 2) {
						amFlag = true;
					}
					if (range == 3) {
						pmFlag = true;
					}
					if (dto.getHolidayRange() == range) {
						addSubHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
						return;
					}
				}
			}
		}
		// 休暇申請
		List<HolidayRequestDtoInterface> holidayRequestList = holidayRequestDao.findForList(personalId, dto
			.getRequestDate());
		for (HolidayRequestDtoInterface requestDto : holidayRequestList) {
			// ワークフローの取得
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書でなく且つ取下でない場合
				if (dto.getHolidayRange() == 1) {
					// 全休の場合
					addSubHolidayTargetDateSubHolidayRequestErrorMessage();
					return;
				}
				int range = requestDto.getHolidayRange();
				if (range == 1) {
					// 全休の場合
					// 表示例 <日付>は出勤日ではありません。代休日を選択し直してください。 
					addSubHolidayTargetNoWorkDateErrorMessage(dto.getRequestDate());
					return;
				} else if (range == 2 || range == 3) {
					// 午前休又は午後休の場合
					if (range == 2) {
						amFlag = true;
					}
					if (range == 3) {
						pmFlag = true;
					}
					if (dto.getHolidayRange() == range) {
						// 表示例 休暇内容が重複しています。代休日または休暇範囲を選択し直してください。
						addSubHolidayTargetDateSubHolidayRequestErrorMessage();
						return;
					}
				}
			}
		}
		if (amFlag && pmFlag) {
			addSubHolidayTargetDateSubHolidayRequestErrorMessage();
			return;
		}
		if (dto.getHolidayRange() == 1) {
			// 全休の場合
			// 残業申請取得
			List<OvertimeRequestDtoInterface> overtimeRequestList = overtimeRequestDao.findForList(personalId, dto
				.getRequestDate());
			// 申請日に残業申請を行っているとエラーメッセージ設定
			for (OvertimeRequestDtoInterface requestDto : overtimeRequestList) {
				// ワークフローの取得
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 下書でなく且つ取下でない場合
					// 表示例 <日付>は既に他の申請が行われています。代休日を選択し直してください。 
					addSubHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
					return;
				}
			}
			// 時差出勤申請取得
			DifferenceRequestDtoInterface differenceRequestDto = differenceRequestDao.findForKeyOnWorkflow(personalId,
					dto.getRequestDate());
			if (differenceRequestDto != null) {
				// 時差出勤申請がある場合
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(differenceRequestDto
					.getWorkflow());
				if (workflowDto != null) {
					if (!PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
							&& !PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
						// 下書でなく且つ取下でない場合
						addSubHolidayTargetDateRequestErrorMessage(dto.getRequestDate());
						return;
					}
				}
			}
		}
		// 設定適用マスタDTOの取得
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(personalId, dto.getRequestDate());
		applicationReference.chkExistApplication(applicationDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// カレンダマスタDTOの取得
		ScheduleDtoInterface scheduleDto = scheduleDao.findForInfo(applicationDto.getScheduleCode(), dto
			.getRequestDate());
		scheduleReference.chkExistSchedule(scheduleDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 代休日のカレンダ上の情報取得
		ScheduleDateDtoInterface scheduleDateDto = scheduleDateDao.findForInfo(applicationDto.getScheduleCode(),
				scheduleDto.getActivateDate(), dto.getRequestDate());
		scheduleDateReference.chkExistScheduleDate(scheduleDateDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		if (scheduleDateDto.getWorkTypeCode() == null || scheduleDateDto.getWorkTypeCode().isEmpty()
				|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())
				|| TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(scheduleDateDto.getWorkTypeCode())) {
			// 代休日に選択されている日が法定休日、所定休日又は勤務形態が登録されていない場合
			addSubHolidayTargetWorkDateHolidayErrorMessage(dto.getRequestDate());
		}
		// 勤怠設定
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), dto.getRequestDate());
		timeSettingReference.chkExistTimeSetting(timeSettingDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 有休設定
		PaidHolidayDtoInterface paidHolidayDto = paidHolidayReference.getPaidHolidayInfo(applicationDto
			.getPaidHolidayCode(), dto.getRequestDate());
		paidHolidayReference.chkExistPaidHoliday(paidHolidayDto, dto.getRequestDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
	}
	
	@Override
	public void checkAttendance(SubHolidayRequestDtoInterface dto) throws MospException {
		AttendanceDtoInterface attendanceDto = attendanceDao.findForKey(dto.getPersonalId(), dto.getRequestDate(), 1);
		if (attendanceDto == null) {
			return;
		}
		WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(attendanceDto.getWorkflow());
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 取下の場合
			return;
		}
		if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
			// 下書の場合
			return;
		}
		if (workflowDto.getWorkflowStage() == 0
				&& PlatformConst.CODE_STATUS_REVERT.equals(workflowDto.getWorkflowStatus())) {
			// 1次戻の場合
			return;
		}
		// （日付）は既に勤怠の申請が行われています。（申請区分毎の日付名称）を選択し直してください。
		addSubHolidayTargetWorkDateAttendanceRequestErrorMessage(dto.getRequestDate());
	}
	
	@Override
	public void checkDay(SubHolidayRequestDtoInterface dto) throws MospException {
		SubHolidayDtoInterface subHolidayDto = subHolidayDao.findForKey(dto.getPersonalId(), dto.getWorkDate(), dto
			.getTimesWork(), dto.getWorkDateSubHolidayType());
		double subHolidayDays = subHolidayDto.getSubHolidayDays();
		List<SubHolidayRequestDtoInterface> list = dao.findForList(dto.getPersonalId(), dto.getWorkDate(), dto
			.getTimesWork(), dto.getWorkDateSubHolidayType());
		for (SubHolidayRequestDtoInterface requestDto : list) {
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
					|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 下書又は取下の場合
				continue;
			}
			if (dto.getWorkflow() == workflowDto.getWorkflow()) {
				// ワークフロー番号が同じ場合は同じ申請
				continue;
			}
			int holidayRange = requestDto.getHolidayRange();
			if (holidayRange == 1) {
				subHolidayDays--;
			} else if (holidayRange == 2 || holidayRange == 3) {
				subHolidayDays -= 0.5;
			}
		}
		int holidayRange = dto.getHolidayRange();
		String errorMes1 = "";
		if (holidayRange == 1) {
			// 全休の場合
			if (subHolidayDays < 1) {
				// 代休日数が1未満の場合
				errorMes1 = "1";
				// 表示例 代休日数が1未満の場合
				mospParams.addErrorMessage(TimeMessageConst.MSG_SUBHOLIDAY_DAY_CHECK, errorMes1, null);
			}
		} else if (holidayRange == 2 || holidayRange == 3) {
			// 午前休又は午後休の場合
			if (subHolidayDays < 0.5) {
				errorMes1 = "0.5";
				// 代休日数が0.5未満の場合
				mospParams.addErrorMessage(TimeMessageConst.MSG_SUBHOLIDAY_DAY_CHECK, errorMes1, null);
			}
		}
	}
	
	@Override
	public void checkTemporaryClosingFinal(SubHolidayRequestDtoInterface dto) throws MospException {
		Integer state = totalTimeEmployeeReference.getCutoffState(dto.getPersonalId(), DateUtility.getYear(dto
			.getRequestDate()), DateUtility.getMonth(dto.getRequestDate()));
		if (state == null) {
			return;
		}
		if (state != 0) {
			// 未締でない場合
			addMonthlyTreatmentErrorMessage(dto.getRequestDate(), mospParams.getName("CompensatoryHoliday")
					+ mospParams.getName("Day"));
		}
	}
	
}
